// 2007-04-27
using System;
using System.IO;
using System.Reflection;
using CliOptionParser = Sgry.CliOptionParser;
using StringDictionary = System.Collections.Specialized.StringDictionary;

[assembly: AssemblyTitle("_dir")]
[assembly: AssemblyProduct("_dir")]
[assembly: AssemblyCopyright("Copyright 2006-2010, YAMAMOTO Suguru")]
[assembly: AssemblyVersion("1.2.0.*")]

namespace _dir
{
	internal class _dirApp
	{
		[STAThread]
		static int Main( string[] args )
		{
			Option	opt = new Option( args );
			string[] dirs;
			string[] files;

			if( opt.ShowUsage )
			{
				ShowUsage();
				return 0;
			}

			try
			{
				// get the file system entries in the directory path.
				dirs = Directory.GetDirectories( opt.WorkDir, opt.Filter );
				files = Directory.GetFiles( opt.WorkDir, opt.Filter );

				// if no file entry here, display message and exit
				if( dirs.Length == 0 && files.Length == 0 )
				{
					Console.WriteLine( "t@C܂B" );
					return 0;
				}

				// display file info
				foreach( string entry in dirs )
				{
					DirectoryInfo dirInfo = new DirectoryInfo( entry );

					// skip if this is a hidden entry and not specified -a/--all
					if( opt.ShowAll == false
						&& (dirInfo.Attributes & FileAttributes.Hidden) != 0 )
					{
						continue;
					}

					PrintDirectoryEntry( dirInfo, opt.UseLongFormat );
				}
				foreach( string entry in files )
				{
					FileInfo fileInfo = new FileInfo( entry );

					// skip if this is a hidden entry and not specified -a/--all
					if( opt.ShowAll == false
						&& (fileInfo.Attributes & FileAttributes.Hidden) != 0 )
					{
						continue;
					}
					PrintFileEntry( fileInfo, opt.UseLongFormat );
				}
			}
			catch( Exception )
			{
				Console.WriteLine( "(error)" );
				return 1;
			}
			
			return 0;
		}

		static void PrintFileEntry( FileInfo fileInfo, bool useLongFormat )
		{
			if( useLongFormat )
			{
				Console.WriteLine( "{0}  {1}  {2}  {3}",
					fileInfo.Name,
					fileInfo.Length,
					fileInfo.LastAccessTime,
					Utl.ParseAttributes(fileInfo.Attributes)
				);
			}
			else
			{
				Console.WriteLine( fileInfo.Name );
			}
		}

		static void PrintDirectoryEntry( DirectoryInfo dirInfo, bool useLongFormat )
		{
			if( useLongFormat )
			{
				Console.WriteLine( "{0}  {1}  {2}",
					dirInfo.Name + Path.DirectorySeparatorChar,
					dirInfo.LastAccessTime,
					Utl.ParseAttributes(dirInfo.Attributes)
				);
			}
			else
			{
				Console.WriteLine( dirInfo.Name + Path.DirectorySeparatorChar );
			}
		}
		
		static int ShowUsage()
		{
			string msg;

			if( System.Globalization.CultureInfo.CurrentCulture.Name == "ja-JP" )
			{
				msg = Usage_Ja;
			}
			else
			{
				msg = Usage_En;
			}

			Console.Write( msg );
			
			return 0;
		}

		const string Usage_En = 
@"Usage: _dir [options] [filter]
List directory contents.

Parameter:
  [filter]
        Specifies file-name pattern to be listed up.
        (can use wildcards: ?, *)

  [options]
        -l          show file's size, last modified date and attributes
        -a, --all   show hidden files
        -?, --help  show this help message

Example:
  _dir a*.txt
        list up all .txt files which name begins with 'a' in current directory.

  _dir ..\*.txt
        list up all files in parent directory.

  _dir C:\work\*
        list up all files in directory 'C:\work'.
";

		const string Usage_Ja = 
@"Usage: _dir [IvV] [tB^[]
fBNg̓eꗗ\܂B

p[^F
  [tB^[]
        ꗗ\t@C̃p^[w肵܂B
        iChJ[hg܂: ?, *j

  [IvV]
        -l          t@C̃TCYAXVA\
        -a, --all   B̃t@C\
        -?, --help  ̃wvbZ[W\܂B

:
  _dir a*.txt
        JgfBNg a Ŏn܂邷ׂĂ .txt t@C\

  _dir ..\*.txt
        efBNgɂ邷ׂĂ .txt t@C\

  _dir C:\work\*
        C:\work ɂ邷ׂẴt@C\
";

		class Utl
		{
			public static string ParseAttributes( FileAttributes attr )
			{
				string text = "";

				if( (attr & FileAttributes.ReparsePoint) != 0 )
				{
					text += "r";
				}
				if( (attr & FileAttributes.Compressed) != 0 )
				{
					text += "c";
				}
				if( (attr & FileAttributes.Directory) != 0 )
				{
					text += "D";
				}
				if( (attr & FileAttributes.ReadOnly) != 0 )
				{
					text += "R";
				}
				if( (attr & FileAttributes.Hidden) != 0 )
				{
					text += "H";
				}
				if( (attr & FileAttributes.System) != 0 )
				{
					text += "S";
				}
				if( (attr & FileAttributes.Archive) != 0 )
				{
					text += "A";
				}

				return text;
			}
		}
	}

	internal class Option
	{
		public bool ShowUsage = false;
		public bool ShowAll = false;
		public bool UseLongFormat = false;
		public string LastToken = "";
		public string WorkDir = Environment.CurrentDirectory;
		public string Filter = "*.*";

		public Option( string[] args )
		{
			StringDictionary options;
			string lastToken;

			options = CliOptionParser.Parse( args );
			if( options == null )
			{
				return;
			}
			
			// -?
			if( options.ContainsKey("?") || options.ContainsKey("-help") )
			{
				ShowUsage = true;
			}

			// -a
			if( options.ContainsKey("a") || options.ContainsKey("-all") )
			{
				ShowAll = true;
			}

			// -l
			if( options.ContainsKey("l") )
			{
				UseLongFormat = true;
			}

			// filter
			if( options.ContainsKey(String.Empty) )
			{
				lastToken = options[String.Empty];
				Utl.ParseLastToken( lastToken, ref WorkDir, ref Filter );
			}
			
		}
		
		class Utl
		{
			public static void ParseLastToken( string lastToken, ref string io_workDir, ref string io_searchPattern )
			{
				string workDir;

				// if the last token indicates a directory, parse it as the target of directory listing
				if( Directory.Exists(lastToken) )
				{
					io_workDir = lastToken;
					return;
				}

				// last token is made with path with pattern or pattern
				workDir = Path.GetDirectoryName( lastToken );
				if( Directory.Exists(workDir) )
				{
					io_workDir = workDir;
				}
				io_searchPattern = Path.GetFileName( lastToken );
			}
		}
	}
}
