// file: RichEdit20Utl.cs
// brief: EDIT と共通でない、RichEdit20W クラスのウィンドウ用ユーティリティ関数群
// encoding: UTF-8
// update: 2008-10-04
//=========================================================
using System;
using System.Text;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace Sgry
{
	/// <summary>
	/// RichEdit20W ウィンドウ専用のユーティリティ関数群。
	/// EDIT ウィンドウとの違いを吸収するためにある。
	/// </summary>
	/// <remarks>
	/// RichEdit(20W/50W) は SysEdit エミュレーションを有効にすると
	/// ファイルが空行で終わる場合に最終行が存在しない事になるバグがある。
	/// それへの対策も行っている。
	/// </remarks>
	public class RichEdit20Utl : EditUtl
	{
		/// <summary>
		/// 画面に表示されている一番下の行のインデックスを取得
		/// </summary>
		public override int GetLastVisibleLine( IntPtr hWnd )
		{
			RECT rect, pRect;
			IntPtr parent;
			Point bottomLeftCharPos;
			int lastLineHeadIndex;
			
			unsafe
			{
				parent = GetParent( hWnd );
				if( parent == IntPtr.Zero )
				{
					return -1;
				}
				GetWindowRect( parent, &pRect );
				GetWindowRect( hWnd, &rect );
			}
			bottomLeftCharPos = new Point( rect.left - pRect.left, rect.bottom - pRect.top );
			lastLineHeadIndex = GetCharIndexFromPosition( hWnd, bottomLeftCharPos );

			return GetLineIndexFromCharIndex( hWnd, lastLineHeadIndex );
		}

		#region 単語の処理
		/// <summary>
		/// 指定したインデックスを含む単語を取得。
		/// 単語間の空白を指定すると空白の直前の単語を取得する。
		/// （修正すべき）
		/// </summary>
		public virtual string GetWordAt( IntPtr richEdit, int index )
		{
			int wordBegin, wordEnd;
			string allText = GetText( richEdit );

			// 前後の単語区切り位置を取得
			wordBegin = SendMessage( richEdit, EM_FINDWORDBREAK, new IntPtr(WB_MOVELEFT), new IntPtr(index) ).ToInt32();
			wordEnd = SendMessage( richEdit, EM_FINDWORDBREAK, new IntPtr(WB_MOVERIGHT), new IntPtr(index) ).ToInt32();
			if( wordEnd <= wordBegin
				|| allText.Length < wordEnd )
			{
				return null; // おかしい
			}

			// 指定位置が単語の開始位置なら指定位置を単語開始位置とする
			// （単語の開始位置だと EM_FINDWORDBREAK が返す値がずれる）
			if( index < allText.Length )
			{
				int leftBreakFromRightChar
					= SendMessage( richEdit, EM_FINDWORDBREAK, new IntPtr(WB_MOVELEFT), new IntPtr(index+1) ).ToInt32();
				if( wordBegin != leftBreakFromRightChar )
				{
					wordBegin = index;
					goto end;
				}
			}

			// 指定位置が単語の終了位置なら指定位置を単語終了位置とする
			// （単語の終了位置だと EM_FINDWORDBREAK が返す値がずれる）
			if( 1 <= index )
			{
				int rightBreakFromLeftChar
					= SendMessage( richEdit, EM_FINDWORDBREAK, new IntPtr(WB_MOVERIGHT), new IntPtr(index-1) ).ToInt32();
				if( wordEnd != rightBreakFromLeftChar )
				{
					wordEnd = index;
				}
			}

			end:
			return allText.Substring( wordBegin, wordEnd - wordBegin ).Trim();
		}

		/// <summary>
		/// 次の単語開始位置を検索
		/// </summary>
		public virtual int GetNextWordStart( IntPtr richEdit, int index )
		{
			return SendMessage( richEdit, EM_FINDWORDBREAK, WordBreakAction.WB_RIGHT, index ).ToInt32();
		}

		/// <summary>
		/// 次の単語終了位置を検索
		/// </summary>
		public virtual int GetNextWordEnd( IntPtr richEdit, int index )
		{
			return SendMessage( richEdit, EM_FINDWORDBREAK, WordBreakAction.WB_RIGHTBREAK, index ).ToInt32();
		}

		/// <summary>
		/// 前の単語開始位置を検索
		/// </summary>
		public virtual int GetPreviousWordStart( IntPtr richEdit, int index )
		{
			return SendMessage( richEdit, EM_FINDWORDBREAK, WordBreakAction.WB_LEFT, index ).ToInt32();
		}

		/// <summary>
		/// 前の単語終了位置を検索
		/// </summary>
		public virtual int GetPreviousWordEnd( IntPtr richEdit, int index )
		{
			return SendMessage( richEdit, EM_FINDWORDBREAK, WordBreakAction.WB_LEFTBREAK, index ).ToInt32();
		}
		#endregion

		#region キャレット
		/// <summary>
		/// キャレット位置を取得
		/// </summary>
		public virtual int GetCaretIndex( IntPtr richEdit )
		{
			int selStart, selEnd;
			
			// 選択範囲を取得
			GetSelection( richEdit, out selStart, out selEnd );

			// 無選択状態なら選択開始位置＝選択終了位置＝キャレット位置
			if( selStart == selEnd )
			{
				return selStart;
			}

			// 選択があるなら、キャレットが選択開始位置にあるかをスクリーン座標で判断
			if( selStart == GetCharIndexFromPosition(richEdit, GetCaretPos()) )
			{
				return selStart;
			}
			else
			{
				return selEnd;
			}
		}

		/// <summary>
		/// キャレット位置を取得
		/// </summary>
		public virtual void GetCaretIndex( IntPtr richEdit, out int line, out int column )
		{
			GetLineColumnIndexFromCharIndex( richEdit, GetCaretIndex(richEdit), out line, out column );
		}
		#endregion

		#region 選択
		/// <summary>
		/// 選択が後方へ行われているか判定
		/// </summary>
		public virtual bool IsSelectingToEnd( IntPtr richEdit )
		{
			int selStart, selEnd, selAnchor;
			GetSelection( richEdit, out selStart, out selEnd );
			selAnchor = GetSelectionAnchor( richEdit );

			return (selAnchor <= selStart);
		}

		/// <summary>
		/// 選択の開始位置を取得
		/// </summary>
		public virtual int GetSelectionAnchor( IntPtr richEdit )
		{
			int selStart, selEnd, selAnchor;
			GetSelection( richEdit, out selStart, out selEnd );

			// 選択が無いなら、後方へ向かって選択中とする
			if( selStart == selEnd )
			{
				selAnchor = selStart;
				return selAnchor;
			}

			// 選択範囲の先頭にキャレットがあれば、先頭に向けて選択を広げている最中
			if( selStart == GetCaretIndex(richEdit) )
			{
				selAnchor = selEnd;
			}
			else
			{
				selAnchor = selStart;
			}

			return selAnchor;
		}
		
		/// <summary>
		/// 選択中テキストを取得
		/// </summary>
		public virtual string GetSelectedText( IntPtr richEdit )
		{
			string str;
			GETTEXTEX rec = new GETTEXTEX();
			unsafe
			{
				int textLength = GetSelectedTextLength( richEdit );
				char[] buf = new char[ textLength + 1 ];
				rec.cb = (UInt32)(textLength + 1) * 2;
				rec.flags = 2; //GT_SELECTION
				rec.codepage = 1200; // unicode
				fixed( char* pBuf = buf )
				{
					SendMessage( richEdit, EM_GETTEXTEX, (void*)&rec, (void*)pBuf );
					str = new string( pBuf );
				}
			}
			return str;
		}

		/// <summary>
		/// 選択中テキストの長さを取得
		/// </summary>
		public virtual int GetSelectedTextLength( IntPtr richEdit )
		{
			unsafe
			{
				GETTEXTLENGTHEX rec = new GETTEXTLENGTHEX();
				rec.codepage = 1200; // unicode
				rec.flags = 2; // GT_SELECTION
				return SendMessage( richEdit, EM_GETTEXTLENGTHEX, (void*)&rec, (void*)0 ).ToInt32();
			}
		}

		/// <summary>
		/// 選択中テキストを置換
		/// </summary>
		public virtual void SetSelectedText( IntPtr richEdit, string text )
		{
			GETTEXTLENGTHEX rec = new GETTEXTLENGTHEX();
			unsafe
			{
				rec.flags = 2; //ST_SELECTION
				rec.codepage = 1200; // unicode
				fixed( char* pBuf = text.ToCharArray() )
				{
					SendMessage( richEdit, EM_SETTEXTEX, (void*)&rec, (void*)pBuf );
				}
			}
		}
		#endregion

		/// <summary>
		/// 通知するを決定
		/// </summary>
		public void SetEventMask( IntPtr richEdit, int mask )
		{
			const int EM_SETEVENTMASK = 0x0400 + 69;
			SendMessage( richEdit, EM_SETEVENTMASK, IntPtr.Zero, new IntPtr(mask) );
		}

		/// <summary>
		/// やり直しの上限回数を設定
		/// </summary>
		public void SetUndoLimit( IntPtr richEdit, int count )
		{
			SendMessage( richEdit, EM_SETUNDOLIMIT, new IntPtr(count), IntPtr.Zero );
		}

		#region 編集操作
		/// <summary>
		/// 選択中の文字列を置き換えます。
		/// </summary>
		public void ReplaceText( IntPtr rich, string text )
		{
			//const int ST_DEFAULT	= 0;
			const int ST_KEEPUNDO	= 1;
			const int ST_SELECTION	= 2;

			unsafe
			{
				fixed( char * buf = text.ToCharArray() )
				{
					SETTEXTEX st = new SETTEXTEX();
					st.codepage = 1200; // Unicode
					st.flags = ST_KEEPUNDO | ST_SELECTION;
					Win32.SendMessage( rich, EM_SETTEXTEX, (void*)&st, (void*)buf );
				}
			}
		}
		#endregion

		#region SysEdit エミュレーション
		/// <summary>
		/// RichEdit の EDIT エミュレーションを有効・無効にする
		/// </summary>
		public virtual void SetSysEditEmulation( IntPtr hWnd, bool doEmulation )
		{
			if( doEmulation )
				SendMessage( hWnd, EM_SETTEXTMODE, new IntPtr(1), IntPtr.Zero );
			else
				SendMessage( hWnd, EM_SETTEXTMODE, new IntPtr(2), IntPtr.Zero );
		}

		/// <summary>
		/// RichEdit の EDIT エミュレーションが有効か確認
		/// </summary>
		public virtual bool IsSysEditEmulation( IntPtr hWnd )
		{
			Int32 rc = SendMessage( hWnd, EM_GETTEXTMODE, IntPtr.Zero, IntPtr.Zero ).ToInt32();
			return ((rc & 1) != 0); // 1 が立っていればエミュレーションON
		}
		#endregion
		
		#region 編集中の内容全体の操作
		/// <summary>
		/// 編集中の内容を取得
		/// </summary>
		public virtual string GetText( IntPtr richEdit )
		{
			string str;
			GETTEXTEX rec = new GETTEXTEX();
			unsafe
			{
				int textLength = GetTextLength( richEdit );
				char[] buf = new char[ textLength + 1 ];
				rec.cb = (UInt32)(textLength + 1) * 2;
				rec.codepage = 1200; // unicode
				fixed( char* pBuf = buf )
				{
					SendMessage( richEdit, EM_GETTEXTEX, (void*)&rec, (void*)pBuf );
					str = new string( pBuf );
				}
			}
			return str;
		}

		/// <summary>
		/// 編集中の内容を設定
		/// </summary>
		public void SetText( IntPtr richEdit, string text )
		{
			GETTEXTLENGTHEX rec = new GETTEXTLENGTHEX();
			String c_text = text + '\0';

			unsafe
			{
				rec.flags = 0; // discard undo stack
				rec.codepage = 1200; // unicode
				fixed( char* pBuf = c_text.ToCharArray() )
				{
					SendMessage( richEdit, EM_SETTEXTEX, (void*)&rec, (void*)pBuf );
				}
			}
		}

		/// <summary>
		/// 入力されているテキストの長さを取得
		/// </summary>
		public override int GetTextLength( IntPtr richEdit )
		{
			unsafe
			{
				GETTEXTLENGTHEX rec = new GETTEXTLENGTHEX();
				rec.codepage = 1200; // unicode
				rec.flags = 0;
				return SendMessage( richEdit, EM_GETTEXTLENGTHEX, &rec, null ).ToInt32();
			}
		}
		
		/// <summary>
		/// 入力されているテキストの一部を取得
		/// </summary>
		public string GetTextRange( IntPtr richEdit, int start, int end )
		{
			System.Diagnostics.Debug.Assert( start < end );
			unsafe
			{
				TEXTRANGE trange = new TEXTRANGE();
				int copiedLength;
				string textPart = null;

				fixed( char* buf = new char[end - start] )
				{
					trange.range.min = start;
					trange.range.max = end;
					trange.text = buf;

					copiedLength = (int)SendMessage( richEdit, EM_GETTEXTRANGE, (void*)0, (void*)&trange );
					if( copiedLength != end-start )
					{
						return String.Empty;
					}

					textPart = new String( buf );
				}

				return textPart;
			}
		}
		#endregion

		#region 行単位の操作
		/// <summary>
		/// 指定したインデックスの行の先頭にある文字の、全体でのインデックスを取得
		/// </summary>
		/// <remarks>RichEdit のバグに対策</remarks>
		public override int GetLineHeadIndex( IntPtr hWnd, int lineIndex )
		{
			// ファイルの末尾が改行コードで
			// 指定位置がファイル末尾なら、
			if( IsSysEditEmulation(hWnd)
				&& lineIndex == base.GetLineCount(hWnd)+1
				&& GetText(hWnd).EndsWith("\r") )
			{
				return GetTextLength( hWnd );
			}

			return base.GetLineHeadIndex( hWnd, lineIndex );
		}
		
		/// <summary>
		/// テキスト全体でのインデックスで指定した文字が含まれる、行のインデックスを取得
		/// </summary>
		/// <remarks>RichEdit のバグに対策</remarks>
		public override int GetLineIndexFromCharIndex( IntPtr hWnd, int charIndex )
		{
			int lineIndex = base.GetLineIndexFromCharIndex( hWnd, charIndex );

			// If it's RichEdit20W in SysEdit emulation mode,
			// and charIndex is same as file length,
			// and the file ends with empty line,
			// then the char at charIndex is reported as "right of the char at (charIndex-1)."
			if( IsSysEditEmulation(hWnd)
				&& charIndex == GetTextLength(hWnd)
				&& GetText(hWnd).EndsWith("\r") )
			{
				lineIndex += 1; // got value was 1 less value because of the bug
			}
			return lineIndex;
		}
		#endregion

		#region インデックスと座標
		/// <summary>
		/// 指定したスクリーン座標に一番近い文字の、全体でのインデックスを取得
		/// </summary>
		/// <remarks>RichEdit のバグに対策</remarks>
		public override int GetCharIndexFromPosition( IntPtr hWnd, Point position )
		{
			unsafe
			{
				POINT pt;
				int index;

				// APIでまずは素直に取得
				pt.x = position.X;
				pt.y = position.Y;
				index = SendMessage( hWnd, EM_CHARFROMPOS, (void*)0, (void*)&pt ).ToInt32();
				
				// RichEdit のバグに対策。
				// SysEdit エミュレーションが有効で末尾が改行文字の時、
				// ファイル末尾の座標を与えると最終行一つ前の行の先頭のインデックスが返る。
				if( IsSysEditEmulation(hWnd) )
				{
					int textLength = GetTextLength( hWnd );
					int lineLength = GetLineLengthFromCharIndex( hWnd, index );
					if( index + lineLength == textLength )
					{
						return textLength;
					}
				}

				return index;
			}
		}

		/// <summary>
		/// テキスト全体でのインデックスが指す文字の、スクリーン座標を取得。
		/// ただし、一行分の内容を改行したのみというファイルで
		/// 二行目先頭を指定すると RichEdit のバグに対策しきれず
		/// (-1, -1) という座標を返す。
		/// </summary>
		/// <remarks>RichEdit のバグに対策</remarks>
		public override Point GetPositionFromCharIndex( IntPtr hWnd, int charIndex )
		{
			// If it's RichEdit20W in SysEdit emulation mode,
			// and charIndex is same as file length,
			// and the file ends with empty line,
			// then the char at charIndex is reported as "right of the char at (charIndex-1)."
			if( IsSysEditEmulation(hWnd)
				&& charIndex == GetTextLength(hWnd)
				&& GetText(hWnd).EndsWith("\r") )
			{
				int line, lineHeadIndex;
				int lineHeight;

				// get previous line end position
				Point pos = base.GetPositionFromCharIndex( hWnd, charIndex - 1 );

				// get previous of previous line end position
				line = GetLineIndexFromCharIndex( hWnd, charIndex-1 );
				lineHeadIndex = GetLineHeadIndex( hWnd, line );
				Point prePrePos = base.GetPositionFromCharIndex( hWnd, lineHeadIndex-1 );
				if( lineHeadIndex == 0 )
				{
					return new Point( -1, -1 );
				}

				// get line height
				lineHeight = pos.Y + (pos.Y-prePrePos.Y);

				// calculate manually
				return new Point( 1, lineHeight );
			}

			// other cases, do normally
			return base.GetPositionFromCharIndex( hWnd, charIndex );
		}
		#endregion

		/// <summary>
		/// 単語検出関数を設定
		/// </summary>
		public void SetWordBreakProc( IntPtr richEdit, EditWordBreakProc newProc )
		{
			SendMessage( richEdit, EM_SETWORDBREAKPROC, IntPtr.Zero, newProc );
		}

		/// <summary>
		/// 単語検出関数
		/// </summary>
		public unsafe delegate Int32 EditWordBreakProc( char* text, Int32 currentIndex, Int32 textLength, WordBreakAction action );

		#region Win32API
		[DllImport("user32.dll", EntryPoint="SendMessageW")]
		static extern IntPtr SendMessage( IntPtr hWnd, int msg, IntPtr zero, EditWordBreakProc proc );
		
		static IntPtr SendMessage( IntPtr hWnd, int msg, WordBreakAction action, int index )
		{
			return SendMessage( hWnd, msg, new IntPtr((int)action), new IntPtr(index) );
		}

		struct CHARRANGE
		{
			public Int32 min;
			public Int32 max;
		}

		unsafe struct TEXTRANGE
		{
			public CHARRANGE range;
			public char* text;
		}

		struct SETTEXTEX
		{
			public UInt32 flags;
			public UInt32 codepage;
		}

		struct GETTEXTEX
		{
			public UInt32 cb;
			public UInt32 flags;
			public UInt32 codepage;
			public UInt32 setZero1;
			public UInt32 setZero2;
		}

		struct GETTEXTLENGTHEX
		{
			public UInt32 flags;
			public UInt32 codepage;
		}

		/// <summary>
		/// 単語検出動作
		/// </summary>
		public enum WordBreakAction : int
		{
			/// <summary>
			/// Finds the beginning of a word to the left of the specified position.
			/// </summary>
			WB_LEFT = 0,
			
			/// <summary>
			/// Finds the beginning of a word to the right of the specified position.
			/// This is useful in right-aligned edit controls.
			/// </summary>
			WB_RIGHT = 1,
			
			/// <summary>
			/// Checks whether the character at the specified position is a delimiter.
			/// </summary>
			WB_ISDELIMITER = 2,
			
			/// <summary>
			/// Retrieves the character class and word break flags of the character
			/// at the specified position.
			/// This value is for use with rich edit controls.
			/// </summary>
			WB_CLASSIFY = 3,
			
			/// <summary>
			/// Finds the end-of-word delimiter to the left of the specified position.
			/// This value is for use with rich edit controls.
			/// </summary>
			WB_LEFTBREAK = 6,
			
			/// <summary>
			/// Finds the beginning of a word to the left of the specified position.
			/// This value is used during CTRL+LEFT key processing.
			/// This value is for use with rich edit controls.
			/// </summary>
			WB_MOVEWORDLEFT = 4,
			
			/// <summary>
			/// Finds the beginning of a word to the right of the specified position.
			/// This value is used during CTRL+RIGHT key processing.
			/// This value is for use with rich edit controls.
			/// </summary>
			WB_MOVEWORDRIGHT = 5,
			
			/// <summary>
			/// Finds the end-of-word delimiter to the right of the specified position.
			/// This is useful in right-aligned edit controls.
			/// This value is for use with rich edit controls.
			/// </summary>
			WB_RIGHTBREAK = 7,
		}
		const int WB_MOVELEFT		= 4;
		const int WB_MOVERIGHT		= 5;
		#endregion
	}
}
