// file: Localizer.cs
// brief: Helper class to localize UI components.
// encoding: UTF-8
// update: 2011-02-26
//=========================================================
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Reflection;
using CultureInfo = System.Globalization.CultureInfo;

namespace Sgry
{
	/// <summary>
	/// ローカライゼーションを行うためのヘルパークラス。
	/// </summary>
	public class Localizer
	{
		Ini _Ini = new Ini();

		#region Property
		/// <summary>
		/// ローカライズに用いる言語名を取得します。
		/// </summary>
		public string LanguageName
		{
			get
			{
				// 環境変数から取得を試みる
				string langEnvVar = Environment.GetEnvironmentVariable( "LANG" );
				if( langEnvVar != null )
				{
					return langEnvVar;
				}

				// 環境変数が設定されていなければ現在の UI カルチャーを採用
				return CultureInfo.CurrentUICulture.Name;
			}
		}
		#endregion

		#region Main Logic
		/// <summary>
		/// 言語リソースファイルを読み込む。
		/// 読み込むファイルのパスは次になる。
		/// ./lang/(指定した名前).(言語名).ini
		/// </summary>
		/// <param name="moduleFileName">読み込むリソースファイル名の一部</param>
		/// <returns>成功したら true、失敗したら false</returns>
		public bool LoadResourceFile( string moduleFileName )
		{
			string resFilePath, resFileName;

			if( moduleFileName == null )
			{
				moduleFileName = Path.GetFileNameWithoutExtension( Application.ExecutablePath );
			}

			// get path of the directory where application module is
			resFilePath = Path.GetDirectoryName( Application.ExecutablePath );
			
			// make language resource file path from it
			resFilePath = Path.Combine( resFilePath, "lang" );
			resFileName = String.Format( "{0}.{1}.ini", moduleFileName, LanguageName );
			resFilePath = Path.Combine( resFilePath, resFileName );
			
			// open language file
			try
			{
				_Ini.Clear();
				_Ini.Load( resFilePath, Encoding.UTF8 );
			}
			catch( IOException )
			{
				return false;
			}

			return true;
		}

		/// <summary>
		/// 文字列リソースを外部 INI ファイルから取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="defaultString">リソースが取得できなかった時の値(null可)</param>
		/// <returns>取得した文字列</returns>
		/// <example>
		/// [source]
		/// Sgry.Localizer localizer = new Sgry.Localizer();
		/// 
		/// localizer.LoadResourceFile( "AiBEdit" );
		/// 
		/// // apply localized string to menu items
		/// localizer.TryGetString( "MainForm._MenuItem_File.Text", _MenuItem_File );
		/// localizer.TryGetString( "MainForm._MenuItem_Edit.Text", _MenuItem_Edit );
		/// 
		/// // apply localized string to child controls
		/// foreach( Control ctrl in this.Controls )
		/// {
		///	    localizer.TryGetString( "MainForm." + ctrl.Name + ".Text", ctrl );
		/// }
		/// 
		/// [resource(.\lang\AiBEdit.ja-JP.ini)]
		/// MainForm._MenuItem_File.Text=開く(&amp;O)
		/// MainForm._MenuItem_Edit.Text=編集(&amp;E)
		/// MainForm._Button_Cancel.Text=キャンセル(&amp;C)
		/// MainForm._Button_Apply.Text=適用(&amp;A)
		/// MainForm.Msg_ExceptionOccured.Text=適用(&amp;A)
		/// </example>
		public string TryGetString( string id, string defaultString )
		{
			return _Ini.Get( "", id, defaultString );
		}

		/// <summary>
		/// 指定したフォームのメンバーをローカライズします。
		/// </summary>
		public void Localize( Form form )
		{
			Type type;
			Type stringType;
			Type stringArrayType;
			Type labelType;
			Type buttonType;
			Type groupType;
			Type checkType;
			Type comboType;
			string localizedString;
			BindingFlags searchFlags;

			type = form.GetType();
			stringType = typeof( String );
			stringArrayType = typeof( String ).MakeArrayType();
			labelType = typeof( Label );
			buttonType = typeof( Button );
			groupType = typeof( GroupBox );
			checkType = typeof( CheckBox );
			comboType = typeof( ComboBox );
			searchFlags = BindingFlags.Instance | BindingFlags.NonPublic;

			// seek all fields with Localizable attribute
			foreach( FieldInfo field in type.GetFields(searchFlags) )
			{
				if( field.FieldType == stringType )
				{
					// just replace it with localized one
					localizedString = _Ini.Get( type.FullName, field.Name, null );
					if( localizedString == null )
					{
						continue;
					}
					field.SetValue( form, localizedString );
				}
				else if( field.FieldType == comboType )
				{
					ComboBox combo = (ComboBox)field.GetValue( form );

					// replace each elements available
					for( int i=0; i<combo.Items.Count; i++ )
					{
						localizedString = _Ini.Get(
								type.FullName,
								field.Name+".Items["+i+"]",
								null
							);
						if( localizedString != null )
						{
							combo.Items[i] = localizedString;
						}
					}
				}
				else if( field.FieldType == labelType
					|| field.FieldType.IsSubclassOf(labelType)
					|| field.FieldType == buttonType
					|| field.FieldType.IsSubclassOf(buttonType)
					|| field.FieldType == checkType
					|| field.FieldType.IsSubclassOf(checkType)
					|| field.FieldType == groupType
					|| field.FieldType.IsSubclassOf(groupType) )
				{
					Control ctrl = (Control)field.GetValue( form );

					// replace Text property
					localizedString = _Ini.Get( type.FullName, field.Name, null );
					if( localizedString == null )
					{
						continue;
					}
					ctrl.Text = localizedString;
				}
			}
		}
		#endregion

		#region Glue Methods
		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		public void TryGetString( string id, ref string str )
		{
			str = TryGetString( id, str );
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいコントロール</param>
		public void TryGetString( string id, Control target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				target.Text = str;
			}
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいメニュー項目</param>
		public void TryGetString( string id, MenuItem target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				int tabPos = target.Text.IndexOf( '\t' );
				if( 0 <= tabPos && str.IndexOf('\t') < 0 )
				{
					target.Text = str + target.Text.Substring( tabPos );
				}
				else
				{
					target.Text = str;
				}
			}
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいメニュー項目</param>
		public void TryGetString( string id, ToolStripMenuItem target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				target.Text = str;
			}
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいリストビューのサブ項目</param>
		public void TryGetString( string id, ListViewItem.ListViewSubItem target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				target.Text = str;
			}
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいリスト列ヘッダ</param>
		public void TryGetString( string id, ColumnHeader target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				target.Text = str;
			}
		}

		/// <summary>
		/// 文字列リソースを取得。
		/// </summary>
		/// <param name="id">リソース名</param>
		/// <param name="target">取得した文字列をTextプロパティにセットしたいツリーノード</param>
		public void TryGetString( string id, TreeNode target )
		{
			string str = TryGetString( id );
			if( str != null )
			{
				target.Text = str;
			}
		}
		#endregion

		#region Utilities
		string TryGetString( string id )
		{
			return TryGetString( id, (string)null );
		}
		#endregion
	}
}
