// file: AutoSpeaker.cs
// brief: Class to make screen readers speak
// author: SGRY (YAMAMOTO Suguru)
// update: 2008-10-05
// version: 1.3.2
// license: MIT License (see END of this file)
//=========================================================
using System;
using System.Text;
using Microsoft.Win32;
using System.Runtime.InteropServices;

namespace Sgry
{
	/// <summary>
	/// NĂXN[[_[ɔb邽߂̃C^tF[X
	/// </summary>
	public interface ISpeaker
	{
		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		void Speak( string message );

		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		void SpeakSub( string message );

		/// <summary>
		/// ǂݏグ~
		/// </summary>
		void Stop();
		
		/// <summary>
		/// ǂݏグɁuԁv}
		/// </summary>
		void Wait();
	}

	/// <summary>
	/// NĂXN[[_[IɑIēǂݏグsNXB
	/// XN[[_NĂȂΉȂ_~[IuWFNgƂȂB
	/// </summary>
	public class AutoSpeaker
	{
		AutoSpeaker()
		{}

		/// <summary>
		/// ÑXN[[_[𐧌䂷Xs[JIuWFNg擾
		/// </summary>
		/// <returns>NXN[[_p̃Xs[JIuWFNg</returns>
		public static ISpeaker Instance
		{
			get
			{
				if( JawsSpeaker.IsAlive() )
					return JawsSpeaker.Instance;
				else if( PcTalkerSpeaker.IsAlive() )
					return PcTalkerSpeaker.Instance;
				else if( FocusTalkSpeaker.IsAlive() )
					return FocusTalkSpeaker.Instance;
				else if( XpReaderSpeaker.IsAlive() )
					return XpReaderSpeaker.Instance;
				else
					return DummySpeaker.Instance;
			}
		}

		/// <summary>
		/// ÑXN[[_[𐧌䂷Xs[JIuWFNg擾ł邩
		/// </summary>
		public static bool IsAlive()
		{
			if( Instance == null || Instance is DummySpeaker )
				return false;

			return true;
		}
	}

	/// <summary>
	/// bȂA_~[̃Xs[J[NX
	/// </summary>
	public class DummySpeaker : ISpeaker
	{
		static ISpeaker _Instance = null;

		DummySpeaker(){}
		
		/// <summary>
		/// B̃CX^X
		/// </summary>
		public static ISpeaker Instance
		{
			get
			{
				if( _Instance == null )
					_Instance = new DummySpeaker();
				return _Instance;
			}
		}

		/// <summary>ǂݏグ</summary>
		public void Speak( string message ){}
		
		/// <summary>ŕǂݏグ</summary>
		public void SpeakSub( string message ){}
		
		/// <summary>ǂݏグ~</summary>
		public void Stop(){}
		
		/// <summary>ǂݏグɁuԁv}</summary>
		public void Wait(){}
	}

	/// <summary>
	/// PC-Talker PȃXs[J[ƂĎg߂̃NX
	/// </summary>
	public class PcTalkerSpeaker : ISpeaker
	{
		const int PS_STATUS = 0x00010001;
		//const int TTSPRIORITY_HIGH = 4;
		const int TTSPRIORITY_HIGHEST = 5;
		const int PKDI_KAKKO = 12;
		const int PKDI_KIGOU = 15;
		const int PKDI_KUTEN = 17;
		const int PKDI_RPITCH = 24;

		static PcTalkerSpeaker _Instance = null;

		PcTalkerSpeaker()
		{}

		/// <summary>
		/// B̃CX^X擾
		/// </summary>
		public static ISpeaker Instance
		{
			get
			{
				if( _Instance == null )
				{
					_Instance = new PcTalkerSpeaker();
				}
				return _Instance;
			}
		}

		/// <summary>
		/// PC-Talker Xs[J[ƂĎg邩
		/// </summary>
		/// <returns>gꍇ true</returns>
		public static bool IsAlive()
		{
			try
			{
				return PCTKStatus();
			}
			catch( DllNotFoundException )
			{
				return false;
			}
		}

		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void Speak( string message )
		{
			UInt32 readsKakko, readsKigou;
			
			// modify non CR+LF EOL code to CR+LF
			if( message.Length == 1
				&& (message[0] == '\r' || message[0] == '\n') )
			{
				message = "\r\n";
			}

			// keep original settings
			readsKakko = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), IntPtr.Zero );
			readsKigou = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), IntPtr.Zero );
			
			// disable reading punctuation
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), new IntPtr(1) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), new IntPtr(1) );

			// speak
			PCTKPRead( message, TTSPRIORITY_HIGHEST, true );

			// restore setting of reading punctuation
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), new IntPtr(readsKakko) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), new IntPtr(readsKigou) );
		}

		/// <summary>
		/// ŕǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void SpeakSub( string message )
		{
			UInt32 readsKakko, readsKigou, pitch;
			
			// modify non CR+LF EOL code to CR+LF
			if( message.Length == 1
				&& (message[0] == '\r' || message[0] == '\n') )
			{
				message = "\r\n";
			}

			// keep original settings
			readsKakko = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), IntPtr.Zero );
			readsKigou = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), IntPtr.Zero );
			pitch = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_RPITCH), IntPtr.Zero );

			// disable reading punctuation
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), new IntPtr(1) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), new IntPtr(1) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_RPITCH), new IntPtr(1) );

			// speak
			PCTKPRead( message, TTSPRIORITY_HIGHEST, true );

			// restore setting of reading punctuation
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KAKKO), new IntPtr(readsKakko) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KIGOU), new IntPtr(readsKigou) );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_RPITCH), new IntPtr(pitch) );
		}

		/// <summary>
		/// ǂݏグ~
		/// </summary>
		public void Stop()
		{
			PCTKVReset();
		}

		/// <summary>
		/// ǂݏグɁuԁv}
		/// </summary>
		public void Wait()
		{
			// disable reading punctuation
			UInt32 readsKuten = PCTKGetStatus( PS_STATUS, new IntPtr(PKDI_KUTEN), IntPtr.Zero );
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KUTEN), IntPtr.Zero );

			// read punctuation to wait a moment
			PCTKPRead( "A", 5, false );

			// restore setting of reading punctuation
			PCTKSetStatus( PS_STATUS, new IntPtr(PKDI_KUTEN), new IntPtr(readsKuten) );
		}

		#region Inner functions
		[DllImport("PCTKUSR.dll")]
		static extern bool PCTKStatus();

		[DllImport("PCTKUSR.dll")]
		static extern UInt32 PCTKGetStatus( UInt32 item, IntPtr param1, IntPtr param2 );
		
		[DllImport("PCTKUSR.dll")]
		static extern UInt32 PCTKSetStatus( UInt32 item, IntPtr param1, IntPtr param2 );

		[DllImport("PCTKUSR.dll", CharSet=CharSet.Ansi)]
		static extern void PCTKPRead( string message, Int32 priority, bool analyze );
		
		[DllImport("PCTKUSR.dll")]
		static extern void PCTKVReset();
		#endregion
	}

	/// <summary>
	/// JAWS PȃXs[J[ƂĎg߂̃NX
	/// </summary>
	public class JawsSpeaker : ISpeaker
	{
		static ISpeaker _Instance = null;
		static string _JawsDir = null;
		
		JawsSpeaker()
		{}

		/// <summary>
		/// B̃CX^X擾
		/// </summary>
		public static ISpeaker Instance
		{
			get
			{
				if( _Instance == null )
				{
					_Instance = new JawsSpeaker();
				}
				return _Instance;
			}
		}

		/// <summary>
		/// JAWS Xs[J[ƂĎg邩
		/// </summary>
		/// <returns>gꍇ true</returns>
		public static bool IsAlive()
		{
			string env_path;
			
			// get installation target of JAWS (for 6/7)
			GetJawsDir();
			if( _JawsDir == null )
			{
				return false; // not installed
			}

			// add to %PATH% env_var for DllImport (effects this process only)
			env_path = GetEnvVar( "PATH" );
			if( env_path.IndexOf(_JawsDir) == -1 )
			{
				SetEnvVar( "PATH", _JawsDir+";"+env_path );
			}

			// ensure JAWS is running
			try
			{
				if( JFWSayString("", false) == false )
				{
					return false; // not running
				}
			}
			catch( DllNotFoundException )
			{
				return false;
			}

			return true;
		}

		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void Speak( string message )
		{
			JFWSayString( message, false );
		}

		/// <summary>
		/// ŕǂݏグB
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void SpeakSub( string message )
		{
			System.Diagnostics.Debug.Fail( "Sgry.Jaws.SpeakSub ͖ł" );
		}

		/// <summary>
		/// ǂݏグ~
		/// </summary>
		public void Stop()
		{
			//JFWStopSpeech(); // sgry: this API did nothing in my environment
			JFWRunScript( "StopSpeech" );
		}

		/// <summary>
		/// ǂݏグɁuԁv}
		/// </summary>
		public void Wait()
		{
			JFWSayString( "A", false );
		}

		#region Inner functions
		static void GetJawsDir()
		{
			RegistryKey key;
			
			if( _JawsDir == null )
			{
				// try to get directory of JAWS 7.1
				key = Registry.LocalMachine.OpenSubKey( @"SOFTWARE\Freedom Scientific\JAWS\7.10" );
				if( key != null )
				{
					_JawsDir = (string)key.GetValue( "Target" );
				}
				
				// try to get directory of JAWS 6.2
				key = Registry.LocalMachine.OpenSubKey( @"SOFTWARE\Freedom Scientific\JAWS\6.20" );
				if( key != null )
				{
					_JawsDir = (string)key.GetValue( "Target" );
				}
			}
		}

		static string GetEnvVar( string name )
		{
			StringBuilder	str	= null;
			uint			length;
			uint			rc; // result code
			
			// get variable length
			length = GetEnvironmentVariable( name, str, (uint)0 );
			if( length == 0 )
			{
				return null; // no such variable
			}
            
			// get variable
			str = new StringBuilder( (int)length );
			rc = GetEnvironmentVariable( name, str, length );
			if( rc == 0 )
			{
				return null; // failed to get variable
			}

			return str.ToString();
		}

		static void SetEnvVar( string name, string value )
		{
			bool rc; // result code
			
			rc = SetEnvironmentVariable( name, value );
			if( rc == false )
			{
				throw new Exception();
			}
		}

		[DllImport("jfwapi.dll")]
		static extern bool JFWSayString( string message, bool interrupt );

		//[DllImport("jfwapi.dll")]
		//static extern bool JFWStopSpeech();

		[DllImport("jfwapi.dll", CharSet=CharSet.Ansi)]
		static extern bool JFWRunScript( string scriptName );

		[DllImport("kernel32.dll", EntryPoint="SetEnvironmentVariableW", CharSet=CharSet.Unicode)]
		static extern bool SetEnvironmentVariable( string name, string val );

		[DllImport("kernel32.dll", EntryPoint="GetEnvironmentVariableW", CharSet=CharSet.Unicode)]
		static extern uint GetEnvironmentVariable( string name, StringBuilder str, uint strLength );
		#endregion
	}
	
	/// <summary>
	/// FocusTalk PȃXs[J[ƂĎg߂̃NX
	/// </summary>
	public class FocusTalkSpeaker : ISpeaker
	{
		const int FT_OUTPUT_ABSOLUTE = 0x01;
		const int FT_OUTPUT_KEYDOWN = 0x20;

		static ISpeaker _Instance = null;

		FocusTalkSpeaker()
		{}

		/// <summary>
		/// B̃CX^X擾
		/// </summary>
		public static ISpeaker Instance
		{
			get
			{
				if( _Instance == null )
				{
					_Instance = new FocusTalkSpeaker();
				}
				return _Instance;
			}
		}

		/// <summary>
		/// FocusTalk Xs[J[ƂĎg邩
		/// </summary>
		/// <returns>gꍇ true</returns>
		public static bool IsAlive()
		{
			// API DLL ̊֐𐳏 P/Invoke łΗp\
			try
			{
				int rc = FT_OutputVoiceText( "", 0 );
				if( rc == 0 )
				{
					return false;
				}
			}
			catch( DllNotFoundException )
			{
				return false;
			}
			
			return true;
		}

		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void Speak( string message )
		{
			FT_OutputVoiceText( message, FT_OUTPUT_ABSOLUTE );
		}

		/// <summary>
		/// ŕǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void SpeakSub( string message )
		{
			int voiceType, subVoiceType;
			
			try
			{
				// ݂̐ɉĕ̐
				voiceType = FT_GetVoiceType();
				if( voiceType == 0 )
					subVoiceType = 1;
				else
					subVoiceType = 0;

				// 𔭐
				FT_ChangeVoiceType( subVoiceType );
				FT_OutputVoiceText( message, 0 );
				FT_ChangeVoiceType( voiceType );
			}
			catch( EntryPointNotFoundException )
			{
				// o[W2.0.2ȍ~łȂH
				// 剹ŔBŒɂB
				FT_OutputVoiceText( message, FT_OUTPUT_ABSOLUTE|FT_OUTPUT_KEYDOWN );
			}
		}

		/// <summary>
		/// ǂݏグ~
		/// </summary>
		public void Stop()
		{
			FT_StopSound();
		}

		/// <summary>
		/// ǂݏグɁuԁv}
		/// </summary>
		public void Wait()
		{
			System.Diagnostics.Debug.Fail( "Sgry.FocusTalkSpeaker.Wait ͖ł" );
		}

		#region Inner functions
		[DllImport("FocusTalkExt.dll", CharSet=CharSet.Ansi)]
		static extern Int32 FT_OutputVoiceText( string text, Int32 flags );
		
		[DllImport("FocusTalkExt.dll")]
		static extern Int32 FT_ChangeVoiceType( Int32 type );
		
		[DllImport("FocusTalkExt.dll")]
		static extern Int32 FT_GetVoiceType();
		
		[DllImport("FocusTalkExt.dll")]
		static extern bool FT_StopSound();
		#endregion
	}

	/// <summary>
	/// 95Reader PȃXs[J[ƂĎg߂̃NX
	/// </summary>
	public class XpReaderSpeaker : ISpeaker
	{
		const int PF_DONT_IGNORE = 0x02;
		const int PF_TONE_DOWN = 0x20;
		const int PF_NATURAL_PROOF = 0x40;

		static ISpeaker _Instance = null;
		static bool _DoesNotExist = false;

		XpReaderSpeaker()
		{}

		/// <summary>
		/// B̃CX^X擾
		/// </summary>
		public static ISpeaker Instance
		{
			get
			{
				if( _Instance == null )
				{
					_Instance = new XpReaderSpeaker();
				}
				return _Instance;
			}
		}

		/// <summary>
		/// 95Reader Xs[J[ƂĎg邩
		/// </summary>
		/// <returns>gꍇ true</returns>
		public static bool IsAlive()
		{
			// ł DLL ƂĂꍇ͉Ȃ
			if( _DoesNotExist )
				return false;

			try
			{
				// API DLL ̊֐𐳏 P/Invoke ł邩mF
				SoundMessage( "", 0 );
				
				// 95Reader ̃EBhȆ݂mF
				if( FindWindow("95Reader",0) == 0 )
				{
					return false;
				}
			}
			catch( DllNotFoundException )
			{
				_DoesNotExist = true;
				return false;
			}
			
			return true;
		}

		/// <summary>
		/// ǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void Speak( string message )
		{
			/*if( message.Length == 1
				|| (0 < message.Length && Char.IsPunctuation(message, 0)) )
			{
				flag |= PF_NATURAL_PROOF;
			}*/

			SoundMessage( message, PF_DONT_IGNORE | PF_NATURAL_PROOF );
		}

		/// <summary>
		/// ŕǂݏグ
		/// </summary>
		/// <param name="message">ǂݏグ镶</param>
		public void SpeakSub( string message )
		{
			SoundMessage( message, PF_DONT_IGNORE|PF_TONE_DOWN|PF_NATURAL_PROOF );
		}

		/// <summary>
		/// ǂݏグ~
		/// </summary>
		public void Stop()
		{
			SoundStop();
		}

		/// <summary>
		/// ǂݏグɁuԁv}
		/// </summary>
		public void Wait()
		{
			SoundMessage( "A", 2 );
		}

		#region Inner Functions
		[DllImport("user32.dll", EntryPoint="FindWindowW", CharSet=CharSet.Unicode)]
		static extern Int32 FindWindow( string windowClassName, int windowText );
		
		[DllImport("SOUNDP.dll", CharSet=CharSet.Ansi)]
		static extern Int32 SoundMessage( string text, Int32 flags );
		
		[DllImport("SOUNDP.dll")]
		static extern bool SoundStop();
		#endregion
	}
}

/*
Version History

[v1.3.2] 2008-10-05
EAutoSpeaker.IsAlive  DummySpeaker ȊOCX^XĂ true Ɂi1.3.1 ̏CӖsȂ̂ōďCj

[v1.3.1] 2007-10-12
EXN[[_NAutoSpeaker.IsAlivefalseԂCiňłDummySpeaker݂邽ߏtruej

[v1.3.0] 2007-08-04
Eł̓ǂݏグ@\ǉ

[v1.2.3] 2007-06-29
EXP Reader {̂̋Lǂݏグ̐ݒ𔽉f悤
EPC-Talker ŋLJbRނKǂݏグ悤

[v1.2.2] 2007-06-22
ERgM

[v1.2.1] 2007-02-24
EXPReaderSpeaker P
EeXs[J Singleton 

[v1.2.0] 2007-02-21
EISpeaker.Wait ǉ

[v1.1.0] 2007-02-20
EJawsSpeaker.Stop ̎ύXiO͊ɂʖj
EJAWS 7.1 ̐ɑΉ

[v1.0.1] 2007-01-09
EW[ւ̈ˑ𖳂

[v1.0.0] 2006-11-30
E[X
*/

/**********************************************************
Copyright (c) 2006-2008 YAMAMOTO Suguru

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
copies of the Software, and to permit persons to whom the Software is 
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
THE SOFTWARE.
**********************************************************/
