// file: SgRichEdit.cs
// brief: a control which wraps native RichEdit20W window
// encoding: UTF-8
// update: 2012-03-25
//=========================================================
using System;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using Sgry;

using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.Gui
{
	/// <summary>
	/// A control which wraps native RichEdit20W window
	/// with some features for source code editing usage.
	/// </summary>
	/// <remarks>
	/// Because native window has "RichEdit20W" window class name,
	/// this can be read by all screen-readers (Windows.Forms.RichTextBox cannot).
	/// </remarks>
	public class SgRichEdit : Control, ITextEditor
	{
		/// <summary>
		/// ウィンドウメッセージをプレビューするデリゲート。
		/// </summary>
		public delegate int WmHookProc( IntPtr hWnd, UInt32 msg, IntPtr wParam, IntPtr lParam, out IntPtr lresult );

		#region Fields
		static RichEdit20Utl _NativeApi = new RichEdit20Utl();
		IntPtr _Rich;
		bool _WordWrap = false;
		bool _AcceptsTab = true;
		bool _IsDirty = false;
		WmHookProc _HookProc = null;
		bool _IsOverwriteMode = false;
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成
		/// </summary>
		public SgRichEdit()
		{
			RecreateRichEdit();
		}

		void RecreateRichEdit()
		{
			const int ENM_CHANGE = 0x00000001;
			const int ENM_SELCHANGE = 0x00080000;

			if( _Rich != null )
			{
				Utl.DestroyWindow( _Rich );
				_Rich = IntPtr.Zero;
			}

			// create window and do some settings
			_Rich = Utl.CreateRichEdit( Handle, WordWrap );
			this.Font = base.Font;
			_NativeApi.SetUndoLimit( _Rich, 256 );
			_NativeApi.SetEventMask( _Rich, ENM_CHANGE | ENM_SELCHANGE );
			_NativeApi.SetSysEditEmulation( _Rich, true );

			// ウィンドウメッセージフックをインストールする
			if( _HookProc == null )
			{
				_HookProc = new WmHookProc( WindowMessageHookProc );
			}
			Utl.SendMessage( _Rich, Win32.WM_USER+11, _HookProc, IntPtr.Zero );
		}

		/// <summary>
		/// リソースを解放
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			base.Dispose( disposing );

			// このオブジェクトをゴミ回収対象に再登録する。
			// というのも、何故かゴミ回収対象から外れるため。
			// またオブジェクト回収後にdisposing==falseで呼び出されるので、
			// 同条件で再登録すると無限に回収し続ける事になる。それは避ける。
			if( disposing )
			{
				GC.ReRegisterForFinalize( this );
			}
			Utl.DestroyWindow( _Rich );
		}
		#endregion

		#region States
		/// <summary>
		/// 未編集状態かどうかを取得または設定します。
		/// </summary>
		[Browsable(false)]
		public bool IsDirty
		{
			get{ return _IsDirty; }
			set
			{
				bool valueChanged = (_IsDirty != value);
				
				_IsDirty = value;
				if( valueChanged )
				{
					InvokeDirtyStateChanged();
				}
			}
		}
		#endregion

		#region Editing Modes
		/// <summary>
		/// 自動インデントモードを表す内部既定値を取得または設定します。
		/// 0 が無し、1 が通常の自動インデント、2 が C 言語用の自動インデントです。
		/// </summary>
		public int AutoIndentMode
		{
			get
			{
				const int RAM_GETAUTOINDENTMODE = 0x400 + 2;
				IntPtr rc = Utl.SendMessage( _Rich, RAM_GETAUTOINDENTMODE, IntPtr.Zero, IntPtr.Zero );
				return rc.ToInt32();
			}
			set
			{
				const int RAM_SETAUTOINDENTMODE = 0x400 + 1;
				Utl.SendMessage( _Rich, RAM_SETAUTOINDENTMODE, new IntPtr((int)value), IntPtr.Zero );
			}
		}

		/// <summary>
		/// このコントロールがラッピングしている RichEdit のハンドル
		/// </summary>
		[Browsable(false)]
		public IntPtr RichEditHandle
		{
			get{ return _Rich; }
		}

		/// <summary>
		/// 上書きモードかどうかを取得または設定します。
		/// </summary>
		[Browsable(false)]
		public bool IsOverwriteMode
		{
			get{ return _IsOverwriteMode; }
			set{ _IsOverwriteMode = value; }
		}

		/// <summary>
		/// 編集内容が読み取り専用かどうか
		/// </summary>
		[DefaultValue(false)]
		public bool ReadOnly
		{
			get{ return _NativeApi.IsReadonly(_Rich); }
			set{ _NativeApi.SetReadonly(_Rich, value); }
		}

		/// <summary>
		/// Home キーで行頭ではなく最初の非空白文字にキャレットを移動するかどうか
		/// </summary>
		[DefaultValue(true)]
		public bool UseSmartHome
		{
			get
			{
				const int RAM_GETSMARTHOME = 0x400 + 10;
				IntPtr rc = Utl.SendMessage( _Rich, RAM_GETSMARTHOME, IntPtr.Zero, IntPtr.Zero );
				return (rc.ToInt32() == 1);
			}
			set
			{
				const int RAM_SETSMARTHOME = 0x400 + 9;
				Utl.SendMessage( _Rich, RAM_SETSMARTHOME, new IntPtr(value ? 1 : 0), IntPtr.Zero );
			}
		}

		/// <summary>
		/// 複数行のテキストを扱うかどうか
		/// </summary>
		[DefaultValue(true)]
		public bool Multiline
		{
			get{ return _NativeApi.IsMultiline(_Rich); }
			set{ _NativeApi.SetMultiline(_Rich, value); }
		}
		
		/// <summary>
		/// 長い行を折り返し表示するかどうか。
		/// 変更するとウィンドウを再生成し、UNDO バッファはクリアされる。
		/// </summary>
		[DefaultValue(false)]
		public bool WordWrap
		{
			get{ return _WordWrap; }
			set
			{
				_WordWrap = value;
				bool readOnly, acceptsTab, acceptsReturn;

				// backup state
				string text = Text;
				int caretIndex = GetCaretIndex();
				readOnly = ReadOnly;
				acceptsTab = AcceptsTab;
				acceptsReturn = AcceptsReturn;

				// recreate window
				RecreateRichEdit();

				// restore state
				Text = text;
				SetSelection( caretIndex, caretIndex );
				ReadOnly = readOnly;
				AcceptsTab = acceptsTab;
				AcceptsReturn = acceptsReturn;

				// simulate resize event to update native window's size
				OnResize( EventArgs.Empty );
			}
		}

		/// <summary>
		/// 指定ファイル種別用のモードに切り替える。
		/// 非対応の種別名を指定すると標準モードになる。
		/// </summary>
		public void SetFileType( string typeName, AutoIndentMode indentMode )
		{
			string typeName_l = typeName.ToLower();
			int indentLevel = (int)indentMode;

			if( typeName_l == "c/c++" )
			{
				if     ( indentLevel == 2 ) AutoIndentMode = 2;
				else if( indentLevel == 1 ) AutoIndentMode = 1;
				else                        AutoIndentMode = 0;
			}
			else if( typeName_l == "java" )
			{
				if     ( indentLevel == 2 ) AutoIndentMode = 2;
				else if( indentLevel == 1 ) AutoIndentMode = 1;
				else                        AutoIndentMode = 0;
			}
			else if( typeName_l == "c#" )
			{
				if     ( indentLevel == 2 ) AutoIndentMode = 2;
				else if( indentLevel == 1 ) AutoIndentMode = 1;
				else                        AutoIndentMode = 0;
			}
			else if( typeName_l == "ruby" )
			{
				if( 0 < indentLevel ) AutoIndentMode = 1;
				else                  AutoIndentMode = 0;
			}
			else if( typeName_l == "xml" )
			{
				if( 0 < indentLevel ) AutoIndentMode = 1;
				else                  AutoIndentMode = 0;
			}
			else
			{
				AutoIndentMode = 0;
			}
		}
		#endregion

		#region Text / Font
		/// <summary>
		/// 現在入力されているテキスト
		/// </summary>
		public override string Text
		{
			get{ return GetText(); }
			set{ SetText(value); }
		}

		/// <summary>
		/// 現在入力されているテキストをコピーして取得ます。
		/// </summary>
		public string GetText()
		{
			return _NativeApi.GetText( _Rich );
		}

		/// <summary>
		/// 現在入力されているテキストを指定した改行コードで取得します。
		/// </summary>
		public string GetText( EolCode eolCode )
		{
			return GetText().Replace( "\r", Utl.ToString(eolCode) );
		}

		/// <summary>
		/// テキストの内容をすべて置き換えます。
		/// </summary>
		public void SetText( string text )
		{
			// 全テキストを置き換える
			SelectAll();
			Replace( text );

			// UNDO 情報をクリア
			ClearUndo();
		}

		/// <summary>
		/// 現在入力されているテキストの文字数を取得
		/// </summary>
		/// <returns>入力されているテキストの文字数</returns>
		[Browsable(false)]
		public int TextLength
		{
			get{ return _NativeApi.GetTextLength(RichEditHandle); }
		}

		/// <summary>
		/// 指定位置の文字を取得
		/// </summary>
		/// <exception cref="ArgumentOutOfRangeException">インデックスが有効な範囲にありません。</exception>
		public char GetCharAt( int index )
		{
			Debug.Assert( 0 <= index, "index must be greater than zero." );
			if( TextLength <= index )
				throw new ArgumentOutOfRangeException( "invalid index was requested; text length is "+ TextLength +" but requested index is "+ index +")" );

			string s = _NativeApi.GetTextRange( _Rich, index, index+1 );
			return s[0];
		}

		/// <summary>
		/// Replace currently selected text into the specified text.
		/// If nothing selected, specified text will be inserted at caret pos.
		/// </summary>
		public void Replace( string text )
		{
			_NativeApi.ReplaceText( _Rich, text );
		}

		/// <summary>
		/// テキストをすべて削除
		/// </summary>
		public void Clear()
		{
			SetText( String.Empty );
		}

		/// <summary>
		/// テキストの入力に使うフォント
		/// </summary>
		public override Font Font
		{
			get
			{
				//Win32.SendMessage( _Rich, Win32.WM_GETFONT, IntPtr.Zero, IntPtr.Zero );
				return base.Font;
			}
			set
			{
				base.Font = value;
				Win32.SendMessage( _Rich, Win32.WM_SETFONT, value.ToHfont(), IntPtr.Zero );
			}
		}
		#endregion

		#region Selection
		/// <summary>
		/// 現在選択されているテキスト
		/// </summary>
		[Browsable(false)]
		public string SelectedText
		{
			get{ return _NativeApi.GetSelectedText( _Rich ); }
			set{ _NativeApi.SetSelectedText( _Rich, value ); }
		}

		/// <summary>
		/// 選択が後方へと行われているかどうか
		/// </summary>
		[Browsable(false)]
		public bool SelectingToEnd
		{
			get
			{
				int begin, end;
				_NativeApi.GetSelection( _Rich, out begin, out end );
				return (SelectionAnchor <= begin);
			}
		}

		/// <summary>
		/// 選択の起点のインデックスを取得
		/// </summary>
		[Browsable(false)]
		public int SelectionAnchor
		{
			get{ return _NativeApi.GetSelectionAnchor(_Rich); }
		}

		/// <summary>
		/// テキストの選択範囲を「以上未満」で取得。
		/// SetSelection と違い、カーソルやアンカーの位置は関係ない。
		/// </summary>
		/// <param name="begin">選択されている最初の文字のインデックス</param>
		/// <param name="end">選択されている最後の文字のインデックス</param>
		public void GetSelection( out int begin, out int end )
		{
			_NativeApi.GetSelection( _Rich, out begin, out end );
		}
		
		/// <summary>
		/// テキストの選択状態を設定。
		/// </summary>
		/// <param name="anchor">選択のアンカーのインデックス</param>
		/// <param name="caretIndex">キャレットのインデックス</param>
		public void SetSelection( int anchor, int caretIndex )
		{
			_NativeApi.SetSelection( _Rich, anchor, caretIndex );
		}

		/// <summary>
		/// 入力されているすべてのテキストを選択
		/// </summary>
		public void SelectAll()
		{
			this.SetSelection( 0, -1 );
		}
		#endregion

		#region Editing actions
		/// <summary>
		/// UNDO を実行
		/// </summary>
		public void Undo()
		{
			Win32.SendMessage( _Rich, Win32.EM_UNDO, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// UNDO バッファをクリア
		/// </summary>
		public void ClearUndo()
		{
			Win32.SendMessage( _Rich, Win32.EM_EMPTYUNDOBUFFER, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// 選択しているテキストを切り取る
		/// </summary>
		public void Cut()
		{
			Win32.SendMessage( _Rich, Win32.WM_CUT, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// 選択しているテキストをクリップボードにコピー
		/// </summary>
		public void Copy()
		{
			Win32.SendMessage( _Rich, Win32.WM_COPY, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// 選択しているテキストを削除
		/// </summary>
		public void Delete()
		{
			Win32.SendMessage( _Rich, Win32.WM_CLEAR, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// クリップボードの内容を貼り付ける
		/// </summary>
		public void Paste()
		{
			Win32.SendMessage( _Rich, Win32.WM_PASTE, IntPtr.Zero, IntPtr.Zero );
		}
		#endregion

		#region GUI
		/// <summary>
		/// Tab キーをフォーカス移動キーではなく入力キーとして扱うかどうか
		/// </summary>
		[DefaultValue(true)]
		public bool AcceptsTab
		{
			get{ return _AcceptsTab; }
			set{ _AcceptsTab = value; }
		}

		/// <summary>
		/// Enter キーを入力キーとして扱うかどうか
		/// </summary>
		[DefaultValue(true)]
		public bool AcceptsReturn
		{
			get{ return _NativeApi.AcceptsReturn(_Rich); }
			set{ _NativeApi.SetAcceptsReturn(_Rich, value); }
		}

		/// <summary>
		/// キャレットが表示されるようにスクロールする
		/// </summary>
		public void ScrollToCaret()
		{
			Win32.SendMessage( _Rich, Win32.EM_SCROLLCARET, IntPtr.Zero, IntPtr.Zero );
		}

		/// <summary>
		/// 指定したインデックスが指す文字のスクリーン座標を取得
		/// </summary>
		/// <param name="index">スクリーン座標を取得する文字のインデックス</param>
		/// <returns>スクリーン座標</returns>
		public Point GetPositionFromCharIndex( int index )
		{
			return _NativeApi.GetPositionFromCharIndex( _Rich, index );
		}
		#endregion

		#region Line Level Handling
		/// <summary>
		/// 読み込んでいるファイルの行数を取得
		/// </summary>
		/// <returns>ファイルの行数</returns>
		public int GetLineCount()
		{
			return _NativeApi.GetLineCount( _Rich );
		}

		/// <summary>
		/// 指定行の長さ（文字数）を取得
		/// </summary>
		/// <param name="lineIndex">長さを取得したい行のインデックス</param>
		/// <returns>行の長さ（文字数）</returns>
		public int GetLineLength( int lineIndex )
		{
			return _NativeApi.GetLineLength( _Rich, lineIndex );
		}

		/// <summary>
		/// 指定行を取得
		/// </summary>
		/// <param name="lineIndex">取得したい行のインデックス</param>
		/// <returns>指定行の内容</returns>
		public string GetLineContent( int lineIndex )
		{
			return _NativeApi.GetLine( _Rich, lineIndex );
		}

		/// <summary>
		/// 指定行の先頭文字の、テキスト全体でのインデックスを取得
		/// </summary>
		/// <param name="lineIndex">行頭文字インデックスを取得したい行の番号</param>
		/// <returns>インデックス（失敗時:-1）</returns>
		public int GetLineHeadIndex( int lineIndex )
		{
			return _NativeApi.GetLineHeadIndex( _Rich, lineIndex );
		}

		/// <summary>
		/// 指定したインデックスが指す文字を含む行の番号を取得
		/// </summary>
		/// <param name="index">このインデックスが指す文字を含む行の番号を取得します。</param>
		/// <returns>行番号</returns>
		public int GetLineIndexFromCharIndex( int index )
		{
			return _NativeApi.GetLineIndexFromCharIndex( _Rich, index );
		}
		#endregion

		#region Word Level Handling
		/// <summary>
		/// 指定したインデックスを含む単語を取得
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>単語</returns>
		public string GetWordAt( int index )
		{
			return _NativeApi.GetWordAt( _Rich, index );
		}

		/// <summary>
		/// 指定した位置から末尾方向へ、単語の開始位置を検索
		/// </summary>
		public int IndexOfNextWordStart( int index )
		{
			return _NativeApi.GetNextWordStart( _Rich, index );
		}

		/// <summary>
		/// 指定した位置から末尾方向へ、単語の終了位置を検索
		/// </summary>
		public int IndexOfNextWordEnd( int index )
		{
			return _NativeApi.GetNextWordEnd( _Rich, index );
		}

		/// <summary>
		/// 指定した位置から先頭方向へ、単語の開始位置を検索
		/// </summary>
		public int IndexOfPreviousWordStart( int index )
		{
			return _NativeApi.GetPreviousWordStart( _Rich, index );
		}

		/// <summary>
		/// 指定した位置から先頭方向へ、単語の終了位置を検索
		/// </summary>
		public int IndexOfPreviousWordEnd( int index )
		{
			return _NativeApi.GetPreviousWordEnd( _Rich, index );
		}
		#endregion

		#region Caret Handling
		/// <summary>
		/// キャレット位置のインデックスを取得
		/// </summary>
		public int GetCaretIndex()
		{
			return _NativeApi.GetCaretIndex( _Rich );
		}

		/// <summary>
		/// キャレット位置のインデックスを取得
		/// </summary>
		/// <param name="lineIndex">キャレットの行インデックス</param>
		/// <param name="columnIndex">キャレットの桁インデックス</param>
		public void GetCaretIndex( out int lineIndex, out int columnIndex )
		{
			_NativeApi.GetCaretIndex( _Rich, out lineIndex, out columnIndex );
		}

		/// <summary>
		/// キャレット位置を設定
		/// </summary>
		/// <param name="lineIndex">キャレット移動先の行インデックス</param>
		/// <param name="columnIndex">キャレット移動先の桁インデックス</param>
		public void SetCaretIndex( int lineIndex, int columnIndex )
		{
			_NativeApi.SetCaretIndex( _Rich, lineIndex, columnIndex );
		}
		#endregion

		#region MSAA feature
		/// <summary>
		/// 独自 MSAA サーバ機能が有効かどうか
		/// </summary>
		public bool EnableMsaa
		{
			set
			{
				const int RAM_SETENABLEMSAA = 0x400 + 7;
				int enable = value ? 1 : 0;
				Utl.SendMessage( _Rich, RAM_SETENABLEMSAA, new IntPtr(enable), IntPtr.Zero );
			}
			get
			{
				const int RAM_GETENABLEMSAA = 0x400 + 8;
				IntPtr rc = Utl.SendMessage( _Rich, RAM_GETENABLEMSAA, IntPtr.Zero , IntPtr.Zero );
				return (rc != IntPtr.Zero);
			}
		}
		#endregion

		#region Wrapping code to make this a container
		/// <summary>
		/// MSAA プロパティの AccessibleName
		/// </summary>
		public new string AccessibleName
		{
			get
			{
				const int RAM_GETACCNAME = (0x0400 + 4);
				string accName;
				unsafe
				{
					char* buf = stackalloc char[256];
					
					Utl.SendMessage( _Rich, RAM_GETACCNAME, buf, (void*)255 );
					accName = new string( buf );
				}
				return accName;
			}
			set
			{
				const int RAM_SETACCNAME = (0x0400 + 3);
				Utl.SendMessage( _Rich, RAM_SETACCNAME, "", IntPtr.Zero );
				base.AccessibleName = value;
			}
		}

		/// <summary>
		/// このコントロールにフォーカスが当たっているかどうか
		/// </summary>
		public override bool Focused
		{
			get{ return (Win32.GetFocus() == _Rich); }
		}

		/// <summary>
		/// このコントロールにフォーカスする
		/// </summary>
		public new bool Focus()
		{
			IntPtr prevFocusedWindow;

			base.Focus(); // .NET を騙すためにこれが必要
			prevFocusedWindow = Win32.SetFocus( _Rich );
			return (prevFocusedWindow != IntPtr.Zero);
		}

		#region dialog focusing key handling
		/// <summary>
		/// define input keys for RichEdit control
		/// </summary>
		protected override bool IsInputKey( Keys keyData )
		{
			Keys modifier = (keyData & Keys.Modifiers);
			Keys key = (keyData ^ modifier);

			// tab without control?
			if( key == Keys.Tab && AcceptsTab )
			{
				if( (modifier & Keys.Control) == 0 )
					return true;
			}
			if( key == Keys.Right )
				return true; // right with shift/ctrl/alt
			if( key == Keys.Left )
				return true; // left with shift/ctrl/alt
			if( key == Keys.Up )
				return true; // up with shift/ctrl/alt
			if( key == Keys.Down )
				return true; // down with shift/ctrl/alt
			if( key == Keys.ShiftKey )
				return true; // shift only
			if( key == Keys.ControlKey )
				return true; // ctrl only

			if( Keys.A <= key && key <= Keys.Z ) // 20070826 note: is this code needed?
			{
				if( modifier == Keys.None || modifier == Keys.Shift )
					return true;
			}

			return false;
		}
		
		/// <summary>
		/// override focusing strategy; use Ctrl+Tab for moving focus instead of Tab
		/// </summary>
		protected override bool ProcessDialogKey( Keys keyData )
		{
			// if Ctrl+Tab comes, act as if only Tab comes
			if( keyData == (Keys.Tab|Keys.Control) )
				return base.ProcessDialogKey( Keys.Tab );

			// if Ctrl+Shift+Tab comes, treat it a Shift+Tab
			if( keyData == (Keys.Tab | Keys.Control | Keys.Shift) )
				return base.ProcessDialogKey( (Keys.Tab | Keys.Shift) );

			return base.ProcessDialogKey( keyData );
		}
		#endregion

		#region dialog mnemonic charactor handling
		/// <remarks>
		/// RichEdit uses all letters, digits and CR (char of Enter key)
		/// </remarks>
		protected override bool IsInputChar( char charCode )
		{
			if( !ReadOnly )
			{
				if( Char.IsLetterOrDigit(charCode) )
					return true;
				if( charCode == '\r' )
					return true;
			}

			return false;
		}
		//protected override bool ProcessDialogChar( char charCode ){...}
		#endregion

		/// <summary>
		/// ラッパのウィンドウプロシージャ。
		/// 隠蔽している RichEdit からの通知を受け取りイベントに変換する。
		/// </summary>
		protected override void WndProc( ref Message m )
		{
			const int EN_CHANGE = 0x300;

			// handle Win32 command message
			if( m.Msg == Win32.WM_COMMAND )
			{
				// if it's EN_CHANGE, invoke event
				if( Utl.HiWord(m.WParam.ToInt32()) == EN_CHANGE )
				{
					InvokeContentChanged();

					// 初回変更であればダーティフラグを更新
					if( _IsDirty == false )
					{
						_IsDirty = true;
						InvokeDirtyStateChanged();
					}
					return;
				}
			}

			base.WndProc( ref m );
			
			// parse notify message
			if( m.Msg == Win32.WM_NOTIFY )
			{
				// if it's EN_SELCHANGE, invoke event
				if( Utl.IsSelectionChangedEvent(ref m) )
				{
					InvokeSelectionChanged();
				}
			}
			// invoke container's GotFocus event when the child got focus
			else if( m.Msg == Win32.WM_SETFOCUS )
			{
				OnGotFocus( EventArgs.Empty );
			}
		}

		/// <summary>
		/// ラッパーのウィンドウが生成された直後の動作
		/// </summary>
		protected override void OnHandleCreated( EventArgs e )
		{
			base.OnHandleCreated( e );

			// ウィンドウサイズを調整するため、親がリサイズされた事にする
			OnResize( EventArgs.Empty );
		}

		/// <summary>
		/// ラッパーのウィンドウが破壊された直後の動作
		/// </summary>
		protected override void OnHandleDestroyed( EventArgs e )
		{
			base.OnHandleDestroyed( e );

			if( _Rich != IntPtr.Zero )
			{
				//NO_NEED//Utl.DestroyWindow( _Rich ); // all children are destroyed with parent
				_Rich = IntPtr.Zero;
			}
		}

		/// <summary>
		/// ラッパーがサイズ変更された直後の動作
		/// </summary>
		protected override void OnResize( EventArgs eventargs )
		{
			base.OnResize( eventargs );

			Utl.SetWindowPos( _Rich, IntPtr.Zero, 0, 0, this.Width, this.Height, 0 );
		}

		/// <summary>
		/// ラッパーにフォーカスが当たった直後の動作
		/// </summary>
		protected override void OnGotFocus( EventArgs e )
		{
			base.OnGotFocus( e );

			Win32.SetFocus( _Rich );
		}
		#endregion

		/// <summary>
		/// キーが押し下げられた直後の動作。
		/// </summary>
		protected override void OnKeyDown( KeyEventArgs e )
		{
			if( e.Handled )
			{
				return;
			}
			base.OnKeyDown( e );
		}

		#region Events
		/// <summary>
		/// 内容が変化した時に発生します。
		/// </summary>
		public event EventHandler ContentChanged;
		void InvokeContentChanged()
		{
			if( ContentChanged != null )
			{
				ContentChanged( this, EventArgs.Empty );
			}
		}

		/// <summary>
		/// 選択状態が変化した時に発生します。
		/// 無選択状態でキャレットが移動した時にも発生します。
		/// </summary>
		public event EventHandler SelectionChanged;
		
		/// <summary>
		/// Invoke SelectionChanged event.
		/// </summary>
		protected void InvokeSelectionChanged()
		{
			if( SelectionChanged != null )
			{
				SelectionChanged( this, EventArgs.Empty );
			}
		}

		/// <summary>
		/// Occurs when a first modification occured in the document.
		/// </summary>
		public event EventHandler DirtyStateChanged;
		void InvokeDirtyStateChanged()
		{
			if( DirtyStateChanged != null )
				DirtyStateChanged( this, EventArgs.Empty );
		}
		#endregion

		#region Window Procedure Hook
		/// <summary>
		/// ウィンドウメッセージのフック関数です。
		/// </summary>
		/// <returns>SgRichEdit の標準動作を行う必要が無ければ 1 を、さもなければ 0 を返します。</returns>
		protected virtual int WindowMessageHookProc( IntPtr hWnd, uint msg, IntPtr wParam, IntPtr lParam, out IntPtr lresult )
		{
			if( msg == Win32.WM_KEYDOWN )
			{
				Keys keyData = ((Keys)wParam | Form.ModifierKeys);
				KeyEventArgs args = new KeyEventArgs( keyData );
				OnKeyDown( args );
				lresult = IntPtr.Zero;
				return (args.Handled ? 1 : 0);
			}
			else if( msg == Win32.WM_KEYUP )
			{
				Keys keyData = ((Keys)wParam | Form.ModifierKeys);

				// イベントを発行
				OnKeyUp( new KeyEventArgs(keyData) );

				// API で状態取得できない上書きモードかどうかを記録しておく
				if( keyData == Keys.Insert )
				{
					_IsOverwriteMode = !( _IsOverwriteMode );
				}

				lresult = IntPtr.Zero;
				return 0;
			}

			lresult = IntPtr.Zero;
			return 0;
		}
		#endregion

		#region Others
		/// <summary>
		/// コマンドキーを処理。
		/// コマンドキーらしいキーは一切処理しないが、
		/// Ctrl キーが押されたら無条件に 95Reader を黙らせる。
		/// </summary>
		protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
		{
			if( keyData == (Keys.Control|Keys.ControlKey) )
			{
				if( XpReaderSpeaker.IsAlive() )
					XpReaderSpeaker.Instance.Stop();
			}

			return base.ProcessCmdKey( ref msg, keyData );
		}
		#endregion

		#region Utilities
		internal class Utl
		{
			public const int GWL_STYLE = -16;
			const int ES_MULTILINE        = 0x0004;
			const int ES_AUTOVSCROLL      = 0x0040;
			const int ES_AUTOHSCROLL      = 0x0080;
			const int ES_NOHIDESEL        = 0x0100;
			const int ES_WANTRETURN       = 0x1000;
			const int WS_TABSTOP          = 0x00010000;
			const int WS_HSCROLL          = 0x00100000;
			const int WS_VSCROLL          = 0x00200000;
			const int WS_VISIBLE          = 0x10000000;
			public delegate IntPtr WindowProc( IntPtr hWnd, Int32 message, IntPtr wParam, IntPtr lParam );

			public static IntPtr CreateRichEdit( IntPtr parent, bool doWordWrap )
			{
				UInt32 style = WS_VISIBLE | WS_TABSTOP | WS_VSCROLL
					| ES_MULTILINE | ES_NOHIDESEL/* | ES_SUNKEN*/
					| ES_WANTRETURN | ES_AUTOVSCROLL;

				if( !doWordWrap )
				{
					style |= WS_HSCROLL | ES_AUTOHSCROLL;
				}

				if( Marshal.SizeOf(IntPtr.Zero) == 8 )
					return CreateSgRichEdit64( parent, style );
				else
					return CreateSgRichEdit32( parent, style );
			}

			public static string ToString( EolCode eolCode )
			{
				switch( eolCode )
				{
					case EolCode.CR:
						return "\r";

					case EolCode.LF:
						return "\n";

					default:
						return "\r\n";
				}
			}

			struct NMHDR
			{
				public IntPtr window;
				public UInt32 id;
				public UInt32 code;
			}

			// NMHDR の使われないメンバに対する警告を消すためのダミー
			static void __DummyFunction__()
			{
				NMHDR dummy = new NMHDR();
				dummy.window = IntPtr.Zero;
				dummy.id = 0;
				dummy.code = 0;
			}

			public static Int16 HiWord( Int32 l )
			{
				return (Int16)(l >> 16);
			}

			public static bool IsSelectionChangedEvent( ref Message msg )
			{
				const int EN_SELCHANGE = 0x0702;

				unsafe
				{
					NMHDR * nmhdr = (NMHDR*)msg.LParam.ToPointer();
					if( nmhdr->code == EN_SELCHANGE )
					{
						return true;
					}

					return false;
				}
			}

			[DllImport("user32", EntryPoint="SendMessageW")]
			public static extern IntPtr SendMessage( IntPtr hWnd, UInt32 msg, WmHookProc wParam, IntPtr lParam );

			[DllImport("SgRichEdit32.dll", EntryPoint="CreateSgRichEdit")]
			public static extern IntPtr CreateSgRichEdit32( IntPtr parent, UInt32 windowStyle );
			
			[DllImport("SgRichEdit64.dll", EntryPoint="CreateSgRichEdit")]
			public static extern IntPtr CreateSgRichEdit64( IntPtr parent, UInt32 windowStyle );
			
			[DllImport("user32", EntryPoint="SendMessageW")]
			public static extern IntPtr SendMessage( IntPtr hWnd, int msg, IntPtr wParam, IntPtr lParam );

			[DllImport("user32", EntryPoint="SendMessageW")]
			public static extern unsafe IntPtr SendMessage( IntPtr hWnd, int msg, void* wParam, void* lParam );

			[DllImport("user32", EntryPoint="SendMessageW")]
			public static extern IntPtr SendMessage( IntPtr hWnd, int msg, string wParam, IntPtr lParam );

			[DllImport("user32")]
			public static extern Int32 DestroyWindow( IntPtr window );

			[DllImport("user32")]
			public static extern IntPtr GetWindowLongW( IntPtr hWnd, Int32 index );

			[DllImport("user32")]
			public static extern IntPtr SetWindowLongW( IntPtr hWnd, Int32 index, WindowProc newLong );
			
			[DllImport("user32")]
			public static extern IntPtr SetWindowLongW( IntPtr hWnd, Int32 index, IntPtr newLong );
			
			[DllImport("user32")]
			public static extern Int32 SetWindowPos( IntPtr window, IntPtr insertAfter, Int32 x, Int32 y, Int32 width, Int32 height, Int32 flags );
		}
		#endregion
	}
}
