﻿// file: NumTextBox.cs
// brief: specialized text box for integer input.
// encoding: UTF-8
// update: 2010-01-31
//=========================================================
using System;
using System.Windows.Forms;

namespace Sgry.AiBTools.Gui
{
	/// <summary>
	/// 数値入力専用のテキストボックス。
	/// </summary>
	public class NumTextBox : TextBox
	{
		int _Min = Int32.MinValue;
		int _Default = 0;
		int _Max = Int32.MaxValue;

		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public NumTextBox()
		{}

		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public NumTextBox( int minValue, int defaultValue, int maxValue )
		{
			if( maxValue < minValue )
			{
				throw new ArgumentException();
			}

			_Min = minValue;
			_Default = defaultValue;
			_Max = maxValue;
		}

		/// <summary>
		/// 入力可能な最大の数値を設定または取得します。
		/// </summary>
		public int Max
		{
			get{ return _Max; }
			set
			{
				if( _Min < value )
					_Max = value;
			}
		}

		/// <summary>
		/// 入力可能な最大の数値を設定または取得します。
		/// </summary>
		public int Min
		{
			get{ return _Min; }
			set
			{
				if( value < _Max )
					_Min = value;
			}
		}

		/// <summary>
		/// 入力が省略されたときに使う標準の値を取得または設定します。
		/// </summary>
		public int Default
		{
			get{ return _Default; }
			set
			{
				if( value < _Min || _Max < value )
					throw new InvalidOperationException( "Default value must not be out of range specified by Min and Max properties." );

				_Default = value;
			}
		}

		/// <summary>
		/// フォーカスを得たときの動作です。
		/// </summary>
		protected override void OnEnter( EventArgs e )
		{
			base.OnEnter( e );

			// フォーカスを得たら全選択
			SelectAll();
		}

		/// <summary>
		/// フォーカスを失ったときの動作です。
		/// </summary>
		protected override void OnLeave( EventArgs e )
		{
			base.OnLeave( e );

			int num;
			bool isNum;

			// フォーカスを失うときに内容が空であれば標準値を適用
			if( Text.Length <= 0 )
			{
				Text = Default.ToString();
			}

			// 値の範囲を確認
			isNum = Int32.TryParse( this.Text, out num );
			if( isNum == false )
			{
				// 値が数値でないので標準値を適用
				this.Text = Default.ToString();
			}
			else if( num < Min )
			{
				// 値が下境界未満なので下境界に修正
				this.Text = Min.ToString();
			}
			else if( Max < num )
			{
				// 値が上境界を超えているので上境界に修正
				this.Text = Max.ToString();
			}
		}

		/// <summary>
		/// ダイアログ上でのキー入力を、このコントロールで消費するかどうかを判定します。
		/// </summary>
		protected override bool IsInputChar( char charCode )
		{
			return ('0' <= charCode && charCode <= '9');
		}

		/// <summary>
		/// 文字入力が発生したときの動作です。
		/// </summary>
		protected override void OnKeyPress( KeyPressEventArgs e )
		{
			base.OnKeyPress( e );
			if( e.Handled )
			{
				return;
			}

			// 数字以外を入力されたら通知音を出して受け付けない
			if( Char.IsLetter(e.KeyChar) )
			{
				Win32.MessageBeep_Notify();
				e.Handled = true;
			}

			// 全角数字が入力された場合は半角に変換してしまう
			if( '０' <= e.KeyChar && e.KeyChar <= '９' )
			{
				e.KeyChar = (char)( '0' + e.KeyChar - '０' );
			}
		}

		/// <summary>
		/// キーが押し下げられたときの動作です。
		/// </summary>
		protected override void OnKeyDown( KeyEventArgs e )
		{
			base.OnKeyDown( e );
			if( e.Handled )
			{
				return;
			}

			int num;
			int diff;
			bool isNum;

			// キーに応じて変更する値の差分を計算
			if( e.KeyCode == Keys.Up )
			{
				if( e.Shift )
					diff = 10;
				else
					diff = 1;
			}
			else if( e.KeyCode == Keys.Down )
			{
				if( e.Shift )
					diff = -10;
				else
					diff = -1;
			}
			else
			{
				// 無関係なキーが押されたので標準動作で良い
				return;
			}

			try
			{
				// 現在の値を取得
				isNum = Int32.TryParse( this.Text, out num );
				if( isNum == false )
				{
					// 数値でないのはおかしいので、標準値にしてしまう
					this.Text = Default.ToString();
					return;
				}

				// すでに境界値に達していた場合は通知して終了
				if( (num == Min && diff < 0)
					|| (num == Max && 0 < diff) )
				{
					Win32.MessageBeep_Notify();
					return;
				}

				// 値を許容範囲内に納まらなければ修正する
				num += diff;
				if( num < Min )
				{
					num = Min;
				}
				else if( Max < num )
				{
					num = Max;
				}

				// 値を更新して全選択
				this.Text = num.ToString();
			}
			finally
			{
				this.SelectAll();
				e.Handled = true;
			}
		}
	}
}
