// file: MyExplorer.cs
// brief: File explorer control
// encoding: UTF-8
// update: 2010-09-02
//=========================================================
// TODO_1: 一度取得したファイルタイプ名などをStringDictionaryなどにキャッシュ
using System;
using System.Collections;
using System.Collections.Specialized;
using System.IO;
using System.Windows.Forms;
using Microsoft.Win32;
using Sgry.AiBTools;
using CultureInfo = System.Globalization.CultureInfo;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.Gui
{
	using AT;

	/// <summary>
	/// ファイル一覧ビューです。
	/// </summary>
	public class MyExplorer : AiBListView
	{
		/// <summary>
		/// マイコンピュータに相当する仮想ディレクトリの名前。
		/// </summary>
		public const string MyComputerTagName = "Drive list";

		#region Fields
		string _CurrentDirectory = Environment.CurrentDirectory;
		string _PrevChildPath = null;
		string _Filter = "*.*";

		/// <summary>
		/// 表示中のディレクトリのパス
		/// </summary>
		public string CurrentDirectory
		{
			get{ return _CurrentDirectory; }
			set
			{
				_CurrentDirectory = value;
				InvokeCurrentDirectoryChanged();
			}
		}

		/// <summary>
		/// セミコロン区切りで指定する、一覧に表示するファイルを限定するフィルター
		/// </summary>
		public string Filter
		{
			get{ return _Filter; }
			set
			{
				_Filter = value;
			}
		}
		#endregion

		#region Init / Dispose
		/// <summary>
		/// インスタンスを生成します。
		/// </summary>
		public MyExplorer()
		{
			InitializeComponent();
			LocalizeComponent();
		}

		/// <summary>
		/// ハンドルが生成された直後の動作。
		/// </summary>
		protected override void OnHandleCreated( EventArgs e )
		{
			base.OnHandleCreated( e );
			UpdateFileList();
		}
		#endregion

		#region Behavior as a Control
		/// <summary>
		/// 指定されたキーが入力として使うキーかどうかを判定します。
		/// </summary>
		protected override bool IsInputKey( Keys keyData )
		{
			if( keyData == Keys.Enter )
			{
				return true;
			}
			return base.IsInputKey( keyData );
		}

		/// <summary>
		/// 入力キーをコマンドとして処理します。
		/// </summary>
		protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
		{
			Keys key = (Keys)msg.WParam.ToInt32();
			if( key == Keys.Back )
			{
				GoUp();
				return true;
			}

			return base.ProcessCmdKey( ref msg, keyData );
		}
		#endregion

		#region イベント
		/// <summary>
		/// SelectedEntryChangedEvent のイベントハンドラ。
		/// </summary>
		/// <param name="path">新しく選択されたファイルエントリのパス。</param>
		public delegate void SelectedEntryChangedEventHandler( string path );

		/// <summary>
		/// 選択されている項目が変化した時に発生します。
		/// </summary>
		public event SelectedEntryChangedEventHandler SelectedEntryChanged;
		void InvokeSelectedEntryChanged()
		{
			if( SelectedEntryChanged != null )
			{
				if( SelectedIndices.Count == 0 )
				{
					return;
				}

				if( SelectedItems[0].Text == @"..\" )
				{
					SelectedEntryChanged( String.Empty );
				}
				else
				{
					SelectedEntryChanged( (string)SelectedItems[0].Tag );
				}
			}
		}

		/// <summary>
		/// TargetChoosedEvent のイベントハンドラ。
		/// </summary>
		/// <param name="path">ユーザが最終的な操作対象を選択した直後の動作。</param>
		public delegate void TargetChoosedEventHandler( string path );

		/// <summary>
		/// ユーザが対象ファイルを決定した直後に発生します。
		/// </summary>
		public event TargetChoosedEventHandler Choosed;
		void InvokeChoosed()
		{
			if( Choosed != null )
			{
				if( SelectedIndices.Count == 0 )
				{
					return;
				}

				Choosed( (string)SelectedItems[0].Tag );
			}
		}

		/// <summary>
		/// 表示中ディレクトリが変化した時に発生します。
		/// </summary>
		public event EventHandler CurrentDirectoryChanged;
		void InvokeCurrentDirectoryChanged()
		{
			if( CurrentDirectoryChanged != null )
			{
				CurrentDirectoryChanged( this, EventArgs.Empty );
			}
		}
		#endregion

		#region その他
		/// <summary>
		/// 選択が変更された直後の動作。
		/// </summary>
		protected override void OnSelectedIndexChanged( EventArgs e )
		{
			base.OnSelectedIndexChanged( e );

			// 選択エントリーの変更イベントをそのまま発生させる
			InvokeSelectedEntryChanged();
		}

		/// <summary>
		/// ファイル一覧で項目が選択された時の動作。
		/// </summary>
		void ListView_ItemActivate( object sender, EventArgs e )
		{
			ListViewItem item = FocusedItem;
			if( item == null || item.Tag == null )
			{
				return;
			}

			string path = (string)item.Tag;

			// ディレクトリなら、その中に入る
			if( path == MyComputerTagName
				|| Directory.Exists(path) )
			{
				GoDown( path );
			}
			// ファイルなら、ファイルが選択されたイベントを発生
			else if( File.Exists(path) )
			{
				InvokeChoosed();
			}
		}
		#endregion

		#region ファイル一覧の動作
		/// <summary>
		/// 指定ディレクトリに移動
		/// </summary>
		/// <param name="path">移動先ディレクトリのパス</param>
		void GoDown( string path )
		{
			NbsEngine.Instance.BeginUpdate();

			_PrevChildPath = null;
			_CurrentDirectory = path;
			UpdateFileList();
			InvokeCurrentDirectoryChanged();

			// 選択変更のイベントを発生
			OnSelectedIndexChanged( EventArgs.Empty );

			NbsEngine.Instance.EndUpdate();
		}

		/// <summary>
		/// 現在のディレクトリの親ディレクトリに移動
		/// </summary>
		void GoUp()
		{
			NbsEngine.Instance.BeginUpdate();

			// 現在位置に応じて次のディレクトリを決定
			if( CurrentDirectory == MyComputerTagName )
			{
				// マイコンピュータより上は無い
				Win32.MessageBeep_Notify();
			}
			else
			{
				// 最初の項目は常に親ディレクトリを指しているのでこれを利用している
				_PrevChildPath = CurrentDirectory;
				CurrentDirectory = (string)Items[0].Tag;
				UpdateFileList();

				// 選択変更のイベントを発生
				OnSelectedIndexChanged( EventArgs.Empty );
			}

			NbsEngine.Instance.EndUpdate();
		}
		#endregion // ファイル一覧の動作

		#region ファイル一覧の表示更新ロジック
		/// <summary>
		/// ファイル一覧の表示を更新
		/// </summary>
		public void UpdateFileList()
		{
			if( DesignMode || Visible == false )
			{
				return;
			}

			Items.Clear();

			if( CurrentDirectory == MyComputerTagName )
			{
				UpdateFileList_MyComputer();
			}
			// ドライブのルートを表示中か？
			else if( Path.GetPathRoot(CurrentDirectory) == CurrentDirectory )
			{
				UpdateFileList_DriveRoot();
			}
			else
			{
				UpdateFileList_Normal();
			}

			// 選択変更イベントを手動で送信
			OnSelectedIndexChanged( EventArgs.Empty );
		}

		/// <summary>
		/// マイコンピュータの内容を表示
		/// </summary>
		// 制御の流れが分かりにくい
		void UpdateFileList_MyComputer()
		{
			ListViewItem item, firstSelectedItem;
			
			// ドライブを追加
			foreach( string drive in Directory.GetLogicalDrives() )
			{
				item = new ListViewItem();
				item.Text = drive;
				item.Tag = drive;
				Items.Add( item );
			}

			// 前回表示されていたディレクトリ（ドライブルート）を含むドライブを選択
			if( _PrevChildPath != null )
			{
				for( int i=0; i<Items.Count; i++ )
				{
					if( _PrevChildPath == (string)Items[i].Tag )
					{
						firstSelectedItem = Items[i];
						goto selection;
					}
				}

				// 前回表示していたディレクトリが見つからない。
				// 恐らくフルパス指定でのジャンプ後なので、問題とは見なさない。
			}

			return;

		selection:
			// 初期選択項目を選択
			firstSelectedItem.Selected = true;
			firstSelectedItem.Focused = true;
			firstSelectedItem.EnsureVisible();
		}

		/// <summary>
		/// ドライブルートの内容を表示
		/// </summary>
		void UpdateFileList_DriveRoot()
		{
			ListViewItem item = new ListViewItem();
			item.Text = @"..\";
			item.Tag = MyComputerTagName;
			Items.Add( item );

			AddFileEntriesInDirectory( CurrentDirectory );
		}

		/// <summary>
		/// 通常のディレクトリの内容を表示
		/// </summary>
		void UpdateFileList_Normal()
		{
			ListViewItem item = new ListViewItem();
			item.Text = @"..\";
			item.Tag = Directory.GetParent( CurrentDirectory ).FullName;
			Items.Add( item );

			AddFileEntriesInDirectory( CurrentDirectory );
		}

		void AddFileEntriesInDirectory( string directory )
		{
			ArrayList			items = new ArrayList();
			ListViewItem		item, firstSelectedItem;
			string[]			dirEntries;
			StringCollection	fileEntries = new StringCollection();

			// ディレクトリを検索
			dirEntries = Directory.GetDirectories( directory );
			foreach( string dir in dirEntries )
			{
				item = new ListViewItem();
				item.Text = Path.GetFileName( dir ) + "\\";
				item.SubItems.Add( "" );
				item.SubItems.Add( Utl.GetDirectoryFileTypeName() );
				item.SubItems.Add( Directory.GetCreationTime(dir).ToString() );
				item.Tag = dir;
				items.Add( item );
			}

			// ファイルを検索
			foreach( string filter in _Filter.Split(';') )
			{
				string[] files = Directory.GetFiles( directory, filter );
				if( files != null )
					fileEntries.AddRange( files );
			}
			foreach( string file in fileEntries )
			{
				FileInfo fileInfo = new FileInfo( file );
				item = new ListViewItem();
				item.Text = fileInfo.Name;
				item.SubItems.Add( fileInfo.Length.ToString() );
				item.SubItems.Add( Utl.GetFileTypeNameByExt(fileInfo.Extension) );
				item.SubItems.Add( fileInfo.LastWriteTime.ToString() );
				item.Tag = file;
				items.Add( item );
			}

			// リストビューに項目を追加
			this.Items.AddRange( (ListViewItem[])items.ToArray(typeof(ListViewItem)) );

			// どの項目を選択しておくか決定する。
			// 前回表示していたディレクトリが一覧にあるようならば、それを選択。
			// 無いならば、最適なものを選択
			if( _PrevChildPath != null )
			{
				for( int i=0; i<Items.Count; i++ )
				{
					if( _PrevChildPath == (string)Items[i].Tag )
					{
						firstSelectedItem = Items[i];
						goto selection;
					}
				}

				// 前回表示していたディレクトリが見つからない。プログラムミス。
				Debug.Fail( "[MyExplorer] directory fullPath synchronization failed." );
				return;
			}
			else
			{
				if( fileEntries.Count == 0 )
				{
					// ファイルが一つも無い。
					if( dirEntries.Length == 0 )
					{
						// ディレクトリも無い。".."を選択。
						firstSelectedItem = Items[0];
					}
					else
					{
						// ディレクトリはある。最初のディレクトリを選択
						firstSelectedItem = Items[1];
					}
				}
				else
				{
					// 通常動作。最初のファイルを選択
					firstSelectedItem = Items[ dirEntries.Length + 1 ];
				}
			}

		selection:
			// 初期選択項目を選択
			firstSelectedItem.Selected = true;
			firstSelectedItem.Focused = true;
			firstSelectedItem.EnsureVisible();
		}
		#endregion

		#region UI Component Initialization
		void LocalizeComponent()
		{
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				_NameColumnHeader.Text = "名前";
				_SizeColumnHeader.Text = "サイズ";
				_TypeColumnHeader.Text = "種類";
				_DateColumnHeader.Text = "更新日時";
			}
		}
		void InitializeComponent()
		{
			_NameColumnHeader = new ColumnHeader();
			_SizeColumnHeader = new ColumnHeader();
			_TypeColumnHeader = new ColumnHeader();
			_DateColumnHeader = new ColumnHeader();
			// 
			// _NameColumnHeader
			// 
			this._NameColumnHeader.Text = "Name";
			this._NameColumnHeader.Width = 200;
			// 
			// _SizeColumnHeader
			// 
			this._SizeColumnHeader.Text = "Size";
			this._SizeColumnHeader.Width = 100;
			// 
			// _TypeColumnHeader
			// 
			this._TypeColumnHeader.Text = "Type";
			this._TypeColumnHeader.Width = 120;
			// 
			// _DateColumnHeader
			// 
			this._DateColumnHeader.Text = "Date";
			this._DateColumnHeader.Width = 120;
			//
			// MyExplorer
			//
			this.View = View.Details;
			this.Columns.AddRange(
					new ColumnHeader[] {_NameColumnHeader, _SizeColumnHeader, _TypeColumnHeader, _DateColumnHeader}
				);
			this.FullRowSelect = true;
			this.GridLines = true;
			this.HideSelection = false;
			this.Location = new System.Drawing.Point( 14, 33 );
			this.Name = "_FileListView";
			this.Size = new System.Drawing.Size( 526, 160 );
			this.TabIndex = 3;
			this.View = System.Windows.Forms.View.Details;
			this.ItemActivate += new EventHandler( ListView_ItemActivate );
			this.Width = 300;
			this.Height = 300;
		}
		#endregion

		#region UI Components
		ColumnHeader	_NameColumnHeader;
		ColumnHeader	_SizeColumnHeader;
		ColumnHeader	_TypeColumnHeader;
		ColumnHeader	_DateColumnHeader;
		#endregion

		class Utl
		{
			public static string GetDirectoryFileTypeName()
			{
				try
				{
					RegistryKey key = Registry.ClassesRoot.OpenSubKey( "Directory" );
					return (string)key.GetValue( null );
				}
				catch
				{
					return "";
				}
			}
			public static string GetFileTypeNameByExt( string ext )
			{
				try
				{
					RegistryKey extKey, typeKey;
					string typeKeyName;

					extKey = Registry.ClassesRoot.OpenSubKey( ext );
					typeKeyName = (string)extKey.GetValue( null );

					typeKey = Registry.ClassesRoot.OpenSubKey( typeKeyName );
					return (string)typeKey.GetValue( null );
				}
				catch
				{
					return "";
				}
			}
		}
	}
}
