// file: AzukiEdit.cs
// brief: a wrapping class for Azuki
// update: 2010-09-04
//=========================================================
using System;
using System.Text;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using Debug = System.Diagnostics.Debug;
using Sgry.Azuki;
using Sgry.Azuki.Windows;
using Highlighters = Sgry.Azuki.Highlighter.Highlighters;

namespace Sgry.AiBTools.Gui
{
	/// <summary>
	/// テキストエディタエンジン Azuki を
	/// AiB Tools 用にラッピングしたコントロール。
	/// </summary>
	public class AzukiEdit : AzukiControl, ITextEditor
	{
		bool _UseSmartHome;

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成
		/// </summary>
		public AzukiEdit()
		{
			UseSmartHome = false; // needed
			IsAccessible = false;
			base.ConvertsFullWidthSpaceToSpace = false;
			base.ConvertsTabToSpaces = false;
			base.HighlightsCurrentLine = true;
			base.ShowsDirtBar = false;

			// 独自イベントにマッピング
			Document.ContentChanged += delegate {
				InvokeContentChanged();
			};

			Document.SelectionChanged += delegate {
				InvokeSelectionChanged();
			};

			Document.DirtyStateChanged += delegate {
				InvokeDirtyStateChanged();
			};
		}
		#endregion

		#region Modes
		/// <summary>
		/// 編集内容が読み取り専用かどうか
		/// </summary>
		[DefaultValue(true)]
		public bool ReadOnly
		{
			get{ return Document.IsReadOnly; }
			set{ Document.IsReadOnly = value; }
		}

		/// <summary>
		/// 未編集状態かどうかを取得または設定します。
		/// </summary>
		[Browsable(false)]
		public bool IsDirty
		{
			get{ return Document.IsDirty; }
			set{ Document.IsDirty = value; }
		}

		/// <summary>
		/// Home キーで行頭ではなく最初の非空白文字にキャレットを移動するかどうか
		/// </summary>
		[DefaultValue(false)]
		public bool UseSmartHome
		{
			get{ return _UseSmartHome; }
			set
			{
				_UseSmartHome = value;
				if( _UseSmartHome )
				{
					SetKeyBind( Keys.Home, Actions.MoveToLineHeadSmart );
					SetKeyBind( Keys.Home|Keys.Shift, Actions.SelectToLineHeadSmart );
				}
				else
				{
					SetKeyBind( Keys.Home, Actions.MoveToLineHead );
					SetKeyBind( Keys.Home|Keys.Shift, Actions.SelectToLineHead );
				}
			}
		}

		/// <summary>
		/// 指定ファイル種別用のモードに切り替える。
		/// 非対応の種別名を指定すると標準モードになる。
		/// </summary>
		public void SetFileType( string typeName, AutoIndentMode indentMode )
		{
			string typeName_l = typeName.ToLower();
			int indentLevel = (int)indentMode;

			if( typeName_l == "c/c++" )
			{
				Highlighter = Highlighters.Cpp;
				if     ( indentLevel == 2 ) AutoIndentHook = AutoIndentHooks.CHook;
				else if( indentLevel == 1 ) AutoIndentHook = AutoIndentHooks.GenericHook;
				else                        AutoIndentHook = null;
			}
			else if( typeName_l == "java" )
			{
				Highlighter = Highlighters.Java;
				if     ( indentLevel == 2 ) AutoIndentHook = AutoIndentHooks.CHook;
				else if( indentLevel == 1 ) AutoIndentHook = AutoIndentHooks.GenericHook;
				else                        AutoIndentHook = null;
			}
			else if( typeName_l == "c#" )
			{
				Highlighter = Highlighters.CSharp;
				if     ( indentLevel == 2 ) AutoIndentHook = AutoIndentHooks.CHook;
				else if( indentLevel == 1 ) AutoIndentHook = AutoIndentHooks.GenericHook;
				else                        AutoIndentHook = null;
			}
			else if( typeName_l == "ruby" )
			{
				Highlighter = Highlighters.Ruby;
				if( 0 < indentLevel ) AutoIndentHook = AutoIndentHooks.GenericHook;
				else                  AutoIndentHook = null;
			}
			else if( typeName_l == "xml" )
			{
				Highlighter = Highlighters.Xml;
				if( 0 < indentLevel ) AutoIndentHook = AutoIndentHooks.GenericHook;
				else                  AutoIndentHook = null;
			}
			else
			{
				Highlighter = null;
				AutoIndentHook = null;
			}
		}
		#endregion

		#region Selection
		/// <summary>
		/// 選択の終了インデックス
		/// </summary>
		[Browsable(false)]
		public int SelectionEnd
		{
			get
			{
				int begin, end;
				Document.GetSelection( out begin, out end );
				return end;
			}
		}

		/// <summary>
		/// 選択が後方へと行われているかどうか
		/// </summary>
		[Browsable(false)]
		public bool SelectingToEnd
		{
			get
			{
				return (Document.AnchorIndex <= Document.CaretIndex);
			}
		}

		/// <summary>
		/// 選択の起点のインデックスを取得
		/// </summary>
		[Browsable(false)]
		public int SelectionAnchor
		{
			get{ return Document.AnchorIndex; }
		}

		/// <summary>
		/// 現在選択されているテキスト
		/// </summary>
		public string SelectedText
		{
			get
			{
				int begin, end;
				Document.GetSelection( out begin, out end );
				return GetTextInRange( begin, end );
			}
			set
			{
				Document.Replace( value );
			}
		}

		/// <summary>
		/// 選択されている文字数
		/// </summary>
		[Browsable(false)]
		public int SelectionLength
		{
			get
			{
				int begin, end;
				Document.GetSelection( out begin, out end );
				return end - begin;
			}
		}
		#endregion

		#region Caret
		/// <summary>
		/// キャレット位置のインデックスを取得
		/// </summary>
		public int GetCaretIndex()
		{
			return base.CaretIndex;
		}

		/// <summary>
		/// キャレット位置を取得
		/// </summary>
		/// <param name="lineIndex">キャレットの行インデックス</param>
		/// <param name="columnIndex">キャレットの桁インデックス</param>
		public void GetCaretIndex( out int lineIndex, out int columnIndex )
		{
			Document.GetCaretIndex( out lineIndex, out columnIndex );
		}

		/// <summary>
		/// キャレット位置を設定します。
		/// </summary>
		/// <param name="lineIndex">キャレット移動先の行インデックス</param>
		/// <param name="columnIndex">キャレット移動先の桁インデックス</param>
		public void SetCaretIndex( int lineIndex, int columnIndex )
		{
			int charIndex;
			
			// 行インデックスが大きすぎる場合は最大値にリセット
			if( Document.LineCount <= lineIndex )
			{
				lineIndex = Document.LineCount - 1;
			}

			// 桁インデックスが大きすぎる場合は最大値にリセット
			columnIndex = Math.Min( columnIndex, GetLineLength(lineIndex) );

			// 行・桁インデックスを文字インデックスに変換
			charIndex = base.GetCharIndexFromLineColumnIndex( lineIndex, columnIndex );
			if( TextLength < charIndex )
			{
				charIndex = Document.Length - 1; // 桁インデックスが大きすぎるので末尾までに制限
			}

			// そこへ移動
			SetSelection( charIndex, charIndex );
		}
		#endregion

		#region Text
		/// <summary>
		/// 現在入力されているテキスト
		/// </summary>
		public override string Text
		{
			get{ return base.Text; }
			set{ base.Text = value; }
		}

		/// <summary>
		/// 指定位置の文字を取得
		/// </summary>
		public char GetCharAt( int index )
		{
			return Document.GetCharAt( index );
		}

		/// <summary>
		/// 現在入力されているテキストを指定した改行コードで取得
		/// </summary>
		public string GetText( EolCode eolCode )
		{
			StringBuilder text = new StringBuilder( Document.Length );
			string eolStr = Utl.ToString( eolCode );
			int lineCount = Document.LineCount;

			// 最後の一行を除いて指定された改行コードで保存
			for( int i=0; i<lineCount-1; i++ )
			{
				text.Append( Document.GetLineContent(i) );
				text.Append( eolStr );
			}

			// 最後の一行（普通は空文字）をそのまま出力
			text.Append( Document.GetLineContent(lineCount-1) );

			return text.ToString();
		}

		/// <summary>
		/// Replace currently selected text into the specified text.
		/// If nothing selected, specified text will be inserted at caret pos.
		/// </summary>
		public void Replace( string text )
		{
			Document.Replace( text );
		}

		/// <summary>
		/// テキストをすべて削除
		/// </summary>
		public void Clear()
		{
			Document.Replace( String.Empty, 0, Document.Length );
		}
		#endregion

		#region GUI
/*		/// <summary>
		/// 入力フォーカスがあるかどうか
		/// </summary>
		public override bool Focused
		{
			get{ return base.Focused; }
		}*/
		#endregion

		#region Event
		/// <summary>
		/// 使用しないで下さい。
		/// </summary>
		public new event EventHandler TextChanged;
		void InvokeTextChanged()
		{
			if( TextChanged != null ) TextChanged( this, EventArgs.Empty );
		}

		/// <summary>
		/// 内容が変化した時に発生します。
		/// </summary>
		public event EventHandler ContentChanged;
		void InvokeContentChanged()
		{
			if( ContentChanged != null )
				ContentChanged( this, EventArgs.Empty );
		}

		/// <summary>
		/// 選択状態が変化した時に発生します。
		/// 無選択状態でキャレットが移動した時にも発生します。
		/// </summary>
		public event EventHandler SelectionChanged;
		void InvokeSelectionChanged()
		{
			if( SelectionChanged != null )
				SelectionChanged( this, EventArgs.Empty );
		}

		/// <summary>
		/// Occurs when a first modification occured in the document.
		/// </summary>
		public event EventHandler DirtyStateChanged;
		void InvokeDirtyStateChanged()
		{
			if( DirtyStateChanged != null )
				DirtyStateChanged( this, EventArgs.Empty );
		}
		#endregion

		#region Edit action
		/// <summary>
		/// UNDO バッファをクリア
		/// </summary>
		public void ClearUndo()
		{
			Document.ClearHistory();
		}
		#endregion

		#region Line level actions
		/// <summary>
		/// 読み込んでいるファイルの行数を取得
		/// </summary>
		/// <returns>ファイルの行数</returns>
		public int GetLineCount()
		{
			return Document.LineCount;
		}

		/// <summary>
		/// 指定行の長さ（文字数）を取得
		/// </summary>
		/// <param name="lineIndex">長さを取得したい行のインデックス</param>
		/// <returns>行の長さ（文字数）</returns>
		/// <exception cref="ArgumentOutOfRangeException">インデックスが不正</exception>
		public new int GetLineLength( int lineIndex )
		{
			return Document.GetLineLength( lineIndex );
		}

		/// <summary>
		/// 指定行を取得
		/// </summary>
		/// <param name="lineIndex">取得したい行のインデックス</param>
		/// <returns>指定行の内容</returns>
		/// <exception cref="ArgumentOutOfRangeException">インデックスが不正</exception>
		public string GetLineContent( int lineIndex )
		{
			return Document.GetLineContent( lineIndex );
		}

		/// <summary>
		/// 指定したインデックスが指す文字を含む行の番号を取得
		/// </summary>
		/// <param name="index">このインデックスが指す文字を含む行の番号を取得します。</param>
		/// <returns>行番号</returns>
		public new int GetLineIndexFromCharIndex( int index )
		{
			int line, column;
			
			base.GetLineColumnIndexFromCharIndex( index, out line, out column );
			
			return line;
		}
		#endregion

		#region Word level actions
		#endregion

		#region Others
		/// <summary>
		/// コマンドキーを処理。
		/// コマンドキーらしいキーは一切処理しないが、
		/// Ctrl キーが押されたら無条件に 95Reader と FocusTalk を黙らせる。
		/// </summary>
		protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
		{
			if( keyData == (Keys.Control|Keys.ControlKey) )
			{
				if( XpReaderSpeaker.IsAlive() )
					XpReaderSpeaker.Instance.Stop();
				else if( FocusTalkSpeaker.IsAlive() )
					FocusTalkSpeaker.Instance.Stop();
			}

			return base.ProcessCmdKey( ref msg, keyData );
		}
		#endregion

		#region Utilities
		class Utl
		{
			public static string ToString( EolCode eolCode )
			{
				switch( eolCode )
				{
					case EolCode.CR:
						return "\r";

					case EolCode.LF:
						return "\n";

					default:
						return "\r\n";
				}
			}
		}
		#endregion
	}
}
