// file: AiBFileDialog.cs
// brief: Base class for file dialog that can use NbsEngine
// encoding: UTF-8
// update: 2010-09-02
//=========================================================
using System;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using Debug = System.Diagnostics.Debug;
using Point = System.Drawing.Point;
using Size = System.Drawing.Size;
using FileTypeComboBoxItem = Sgry.AiBTools.Gui.MyComboBoxItem<string>;
using CultureInfo = System.Globalization.CultureInfo;

namespace Sgry.AiBTools.Gui
{
	using AT;

	/// <summary>
	/// ファイル選択ダイアログの基底クラス
	/// </summary>
	public abstract class AiBFileDialog : AiBDialogBase
	{
		#region Fields
		readonly static char[] WildcardChars = new char[]{ '?', '*' };
		string		_DefaultFilter = "All files (*.*)|*.*";
		string		_FileName = null;
		string		_InitFileName = null;
		Localizer	_Localizer	= new Localizer();

		string _Msg_WentTo = "Went to {0}.";
		string _Msg_InvalidFileName = "Input file name is invalid. Please input again.";
		#endregion

		#region Init / Dispose
		/// <summary>
		/// インスタンスを初期化
		/// </summary>
		public AiBFileDialog()
		{
			InitializeComponents();
			LocalizeComponent();
			AutoScaleMode = AutoScaleMode.Font;
			Font = SystemInformation.MenuFont;

			// 初期状態を設定
			_FileNameTextBox.Select(); // 最初はファイル名のテキストボックスをアクティブに
		}

		/// <summary>
		/// GUI の初期化が終わった直後の動作。
		/// </summary>
		protected override void OnLoad( EventArgs e )
		{
			base.OnLoad( e );

			// set initial value
			_FileNameTextBox.Text = _InitFileName;
			_FileNameTextBox.SelectAll();
			_DirectoryPathTextBox.Text = _FileListView.CurrentDirectory;

			// ファイルリストの列幅を必要に応じて広げる
			// （AutoScale はリストビューの列幅には影響しないため）
			int[] orgWidthList = new int[4];
			for( int i=0; i<4; i++ )
			{
				orgWidthList[i] = _FileListView.Columns[i].Width;
			}
			_FileListView.AutoResizeColumns( ColumnHeaderAutoResizeStyle.ColumnContent );
			for( int i=0; i<4; i++ )
			{
				if( _FileListView.Columns[i].Width < orgWidthList[i] )
				{
					_FileListView.Columns[i].Width = orgWidthList[i];
				}
			}
		}

		/// <summary>
		/// GUI が最初に表示されたときの動作。
		/// </summary>
		protected override void OnShown( EventArgs e )
		{
			base.OnShown( e );

			// 点字を表示
			_TextBoxBrailler.UpdateDisplay();
		}
		#endregion

		#region Dialog Customization
		/// <summary>
		/// 入力されたファイル名を確定する直前の検証を行います。
		/// </summary>
		protected abstract bool ValidateFileName( string path );

		/// <summary>
		/// カスタムGUIパネルを設定します。
		/// </summary>
		protected abstract void SetupCustomPanel();
		#endregion

		#region Properties
		/// <summary>
		/// ダイアログが表示された時に表示しているディレクトリ
		/// </summary>
		public string InitialDirectory
		{
			get{ return _FileListView.CurrentDirectory; }
			set{ _FileListView.CurrentDirectory = value; }
		}

		/// <summary>
		/// ダイアログ上で選択されたファイル名
		/// </summary>
		public string FileName
		{
			get{ return _FileName; }
			set
			{
				if( _InitFileName == null )
				{
					_InitFileName = value;
				}
				_FileName = value;
				_FileNameTextBox.Text = value;
			}
		}

		/// <summary>
		/// ダイアログでリストアップするファイルの種類を限定するフィルター
		/// </summary>
		public string Filter
		{
			get{ return _DefaultFilter; }
			set
			{
				// clear combo box
				_FileTypeComboBox.Items.Clear();

				// setup combo box
				string[] tokens = value.Split( '|' );
				for( int i=0; i<tokens.Length; i+=2 )
				{
					FileTypeComboBoxItem item = new FileTypeComboBoxItem( tokens[i], tokens[i+1] );
					_FileTypeComboBox.Items.Add( item );
				}

				// select first one
				_FileTypeComboBox.SelectedIndex = 0;
			}
		}

		/// <summary>
		/// 設定されている表示フィルターのうち、選択されているもののインデックスを取得、設定します。
		/// </summary>
		public int FilterIndex
		{
			get{ return _FileTypeComboBox.SelectedIndex; }
			set{ _FileTypeComboBox.SelectedIndex = value; }
		}
		#endregion

		#region Behavior as a Control
		/// <summary>
		/// ダイアログのフォーカス移動キーの動作を変更
		/// </summary>
		protected override bool ProcessDialogKey( Keys keyData )
		{
			// ファイル一覧がフォーカスされている時は Enter でダイアログを決定しない
			if( keyData == Keys.Enter && _FileListView.Focused )
			{
				return false;
			}

			return base.ProcessDialogKey( keyData );
		}

		/// <summary>
		/// メニューオブジェクトで利用できないショートカットキーを実装
		/// </summary>
		protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
		{
			int key = msg.WParam.ToInt32();
			
			// Ctrl+\ でカレントドライブのルートに移動
			if( Sgry.Win32.IsControlKeyDown()
				&& (key == 226 || key == 220) )
			{
				_MI_View_GoToRoot_Click( this, EventArgs.Empty );
				return true;
			}

			return base.ProcessCmdKey( ref msg, keyData );
		}
		#endregion

		#region UI event handlers
		/// <summary>
		/// OKボタンが押された時の動作。
		/// </summary>
		void OkButton_Click( object sender, EventArgs e )
		{
			string	fullPath;
			string	dirPath;
			string	fileName;
			bool	isValid;

			try
			{
				// 何も入力されていなければ何もしない
				if( _FileNameTextBox.Text == String.Empty )
				{
					Utl.MessageBeep_Notify();
					return;
				}

				// 入力されたパスを解析
				// （絶対パスを作り、ディレクトリ部とファイル部を分割。".."の解決も行う）
				fullPath = Utl.ReGetFilePath( _FileListView.CurrentDirectory, _FileNameTextBox.Text );
				if( Directory.Exists(fullPath) )
				{
					dirPath = fullPath;
					fileName = String.Empty;
				}
				else
				{
					dirPath = Path.GetDirectoryName( fullPath );
					fileName = Path.GetFileName( fullPath );
					if( dirPath == null )
					{
						new FileInfo(fullPath);
					}
				}

				// ファイル以外が指定された場合の処理を実行
				if( dirPath == MyExplorer.MyComputerTagName )
				{
					Utl.MessageBeep_Notify();
					return;
				}
				if( fileName == String.Empty )
				{
					//-- ディレクトリが指定された --
					// そのディレクトリに移動
					_FileListView.CurrentDirectory = fullPath;
					_FileListView.UpdateFileList();
					_FileNameTextBox.Text = String.Empty;
					return;
				}
				else if( fileName.IndexOfAny(WildcardChars) != -1 )
				{
					//-- ワイルドカードが入力された --
					// 指定ディレクトリへ移動
					if( dirPath != _FileListView.CurrentDirectory )
					{
						_FileListView.CurrentDirectory = dirPath;
					}

					// そのワイルドカードでフィルタ適用
					_FileListView.Filter = fileName;
					_FileListView.UpdateFileList();
					_FileNameTextBox.SelectAll();
					return;
				}

				// ファイルが指定されたなら、入力されたファイル名を検証
				isValid = ValidateFileName( fullPath );
				if( isValid != true )
				{
					return;
				}

				// ユーザの決定内容を記憶
				_FileName = fullPath;

				this.DialogResult = DialogResult.OK;
			}
			catch( Exception ex )
			{
				NbsMessageBox.Show( this, _Msg_InvalidFileName+ex.Message, MessageBoxButtons.OK, MessageBoxIcon.Error );
				_FileNameTextBox.SelectAll();
				_FileNameTextBox.Focus();
				return;
			}
		}

		void _MI_File_Close_Click( object sender, EventArgs e )
		{
			DialogResult = DialogResult.Cancel;
			this.Close();
		}

		/// <summary>
		/// ファイル一覧で項目が選択された時の動作。
		/// </summary>
		void _FileListView_TargetFileSelected( string path )
		{
			// 存在するファイルなら、そのファイルに決定する
			if( File.Exists(path) )
			{
				OkButton_Click( this, EventArgs.Empty );
			}
		}

		/// <summary>
		/// 「ファイル一覧」で選択が変更された時の動作。
		/// 選択されたファイル項目を「ファイル名」テキストボックスに設定
		/// </summary>
		void _FileListView_SelectedEntryChanged( string path )
		{
			// ファイル名入力時に起こった変更に対しては何もしない
			// （ワイルドカード入力で表示更新が起こった場合など）
			if( _FileNameTextBox.Focused )
				return;

			// 「ファイル名」テキストボックスを更新
			if( path == MyExplorer.MyComputerTagName )
			{
				_FileNameTextBox.Text = String.Empty;
			}
			else
			{
				_FileNameTextBox.Text = Path.GetFileName( path );
			}
		}

		/// <summary>
		/// 「ファイル一覧」で表示中ディレクトリが変化した直後の動作。
		/// </summary>
		void _FileListView_CurrentDirectoryChanged( object sender, EventArgs e )
		{
			string newDirName;
			
			// 読み上げる
			newDirName = Path.GetFileName( _FileListView.CurrentDirectory );
			if( newDirName == String.Empty )
			{
				newDirName = _FileListView.CurrentDirectory;
			}

			// カレントディレクトリの表示を変更
			_DirectoryPathTextBox.Text = _FileListView.CurrentDirectory;
			AutoSpeaker.Instance.Speak( String.Format(_Msg_WentTo, newDirName) );
			if( _FileListView.FocusedItem != null )
			{
				AutoSpeaker.Instance.Speak( _FileListView.FocusedItem.Text );
			}
		}

		/// <summary>
		/// ファイル種別を変更された時の動作。
		/// </summary>
		void _FileTypeComboBox_SelectedIndexChanged( object sender, EventArgs e )
		{
			FileTypeComboBoxItem item = (FileTypeComboBoxItem)_FileTypeComboBox.SelectedItem;
			_FileListView.Filter = item.Value;
			_FileListView.UpdateFileList();
		}

		void _MI_View_GoToDriveList_Click( object sender, EventArgs e )
		{
			// ドライブ一覧に移動
			_FileListView.CurrentDirectory = MyExplorer.MyComputerTagName;
			_FileListView.UpdateFileList();
		}

		void _MI_View_GoToDesktop_Click( object sender, EventArgs e )
		{
			// デスクトップのパスを取得
			string desktopPath = Environment.GetFolderPath( Environment.SpecialFolder.DesktopDirectory );

			// デスクトップへ移動
			_FileListView.CurrentDirectory = desktopPath;
			_FileListView.UpdateFileList();
		}

		void _MI_View_GoToMyDocuments_Click( object sender, EventArgs e )
		{
			// マイドキュメントのパスを取得
			string mydocPath = Environment.GetFolderPath( Environment.SpecialFolder.MyDocuments );

			// 移動
			_FileListView.CurrentDirectory = mydocPath;
			_FileListView.UpdateFileList();
		}

		void _MI_View_GoToRoot_Click( object sender, EventArgs e )
		{
			string rootPath;

			// 現在がマイコンピュータなら、それが属するルートなど無い
			if( _FileListView.CurrentDirectory == MyExplorer.MyComputerTagName )
			{
				Win32.MessageBeep_Notify();
				return;
			}

			// ルートへ移動
			rootPath = Path.GetPathRoot( _FileListView.CurrentDirectory );
			_FileListView.CurrentDirectory = Path.GetPathRoot( _FileListView.CurrentDirectory );
			_FileListView.UpdateFileList();
		}
		#endregion

		#region UI Component Initialization
		/// <summary>
		/// ダイアログで使用する点字表示クライアントを初期化します。
		/// </summary>
		protected override void InstallBraillers()
		{
			_ControlBrailler.Init();
			_ControlBrailler.Add( _OkButton );
			_ControlBrailler.Add( _CancelButton );

			_TextBoxBrailler.Init();
			_TextBoxBrailler.Add( _FileNameTextBox );
			_TextBoxBrailler.Add( _DirectoryPathTextBox );

			_ComboBoxBrailler.Init();
			_ComboBoxBrailler.Add( _FileTypeComboBox );

			_ListViewBrailler.Init();
			_ListViewBrailler.Add( _FileListView );
		}

		/// <summary>
		/// ダイアログで使用した点字表示クライアントのリソースを解放します。
		/// </summary>
		protected override void UninstallBraillers()
		{
			_ControlBrailler.Dispose();
			_TextBoxBrailler.Dispose();
			_ComboBoxBrailler.Dispose();
			_ListViewBrailler.Dispose();
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			_TextBoxReader.Add( _DirectoryPathTextBox );
			_TextBoxReader.Add( _FileNameTextBox );
			_ComboBoxReader.Add( _FileTypeComboBox );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void UninstallReaders()
		{
			_TextBoxReader.Dispose();
			_ComboBoxReader.Dispose();
		}

		/// <summary>
		/// GUI 要素のテキストを実行環境の言語に翻訳します。
		/// </summary>
		protected virtual void LocalizeComponent()
		{
			_Localizer.LoadResourceFile( "CommonUtl" );

			_Localizer.TryGetString( "AiBFileDialog._MI_File.Text", _MI_File );
			_Localizer.TryGetString( "AiBFileDialog._MI_File_Close.Text", _MI_File_Close );
			_Localizer.TryGetString( "AiBFileDialog._MI_View.Text", _MI_View );
			_Localizer.TryGetString( "AiBFileDialog._MI_View_GoToMyComputer.Text", _MI_View_GoToMyComputer );
			_Localizer.TryGetString( "AiBFileDialog._MI_View_GoToDesktop.Text", _MI_View_GoToDesktop );
			_Localizer.TryGetString( "AiBFileDialog._MI_View_GoToMyDocuments.Text", _MI_View_GoToMyDocuments );
			_Localizer.TryGetString( "AiBFileDialog._MI_View_GoToRoot.Text", _MI_View_GoToRoot );
			_Localizer.TryGetString( "AiBFileDialog._DirectoryPathLabel.Text", _DirectoryPathLabel );
			_Localizer.TryGetString( "AiBFileDialog._FileListLabel.Text", _FileListLabel );
			_Localizer.TryGetString( "AiBFileDialog._FileNameLabel.Text", _FileNameLabel );
			_Localizer.TryGetString( "AiBFileDialog._FileTypeLabel.Text", _FileTypeLabel );
			_Localizer.TryGetString( "AiBFileDialog._Button_Cancel.Text", _CancelButton );
			_Localizer.TryGetString( "AiBFileDialog._Msg_WentTo.Text", ref _Msg_WentTo );
			_Localizer.TryGetString( "AiBFileDialog._Msg_InvalidFileName", ref _Msg_InvalidFileName );

			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				_DefaultFilter = "すべてのファイル (*.*)|*.*";
			}

			this.Filter = _DefaultFilter;
		}

		void InitializeComponents()
		{
			SuspendLayout();
			// 
			// _MainMenu
			// 
			_MainMenu.MenuItems.AddRange( new MenuItem[] {
            _MI_File,
            _MI_View} );
			// 
			// _MI_File
			// 
			_MI_File.Index = 0;
			_MI_File.MenuItems.AddRange( new MenuItem[] {
            _MI_File_Close} );
			_MI_File.Text = "&File";
			// 
			// _MI_File_Close
			// 
			_MI_File_Close.Index = 0;
			_MI_File_Close.Text = "&Close";
			_MI_File_Close.Click += _MI_File_Close_Click;
			// 
			// _MI_View
			// 
			_MI_View.Index = 1;
			_MI_View.MenuItems.AddRange(
					new MenuItem[]{ _MI_View_GoToMyComputer, _MI_View_GoToDesktop, _MI_View_GoToMyDocuments, _MI_View_GoToRoot}
				);
			_MI_View.Text = "&View";
			//
			// _MI_View_GoToMyComputer
			//
			_MI_View_GoToMyComputer.Index = 0;
			_MI_View_GoToMyComputer.Shortcut = Shortcut.CtrlL;
			_MI_View_GoToMyComputer.Text = "Go to drive &list";
			_MI_View_GoToMyComputer.Click += _MI_View_GoToDriveList_Click;
			// 
			// _MI_View_GoToDesktop
			// 
			_MI_View_GoToDesktop.Index = 1;
			_MI_View_GoToDesktop.Shortcut = Shortcut.CtrlT;
			_MI_View_GoToDesktop.Text = "Go to desk&top";
			_MI_View_GoToDesktop.Click += _MI_View_GoToDesktop_Click;
			//
			// _MI_View_GoToMyDocuments
			//
			_MI_View_GoToMyDocuments.Index = 2;
			_MI_View_GoToMyDocuments.Shortcut = Shortcut.CtrlD;
			_MI_View_GoToMyDocuments.Text = "Go to My &Documents";
			_MI_View_GoToMyDocuments.Click += _MI_View_GoToMyDocuments_Click;
			// 
			// _MI_View_GoToRoot
			// 
			_MI_View_GoToRoot.Index = 3;
			_MI_View_GoToRoot.Text = "Go to &root directory\tCtrl+\\";
			_MI_View_GoToRoot.Click += _MI_View_GoToRoot_Click;
			// 
			// _DirectoryPathLabel
			// 
			_DirectoryPathLabel.Location = new Point( 12, 8 );
			_DirectoryPathLabel.Name = "_DirectoryPathLabel";
			_DirectoryPathLabel.Size = new Size( 155, 19 );
			_DirectoryPathLabel.Text = "Current D&irectory";
			// 
			// _DirectoryPathTextBox
			// 
			_DirectoryPathTextBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
			_DirectoryPathTextBox.Location = new Point( 175, 6 );
			_DirectoryPathTextBox.Name = "_DirectoryPathTextBox";
			_DirectoryPathTextBox.ReadOnly = true;
			_DirectoryPathTextBox.Size = new Size( 395, 19 );
			// 
			// _FileListLabel
			// 
			_FileListLabel.Location = new Point( 14, 29 );
			_FileListLabel.Name = "_FileListLabel";
			_FileListLabel.Size = new Size( 556, 19 );
			_FileListLabel.Text = "File &list";
			_FileListLabel.TextAlign = System.Drawing.ContentAlignment.BottomLeft;
			// 
			// _FileListView
			// 
			_FileListView.Anchor = AnchorStyles.Top | AnchorStyles.Bottom
						| AnchorStyles.Left | AnchorStyles.Right;
			_FileListView.FullRowSelect = true;
			_FileListView.GridLines = true;
			_FileListView.HideSelection = false;
			_FileListView.Location = new Point( 14, 51 );
			_FileListView.Name = "_FileListView";
			_FileListView.Size = new Size( 556, 178 );
			_FileListView.View = View.Details;
			_FileListView.Choosed += _FileListView_TargetFileSelected;
			_FileListView.SelectedEntryChanged += _FileListView_SelectedEntryChanged;
			_FileListView.CurrentDirectoryChanged += _FileListView_CurrentDirectoryChanged;
			// 
			// _FileNameLabel
			// 
			_FileNameLabel.Anchor = AnchorStyles.Bottom | AnchorStyles.Left;
			_FileNameLabel.Location = new Point( 12, 240 );
			_FileNameLabel.Name = "_FileNameLabel";
			_FileNameLabel.Size = new Size( 135, 20 );
			_FileNameLabel.Text = "File &name";
			// 
			// _FileNameTextBox
			// 
			_FileNameTextBox.Anchor = AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right;
			_FileNameTextBox.Location = new Point( 148, 237 );
			_FileNameTextBox.Name = "_FileNameTextBox";
			_FileNameTextBox.Size = new Size( 324, 20 );
			// 
			// _FileTypeLabel
			// 
			_FileTypeLabel.Anchor = AnchorStyles.Top | AnchorStyles.Left;
			_FileTypeLabel.Location = new Point( 12, 265 );
			_FileTypeLabel.Name = "_FileTypeLabel";
			_FileTypeLabel.Size = new Size( 135, 20 );
			_FileTypeLabel.Text = "File &type";
			// 
			// _FileTypeComboBox
			// 
			_FileTypeComboBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
			_FileTypeComboBox.DropDownStyle = ComboBoxStyle.DropDownList;
			_FileTypeComboBox.Location = new Point( 148, 262 );
			_FileTypeComboBox.Name = "_FileTypeComboBox";
			_FileTypeComboBox.Size = new Size( 324, 20 );
			_FileTypeComboBox.SelectedIndexChanged += _FileTypeComboBox_SelectedIndexChanged;
			//
			// _CustomPanel
			//
			_CustomPanel.Anchor = AnchorStyles.Left | AnchorStyles.Bottom | AnchorStyles.Right;
			_CustomPanel.Location = new Point( 12, 287 );
			_CustomPanel.Name = "_CustomPanel";
			_CustomPanel.Size = new Size( 460, 0 );
			// 
			// _Button_Ok
			// 
			_OkButton.FlatStyle = FlatStyle.System;
			_OkButton.Anchor = AnchorStyles.Bottom | AnchorStyles.Right;
			_OkButton.Location = new Point( 486, 235 );
			_OkButton.Name = "_Button_Ok";
			_OkButton.Size = new Size( 84, 24 );
			_OkButton.Text = "OK";
			_OkButton.Click += OkButton_Click;
			// 
			// _Button_Cancel
			// 
			_CancelButton.FlatStyle = FlatStyle.System;
			_CancelButton.Anchor = AnchorStyles.Bottom | AnchorStyles.Right;
			_CancelButton.DialogResult = DialogResult.Cancel;
			_CancelButton.Location = new Point( 486, 265 );
			_CancelButton.Name = "_Button_Cancel";
			_CancelButton.Size = new Size( 84, 24 );
			_CancelButton.Text = "Cancel";
			// 
			// AiBFileDialog
			// 
			AcceptButton = _OkButton;
			CancelButton = _CancelButton;
			Controls.Add( _DirectoryPathLabel );
			Controls.Add( _DirectoryPathTextBox );
			Controls.Add( _FileListLabel );
			Controls.Add( _FileListView );
			Controls.Add( _FileNameLabel );
			Controls.Add( _FileNameTextBox );
			Controls.Add( _FileTypeLabel );
			Controls.Add( _FileTypeComboBox );
			Controls.Add( _CustomPanel );
			Controls.Add( _OkButton );
			Controls.Add( _CancelButton );
			Menu = _MainMenu;
			MinimizeBox = false;
			Name = "AiBFileDialog";
			ShowInTaskbar = false;
			SizeGripStyle = SizeGripStyle.Show;

			// カスタムパネルを設定し、そのサイズに合わせてダイアログサイズを変更
			SetupCustomPanel();
			ClientSize = new Size(
					582,
					Math.Max(_CustomPanel.Bottom, _CancelButton.Bottom)
				);

			ResumeLayout( false );
			PerformLayout();
		}
		#endregion

		#region UI Components
		MainMenu	_MainMenu			= new MainMenu();
		MenuItem	_MI_File			= new MenuItem();
		MenuItem	_MI_File_Close		= new MenuItem();
		MenuItem	_MI_View			= new MenuItem();
		MenuItem	_MI_View_GoToDesktop		= new MenuItem();
		MenuItem	_MI_View_GoToRoot			= new MenuItem();
		MenuItem	_MI_View_GoToMyComputer		= new MenuItem();
		MenuItem	_MI_View_GoToMyDocuments	= new MenuItem();

		Label					_DirectoryPathLabel	= new Label();
		TextBox					_DirectoryPathTextBox	= new TextBox();
		Label					_FileListLabel		= new Label();

		/// <summary>ファイル一覧ビュー</summary>
		protected MyExplorer	_FileListView		= new MyExplorer();
		Label					_FileNameLabel		= new Label();
		TextBox					_FileNameTextBox	= new TextBox();
		Label					_FileTypeLabel		= new Label();
		ComboBox				_FileTypeComboBox	= new ComboBox();

		/// <summary>OK ボタンです。</summary>
		protected Button	_OkButton			= new Button();

		/// <summary>キャンセルボタンです。</summary>
		protected Button	_CancelButton		= new Button();

		/// <summary>
		/// カスタムコンポーネントを配置するパネルです。
		/// </summary>
		protected Panel		_CustomPanel		= new Panel();

		/// <summary></summary>
		protected TextBoxReader		_TextBoxReader	= new TextBoxReader();

		/// <summary></summary>
		protected ComboBoxReader	_ComboBoxReader	= new ComboBoxReader();

		/// <summary></summary>
		protected ControlBrailler	_ControlBrailler	= new ControlBrailler();
		
		/// <summary></summary>
		protected TextBoxBrailler	_TextBoxBrailler	= new TextBoxBrailler();

		/// <summary></summary>
		protected ComboBoxBrailler	_ComboBoxBrailler	= new ComboBoxBrailler();

		/// <summary></summary>
		protected ListViewBrailler	_ListViewBrailler	= new ListViewBrailler();
		#endregion

		#region Utilities
		/// <summary>
		/// ローカライズユーティリティ
		/// </summary>
		protected Localizer Localizer
		{
			get{ return _Localizer; }
		}

		class Utl
		{
			/// <summary>
			/// ファイルシステムから正確なパス文字列を再取得する。
			/// </summary>
			/// <remarks>
			/// Windows では大文字小文字を区別すると不正なパスでも使えてしまう。
			/// 大文字小文字の別も正確に取得するために呼ぶ。
			/// </remarks>
			/// <exception cref="System.IO.IOException">I/Oエラー</exception>
			public static string ReGetFilePath( string baseDir, string filePath )
			{
				Debug.Assert( baseDir != null && filePath != null );

				string dir, filter;

				// まずはフルパスに変換
				if( filePath.StartsWith(@"\\") )
				{
					// UNC フルパス。何もしない
				}
				else if( filePath.StartsWith(@"\") )
				{
					// Path.Combine に C:\ と \hoge.txt を与えると
					// C:\hoge.txt ではなく \hoge.txt が返ってくるので対策
					dir = Path.GetPathRoot(baseDir);
					filePath = dir + filePath.Substring( 1 );
				}
				else if( Path.IsPathRooted(filePath) == false )
				{
					filePath = Path.Combine( baseDir, filePath );
				}

				// 存在するファイルならばファイルシステムから名前を取得し直す
				if( File.Exists(filePath)
					|| Directory.Exists(filePath) )
				{
					dir = Path.GetDirectoryName( filePath );
					if( dir == null )
					{
						// パスがルートディレクトリを指しているはず。
						// ドライブレターだけ大文字に
						filePath = Char.ToUpper(filePath[0]) + filePath.Substring( 1 );
					}
					else
					{
						filter = Path.GetFileName( filePath );
						filePath = Directory.GetFileSystemEntries( dir, filter )[0];
					}
				}
				return filePath;
			}

			/// <summary>
			/// 重要でなくユーザを妨げるほどでもない何かを通知する
			/// </summary>
			public static void MessageBeep_Notify()
			{
				MessageBeep( 0 );
			}

			[DllImport("user32", EntryPoint="MessageBeep")]
			static extern int MessageBeep( int type );
		}
		#endregion
	}
}

