// file: JavacCompiler.cs
// brief: concrete compiler object for javac
// encoding: UTF-8
// update: 2008-12-21
//=========================================================
using System;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using Process = System.Diagnostics.Process;
using ProcessStartInfo = System.Diagnostics.ProcessStartInfo;
using ProcessWindowStyle = System.Diagnostics.ProcessWindowStyle;

namespace Sgry.AiBTools.Compilation
{
	/// <summary>
	/// コンパイルを実行するクラス
	/// </summary>
	public class JavacCompiler : Compiler
	{
		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		/// <param name="argument">コンパイラに渡す引数</param>
		/// <param name="workDirPath">コンパイラを起動するディレクトリ</param>
		public JavacCompiler( string argument, string workDirPath )
			: base( argument, workDirPath )
		{}
		#endregion

		#region Properties
		/// <summary>
		/// コンパイラの実行ファイル名を取得します。
		/// </summary>
		public override string ExecutableName
		{
			get{ return "javac.exe"; }
		}

		/// <summary>
		/// 直前のコンパイルが成功したかどうかを示す値を取得します。
		/// </summary>
		public override bool Succeeded
		{
			get{ return (_ExitCode == 0); }
		}

		/// <summary>
		/// 直前のコンパイルが失敗したかどうかを示す値を取得します。
		/// </summary>
		public override bool Failed
		{
			get{ return (_ExitCode == 1 && 0 < _Results.Count); }
		}
		#endregion

		#region Interface
		/// <summary>
		/// コンパイラを実行
		/// </summary>
		public override void Compile()
		{
			Process process = new Process();
			ProcessStartInfo pInfo;

			// 起動するプロセスを設定
			pInfo = new ProcessStartInfo();
			pInfo.FileName = "javac";
			pInfo.WorkingDirectory = _WorkDirPath;
			pInfo.UseShellExecute = false;
			pInfo.RedirectStandardError = true;
			pInfo.WindowStyle = ProcessWindowStyle.Hidden;
			pInfo.CreateNoWindow = true;
			pInfo.Arguments = _Argument;
			
			// プロセスを起動
			process.StartInfo = pInfo;
			process.Start();

			// 標準エラー出力の取得を開始
			_Log = process.StandardError.ReadToEnd();

			// プロセス終了まで待つ
			process.WaitForExit();

			// ログを解析
			_Results.Clear();
			ParseLog();

			// コンパイラの終了コードを記録
			_ExitCode = process.ExitCode;
		}
		#endregion

		#region Parse Logic
		void ParseLog()
		{
			Localizer	localizer = new Localizer();
			string[]	lines;
			string		filePath;
			int			lineIndex, columnIndex;
			string		message;
			ResultItem	result;

			localizer.LoadResourceFile( "CommonUtl" );

			lines = Utl.SplitToLine( _Log );
			for( int i=0; i<lines.Length; i++ )
			{
				// エラーブロック開始行を処理
				Match match = Regex.Match( lines[i], @"((?:[a-zA-Z]:)?[^:]+):(\d+): (.+)" );
				if( !match.Success )
				{
					continue; // エラーブロック開始行でない
				}

				// 解析結果の後処理を実行
				filePath = match.Groups[1].ToString();
				Int32.TryParse( match.Groups[2].ToString(), out lineIndex );
				columnIndex = ParseErrorColumn( lines, i );
				message = match.Groups[3].ToString().TrimStart();
				if( message.StartsWith("シンボルを")
					|| message.StartsWith("cannot find symbol") )
				{
					string		format;
					string[]	tokens;
					
					// 未解決のシンボル名を切り出す
					tokens = lines[i+1].Split( ':' );
					if( tokens == null || tokens.Length < 2 )
					{
						return; // 正常なエラーメッセージではない？
					}

					format = localizer.TryGetString(
							"ErrorListForm.Msg_CannotResolveSymbol",
							"{0} was not found."
						);
					message = String.Format( format, tokens[1].Trim() );
				}

				result = new ResultItem( message, filePath, lineIndex, columnIndex, ResultType.Error );
				_Results.Add( result );
			}
		}

		/// <summary>
		/// コンパイラのエラーメッセージを解析します。
		/// </summary>
		// 例
		// C:\work\dev\AiBTools\_javac\TestFiles\HelloJava.java:7: ')' がありません。
		// }
		// ^
		// ..\TestFiles\HelloJava.java:7: ')' がありません。
		// }
		// ^
		ResultItem ParseErrorLine( string[] logLines, int index )
		{
			string filePath;
			int lineIndex, columnIndex;
			string message;
			ResultType type;

			Match match = Regex.Match( @"((?:[a-zA-Z]:)?[^:]+):(\d+): (.+)", logLines[index] );
			if( match.Success )
			{
				filePath = match.Groups[1].ToString();
				Int32.TryParse( match.Groups[2].ToString(), out lineIndex );
				columnIndex = ParseErrorColumn( logLines, index );
				type = ResultType.Error;
				message = match.Groups[3].ToString().TrimStart();

				return new ResultItem( message, filePath, lineIndex, 0, type );
			}

			return null;
		}

		/// <summary>
		/// エラーメッセージからエラーが発生した桁位置を計算
		/// </summary>
		/// <param name="logLines">エラーメッセージ全体の各行を含む配列</param>
		/// <param name="index">解析中の行の、配列中のインデックス</param>
		/// <returns>何桁目でエラーが起きたか</returns>
		/// <remarks>失敗したら 0 が返る</remarks>
		static int ParseErrorColumn( string[] logLines, int index )
		{
			char[] white_spaces = new char[]{' ', '\t'};
			string hatOnlyLine = null;
			
			// このエラーメッセージの、'^' のみを含む行を抽出
			for( int i=index; i<logLines.Length; i++ )
			{
				// 最初の非空白文字が '^' か？
				if( logLines[i].TrimStart(white_spaces).StartsWith("^") )
				{
					hatOnlyLine = logLines[i];
					break;
				}
			}

			// 抽出できなければ終了
			if( hatOnlyLine == null )
			{
				return 0;
			}

			// '^' があるインデックス＋１を返す
			// （Tab 文字は空白８つとして計算）
			return hatOnlyLine.IndexOf( '^' ) + 1; // 見つからないと、計算結果 (-1+1=)0 が返る
		}
		#endregion
	}
}
