﻿// file: ErrorListForm.cs
// brief: form to list up compile errors.
// encoding: UTF-8
// update: 2008-12-14
//=========================================================
using System;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;

namespace Sgry.AiBTools.Compilation
{
	using Gui;
	using AT;

	/// <summary>
	/// コンパイラが報告したコンパイルエラーを表示するフォーム
	/// </summary>
	public class ErrorListForm : AiBDialogBase
	{
		#region フィールド
		Compiler	_Compiler;
		string		_EditorPath;
		Localizer	_Localizer = new Localizer();
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public ErrorListForm( Compiler compiler )
		{
			// UI を初期化
			InitializeComponent();
			LocalizeComponents();

			Load += Form_Load;
			Shown += Form_Shown;

			// パラメータを保存
			_Compiler = compiler;

			// AiB Edit のパスを生成
			_EditorPath = Path.GetDirectoryName( Application.ExecutablePath );
			_EditorPath = Path.Combine( _EditorPath, "AiBEditWin.exe" );
		}

		void Form_Load( object sender, EventArgs e )
		{
			// リスト表示を更新
			ParseLogAndUpdateList();

			// カラム幅を自動調整
			_LVH_File.Width = -2;
			_LVH_Message.Width = -2;
		}

		void Form_Shown( object sender, EventArgs e )
		{
			// 最初の項目を選択
			if( 0 < _ListView.Items.Count )
			{
				_ListView.Items[0].Focused = true;
				_ListView.Items[0].Selected = true;
				_ListView.EnsureVisible( 0 );
			}
		}
		#endregion

		#region エラーメッセージの処理
		void ParseLogAndUpdateList()
		{
			string titleFormat;

			// コンパイル結果をリストに表示
			_ListView.Items.Clear();
			for( int i=0; i<_Compiler.Results.Count; i++ )
			{
				ListViewItem lvItem = new ListViewItem();
				ResultItem item = _Compiler.Results[i];
				lvItem.Text = String.Format( "{0} {1}", i+1, item.Message );
				lvItem.SubItems.Add( item.FilePath );
				lvItem.SubItems.Add( item.LineIndex.ToString() );
				lvItem.SubItems.Add( item.ColumnIndex.ToString() ); // 隠れた列に追加。表示はされないが、後で取得。
				_ListView.Items.Add( lvItem );
			}

			// ウィンドウタイトルを設定
			if( 1 == _Compiler.Results.Count ) // 1 error
			{
				titleFormat = _Localizer.TryGetString( "ErrorListForm.TitleFormat_SingleError", "{0} compile error" );
			}
			else // multiple errors
			{
				titleFormat = _Localizer.TryGetString( "ErrorListForm.TitleFormat_MultipleErrors", "{0} compile errors" );
			}
			this.Text = Application.ProductName
				+ " - "
				+ String.Format( titleFormat, _Compiler.Results.Count );
		}

		/// <summary>
		/// 選択中のエラー箇所をエディタで開く
		/// </summary>
		void OpenErrorLocationWithEditor()
		{
			string filePath;
			string line, column;
			string argument;
			ListViewItem selectedItem;

			// 選択しているアイテムを取得
			if( _ListView.SelectedItems.Count == 0 )
			{
				return;
			}
			selectedItem = _ListView.SelectedItems[0];

			// 選択アイテムからファイルパス、行位置や桁位置を取得
			filePath = selectedItem.SubItems[1].Text;
			line = selectedItem.SubItems[2].Text;
			column = selectedItem.SubItems[3].Text;
			if( filePath == String.Empty )
			{
				// 開けないと告げて終了
				Win32.MessageBeep_Notify();
				if( AutoSpeaker.IsAlive() )
				{
					string message = _Localizer.TryGetString( "ErrorListForm.Msg_UnopenableItem", "Cannot open this item." );
					AutoSpeaker.Instance.Stop();
					AutoSpeaker.Instance.Speak( message );
				}
				return;
			}

			// setup parameters for editor
			filePath = Path.Combine( _Compiler.ProjectDirPath, filePath );
			argument = "--line=" + line
				+ " --column=" + column
				+ " \"" + filePath + "\"";

			// launch editor
			try
			{
				System.Diagnostics.Process.Start( _EditorPath, argument );
			}
			catch( Exception ex )
			{
				// メッセージを生成
				string format = _Localizer.TryGetString(
					"ErrorListForm.Msg_FailedToLaunchEditor",
					"IsCompileOk to launch text editor. {0}");
				string message = String.Format( format, ex.Message );

				// メッセージを表示
				NbsMessageBox.Show( this, message );
			}
		}

		/// <summary>
		/// 再コンパイル
		/// </summary>
		void Recompile()
		{
			// コンパイル
			using( Form dialog = new NowCompilingForm(_Compiler) )
			{
				dialog.ShowDialog( this );
				Win32.MessageBeep_Information();
			}

			// コンパイル結果を判定
			if( _Compiler.Succeeded )
			{
				// コンパイル成功
				string message = _Localizer.TryGetString( "Msg_CompileSucceeded", "Successfully compiled." );
				NbsMessageBox.Show( this, message );
				this.Close(); // 終了
			}
			else if( _Compiler.Failed )
			{
				// コンパイルエラー発生。
				// パースしてリストに表示
				ParseLogAndUpdateList();

				// 点字ディスプレイの表示を更新
				if( _ListViewBrailler != null )
				{
					_ListViewBrailler.UpdateDisplay();
				}
			}
			else
			{
				// その他のエラー。(ここには本当は来ないはずだが)
				// そのまま表示
				using( OutputForm form = new OutputForm(_Compiler.Log) )
				{
					form.Text = "";
					Win32.MessageBeep_Error();
					form.ShowDialog();
				}
			}
		}
		#endregion

		#region リストビュー
		void _ListView_DoubleClick( object sender, EventArgs e )
		{
			OpenErrorLocationWithEditor();
		}

		void _ListView_KeyDown( object sender, KeyEventArgs e )
		{
			// Enter ならばエディタでエラー箇所を開く
			if( e.KeyCode == Keys.Enter )
			{
				OpenErrorLocationWithEditor();
			}
		}
		#endregion // リストビュー

		#region メニュー
		void _MI_OpenError_Click( object sender, EventArgs e )
		{
			OpenErrorLocationWithEditor();
		}

		void _MI_ViewLog_Click( object sender, EventArgs e )
		{
			using( OutputForm logForm = new OutputForm(_Compiler.Log) )
			{
				logForm.Text = "Error log";
				logForm.ShowDialog( this );
			}
		}

		void _MI_Close_Click( object sender, EventArgs e )
		{
			this.Close();
		}

		void _MI_Recompile_Click( object sender, EventArgs e )
		{
			Recompile();
		}
		#endregion // メニュー

		#region UI Component Initialization
		/// <summary>
		/// ダイアログで使用する点字表示クライアントを初期化します。
		/// </summary>
		protected override void InstallBraillers()
		{
			_ListViewBrailler.Init();
			_ListViewBrailler.Add( _ListView );
		}

		/// <summary>
		/// ダイアログで使用した点字表示クライアントのリソースを解放します。
		/// </summary>
		protected override void UninstallBraillers()
		{
			_ListViewBrailler.Dispose();
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void UninstallReaders()
		{}

		/// <summary>
		/// 文字列をローカライズ
		/// </summary>
		void LocalizeComponents()
		{
			_Localizer.LoadResourceFile( "CommonUtl" );

			_Localizer.TryGetString( "ErrorListForm._LVH_Message.Text", _LVH_Message );
			_Localizer.TryGetString( "ErrorListForm._LVH_File.Text", _LVH_File );
			_Localizer.TryGetString( "ErrorListForm._LVH_Line.Text", _LVH_Line );
			_Localizer.TryGetString( "ErrorListForm._MI_File.Text", _MI_File );
			_Localizer.TryGetString( "ErrorListForm._MI_File_OpenError.Text", _MI_File_OpenError );
			_Localizer.TryGetString( "ErrorListForm._MI_File_ViewLog.Text", _MI_File_ViewLog );
			_Localizer.TryGetString( "ErrorListForm._MI_File_Close.Text", _MI_File_Close );
			_Localizer.TryGetString( "ErrorListForm._MI_Tool.Text", _MI_Tool );
			_Localizer.TryGetString( "ErrorListForm._MI_Tool_Recompile.Text", _MI_Tool_Recompile );
		}

		/// <summary>
		/// デザイナ サポートに必要なメソッドです。このメソッドの内容を
		/// コード エディタで変更しないでください。
		/// </summary>
		void InitializeComponent()
		{
			_ListView = new AiBListView();
			_LVH_Message = new ColumnHeader();
			_LVH_File = new ColumnHeader();
			_LVH_Line = new ColumnHeader();
			_MainMenu = new MainMenu();
			_MI_File = new MenuItem();
			_MI_File_OpenError = new MenuItem();
			_MI_File_ViewLog = new MenuItem();
			_MI_File_Separator = new MenuItem();
			_MI_File_Close = new MenuItem();
			_MI_Tool = new MenuItem();
			_MI_Tool_Recompile = new MenuItem();
			SuspendLayout();
			// 
			// listView
			// 
			_ListView.AutoArrange = false;
			_ListView.Columns.AddRange(
					new ColumnHeader[] {_LVH_Message, _LVH_File, _LVH_Line}
				);
			_ListView.Dock = DockStyle.Fill;
			_ListView.FullRowSelect = true;
			_ListView.GridLines = true;
			_ListView.HeaderStyle = ColumnHeaderStyle.Nonclickable;
			_ListView.HideSelection = false;
			_ListView.Location = new System.Drawing.Point(0, 0);
			_ListView.Name = "listView";
			_ListView.Size = new System.Drawing.Size(672, 273);
			_ListView.TabIndex = 0;
			_ListView.View = View.Details;
			_ListView.KeyDown += new KeyEventHandler(_ListView_KeyDown);
			_ListView.DoubleClick += _ListView_DoubleClick;
			// 
			// _LVH_Message
			// 
			_LVH_Message.Text = "Message";
			_LVH_Message.Width = 300;
			// 
			// _LVH_File
			// 
			_LVH_File.Text = "File";
			_LVH_File.Width = 130;
			// 
			// _LVH_Line
			// 
			_LVH_Line.Text = "Line";
			// 
			// _MainMenu
			// 
			_MainMenu.MenuItems.AddRange(
				new MenuItem[] {_MI_File, _MI_Tool}
			);
			// 
			// _MI_File
			// 
			_MI_File.Index = 0;
			_MI_File.MenuItems.AddRange(
				new MenuItem[] {_MI_File_OpenError, _MI_File_ViewLog, _MI_File_Separator, _MI_File_Close}
			);
			_MI_File.Text = "&File";
			// 
			// _MI_OpenError
			// 
			_MI_File_OpenError.Index = 0;
			_MI_File_OpenError.ShowShortcut = false;
			_MI_File_OpenError.Text = "&Open error location\t\tEnter";
			_MI_File_OpenError.Click += _MI_OpenError_Click;
			// 
			// _MI_ViewLog
			// 
			_MI_File_ViewLog.Index = 1;
			_MI_File_ViewLog.Shortcut = Shortcut.CtrlL;
			_MI_File_ViewLog.Text = "View &log...";
			_MI_File_ViewLog.Click += _MI_ViewLog_Click;
			// 
			// _MI_File_Separator
			// 
			_MI_File_Separator.Index = 2;
			_MI_File_Separator.Text = "-";
			// 
			// _MI_Close
			// 
			_MI_File_Close.Index = 3;
			_MI_File_Close.Shortcut = Shortcut.CtrlW;
			_MI_File_Close.Text = "&Close";
			_MI_File_Close.Click += _MI_Close_Click;
			// 
			// _MI_Tool
			// 
			_MI_Tool.Index = 1;
			_MI_Tool.MenuItems.AddRange(
					new MenuItem[] {_MI_Tool_Recompile}
				);
			_MI_Tool.Text = "&Tool";
			// 
			// _MI_Recompile
			// 
			_MI_Tool_Recompile.Index = 0;
			_MI_Tool_Recompile.Shortcut = Shortcut.CtrlR;
			_MI_Tool_Recompile.Text = "&Recompile...";
			_MI_Tool_Recompile.Click += _MI_Recompile_Click;
			// 
			// ErrorListForm
			// 
			AutoScaleBaseSize = new System.Drawing.Size(5, 12);
			ClientSize = new System.Drawing.Size(672, 273);
			Controls.Add(_ListView);
			Menu = _MainMenu;
			Name = "ErrorListForm";
			ResumeLayout(false);
		}
		#endregion

		#region UI Components
		ListView		_ListView;
		ColumnHeader	_LVH_File;
		ColumnHeader	_LVH_Line;
		ColumnHeader	_LVH_Message;
		MainMenu		_MainMenu;
		MenuItem		_MI_Tool;
		MenuItem		_MI_Tool_Recompile;
		MenuItem		_MI_File;
		MenuItem		_MI_File_OpenError;
		MenuItem		_MI_File_ViewLog;
		MenuItem		_MI_File_Separator;
		MenuItem		_MI_File_Close;
		ListViewBrailler	_ListViewBrailler = new ListViewBrailler();
		#endregion // UI Components
	}
}

