// file: CscCompiler.cs
// brief: concrete compiler object for csc.exe
// encoding: UTF-8
// update: 2008-12-21
//=========================================================
using System;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.IO;
using Win32Exception = System.ComponentModel.Win32Exception;
using Process = System.Diagnostics.Process;
using ProcessStartInfo = System.Diagnostics.ProcessStartInfo;
using ProcessWindowStyle = System.Diagnostics.ProcessWindowStyle;

namespace Sgry.AiBTools.Compilation
{
	/// <summary>
	/// コンパイルを実行するクラス
	/// </summary>
	public class CscCompiler : Compiler
	{
		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		/// <param name="argument">コンパイラに渡す引数</param>
		/// <param name="workDirPath">コンパイラを起動するディレクトリ</param>
		public CscCompiler( string argument, string workDirPath )
			: base( argument, workDirPath )
		{}
		#endregion

		#region Properties
		/// <summary>
		/// コンパイラの実行ファイル名を取得します。
		/// </summary>
		public override string ExecutableName
		{
			get{ return "csc.exe"; }
		}

		/// <summary>
		/// 直前のコンパイルが成功したかどうかを示す値を取得します。
		/// </summary>
		public override bool Succeeded
		{
			get{ return (0 == _Results.Count); }
		}

		/// <summary>
		/// 直前のコンパイルが失敗したかどうかを示す値を取得します。
		/// </summary>
		public override bool Failed
		{
			get{ return (0 < _Results.Count); }
		}
		#endregion

		#region Interface
		/// <summary>
		/// コンパイルを実行します。
		/// </summary>
		/// <exception cref="System.IO.FileNotFoundException">コンパイラが見つからない</exception>
		/// <exception cref="System.ComponentModel.Win32Exception">コンパイラが見つからない</exception>
		public override void Compile()
		{
			Process process = new Process();
			ProcessStartInfo pInfo;

			try
			{
				// 起動するプロセスを設定
				pInfo = new ProcessStartInfo();
				pInfo.FileName = "csc.exe";
				pInfo.WorkingDirectory = _WorkDirPath;
				pInfo.UseShellExecute = false;
				pInfo.RedirectStandardOutput = true;
				pInfo.WindowStyle = ProcessWindowStyle.Hidden;
				pInfo.CreateNoWindow = true;
				pInfo.Arguments = "-nologo " + _Argument;
				
				// プロセスを起動
				process.StartInfo = pInfo;
				process.Start();

				// 標準出力の取得を開始
				_Log = process.StandardOutput.ReadToEnd();

				// プロセス終了まで待つ
				process.WaitForExit();

				// ログを解析
				_Results.Clear();
				string[] lines = Utl.SplitToLine( _Log );
				foreach( string line in lines )
				{
					ResultItem result = ParseErrorLine( line );
					if( result != null )
					{
						_Results.Add( result );
					}
				}

				// 返値を記録
				base._ExitCode = process.ExitCode;
			}
			catch( Win32Exception ex )
			{
				const int ERROR_FILE_NOT_FOUND = 2;
				if( ex.ErrorCode == ERROR_FILE_NOT_FOUND )
					throw new FileNotFoundException( ex.Message, ex );
				else
					throw ex;
			}
			finally
			{
				process.Dispose();
			}
		}
		#endregion

		#region Parse Logic
		/// <summary>
		/// ログの一行を解析
		/// </summary>
		// 例（空行が最後に入る）：
		// ----
		// warning CS1607: アセンブリの生成 -- 参照アセンブリ 'System.Data.dll' は異なるプロセッサを対象にしています。
		// Sgry\AutoSpeaker.cs(53,9): warning CS0168: 変数 'i' は宣言されていますが、使用されませんでした。
		// 
		// ----
		static ResultItem ParseErrorLine( string logLine )
		{
			Match match;
			string filePath;
			string message;
			int lineIndex, columnIndex;
			ResultType type;
			
			// 位置を特定可能なエラーか判定
			match = Regex.Match( logLine, @"([^(]+)\((\d+),(\d+)\): (warning|error) CS\d{4}: (.*)" );
			if( match.Success )
			{
				filePath = match.Groups[1].ToString();
				Int32.TryParse( match.Groups[2].ToString(), out lineIndex );
				Int32.TryParse( match.Groups[3].ToString(), out columnIndex );
				type = (match.Groups[4].ToString() == "error") ? ResultType.Error : ResultType.Warning;
				message = match.Groups[5].ToString().TrimStart();
				return new ResultItem( message, filePath, lineIndex, columnIndex, type );
			}

			// 位置を特定不能なエラーか判定
			match = Regex.Match( logLine, @"(warning|error) CS\d{4}: (.*)" );
			if( match.Success )
			{
				type = (match.Groups[1].ToString() == "error") ? ResultType.Error : ResultType.Warning;
				message = match.Groups[2].ToString().TrimStart();
				return new ResultItem( message, "", 0, 0, type );
			}

			return null;
		}
		#endregion
	}
}

/*
= エラーメッセージのサンプル =
== 通常のコンパイルエラー ==
AssemblyInfo.cs(6,1): error CS0116:
        名前空間にフィールドやメソッドのようなメンバを直接含めることはできません
----
ErrorListForm.cs(308,25): error CS0246: 型または名前空間名 'AiBListView'
        が見つかりませんでした。using
        ディレクティブまたはアセンブリ参照が不足しています。
----
AssemblyInfo.cs(9,27): error CS1010: 定数の 新しい行です。
----
Sgry\AutoSpeaker.cs(53,21): error CS0117: 'Sgry.JawsSpeaker' に 'IsAlivez'
        の定義がありません。

== 警告 ==
Sgry\AutoSpeaker.cs(53,9): warning CS0168: 変数 'i'
        は宣言されていますが、使用されませんでした。

== 位置を特定できないエラー ==
error CS1569: XML ドキュメント ファイル
        'c:\work\dev\AiBTools\package\AiBTools.CommonUtl.xml'
        を生成中にエラーが発生しました ('アクセスが拒否されました。 ')。

== コンパイラ実行のエラー ==
warning CS1607: アセンブリの生成 -- 参照アセンブリ 'System.Data.dll'
        は異なるプロセッサを対象にしています。
----
error CS1672: /platform に対する無効なオプション 'x886'
        です。anycpu、x86、Itanium または x64 でなければなりません。
----
fatal error CS2007: 認識できないオプションです: '-ooo'

----
*/
