// file: Compiler.cs
// brief: common interface of abstract compilers.
// encoding: UTF-8
// update: 2008-12-21
//=========================================================
using System;
using System.Collections.Generic;
using System.Text;

namespace Sgry.AiBTools.Compilation
{
	/// <summary>
	/// 抽象化したコンパイラの共通インタフェース。
	/// </summary>
	public abstract class Compiler
	{
		#region Fields
		/// <summary>コンパイラに与える引数</summary>
		protected string			_Argument = "";

		/// <summary>コンパイラの出力</summary>
		protected string			_Log;

		/// <summary>コンパイラを起動するディレクトリ</summary>
		protected string			_WorkDirPath;

		/// <summary>コンパイラの終了コード</summary>
		protected int				_ExitCode;

		/// <summary>解析したコンパイル結果のリスト</summary>
		protected List<ResultItem>	_Results = new List<ResultItem>();
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		/// <param name="argument">コンパイラに渡す引数</param>
		/// <param name="workDirPath">コンパイラを起動するディレクトリ</param>
		public Compiler( string argument, string workDirPath )
		{
			_Argument = argument;
			_WorkDirPath = workDirPath;
		}
		#endregion

		#region Properties
		/// <summary>
		/// コンパイラの実行ファイル名を取得します。
		/// </summary>
		public abstract string ExecutableName
		{
			get;
		}

		/// <summary>
		/// コンパイラの出力したログを取得します。
		/// </summary>
		public string Log
		{
			get{ return _Log; }
		}

		/// <summary>
		/// コンパイラが返した返値を取得します。
		/// </summary>
		public int ExitCode
		{
			get{ return _ExitCode; }
		}

		/// <summary>
		/// コンパイルするプロジェクトのディレクトリパスを取得または設定します。
		/// </summary>
		public string ProjectDirPath
		{
			get{ return _WorkDirPath; }
		}

		/// <summary>
		/// コンパイル結果を取得します。
		/// </summary>
		public List<ResultItem> Results
		{
			get{ return _Results; }
		}

		/// <summary>
		/// 直前のコンパイルが成功したかどうかを示す値を取得します。
		/// </summary>
		public abstract bool Succeeded
		{
			get;
		}

		/// <summary>
		/// 直前のコンパイルが失敗したかどうかを示す値を取得します。
		/// </summary>
		public abstract bool Failed
		{
			get;
		}
		#endregion

		#region Interface
		/// <summary>
		/// コンパイルを実行します。
		/// </summary>
		/// <returns>コンパイラプログラムの返値</returns>
		public abstract void Compile();
		#endregion

		#region Utilities
		/// <summary>
		/// Utilities for parsing log.
		/// </summary>
		protected static class Utl
		{
			/// <summary>
			/// 文字列を行ごとに分割します。
			/// </summary>
			public static string[] SplitToLine( string str )
			{
				StringBuilder line = new StringBuilder();
				List<string> lines = new List<string>();
				char ch;

				for( int i=0; i<str.Length; i++ )
				{
					ch = str[i];
					if( ch == '\r' )
					{
						if( i+1 < str.Length && str[i+1] == '\n' )
						{
							i++;
							lines.Add( line.ToString() );
							line.Length = 0;
						}
						else
						{
							lines.Add( line.ToString() );
							line.Length = 0;
						}
					}
					else if( ch == '\n' )
					{
						lines.Add( line.ToString() );
						line.Length = 0;
					}
					else
					{
						line.Append( ch );
					}
				}

				return lines.ToArray();
			}
		}
		#endregion
	}
}
