// file: TreeViewReader.cs
// brief: internal screen reader for System.Windows.Forms.TreeView
// encoding: UTF-8
// update: 2010-01-24
//=========================================================
using System;
using System.Collections;
using System.Text;
using System.Windows.Forms;
using CultureInfo = System.Globalization.CultureInfo;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// ツリービューを音声読み上げする内部スクリーンリーダー
	/// </summary>
	public class TreeViewReader : IControlReader<TreeView>
	{
		#region Fields
		static string Msg_OnExpand = "open {0} {1} items";
		static string Msg_OnCollapse = "close {0} {1} in {2}";
		static string Msg_OnSiblingSelected = "{0} {1} in {2}";
		static string Msg_OnSiblingParentSelected = "{0} {1} in {2} {3}";
		static string Msg_OnNonSiblingSelected = "level {3} {0} {1} in {2}";
		static string Msg_OnNonSiblingParentSelected = "level {3} {0} {1} in {2} {4}";
		static string Msg_Collapsed = "close";
		static string Msg_Expanded = "open";
		ArrayList _Controls = new ArrayList();
		Hashtable _PrevSelectedNodes = new Hashtable();
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成
		/// </summary>
		public TreeViewReader()
		{
			// localize
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				Msg_OnExpand = "オープン {0} {1}項目";
				Msg_OnCollapse = "クローズ {0} {2}の{1}";
				Msg_OnSiblingSelected = "{0} {2}の{1}";
				Msg_OnSiblingParentSelected = "{0} {2}の{1} {3}";
				Msg_OnNonSiblingSelected = "レベル{3} {0} {2}の{1}";
				Msg_OnNonSiblingParentSelected = "レベル{3} {0} {2}の{1} {4}";
				Msg_Expanded = "オープン";
				Msg_Collapsed = "クローズ";
			}
		}

		/// <summary>
		/// リソースを解放
		/// </summary>
		public void Dispose()
		{
			Clear();
		}
		#endregion

		#region IControlReader Interface
		/// <summary>
		/// 読み上げ対象のコントロールを追加
		/// </summary>
		public void Add( TreeView control )
		{
			control.AfterExpand += new TreeViewEventHandler( Target_AfterExpand );
			control.AfterCollapse += new TreeViewEventHandler( Target_AfterCollapse );
			control.AfterSelect += new TreeViewEventHandler( Target_AfterSelect );
			
			_Controls.Add( control );
		}

		/// <summary>
		/// 読み上げ対象をすべて登録解除
		/// </summary>
		public void Clear()
		{
			foreach( TreeView tree in _Controls )
			{
				tree.AfterExpand -= new TreeViewEventHandler( Target_AfterExpand );
				tree.AfterCollapse -= new TreeViewEventHandler( Target_AfterCollapse );
				tree.AfterSelect -= new TreeViewEventHandler( Target_AfterSelect );
			}
			_PrevSelectedNodes.Clear();
		}
		#endregion

		#region Reading logic
		void Target_AfterCollapse( object sender, TreeViewEventArgs e )
		{
			TreeView tree = (TreeView)sender;
			String speechText;
			int siblingCount;

			// フォーカスが当たっていないものの場合、読み上げは行わない
			if( tree.Focused != true )
			{
				return;
			}

			// 同列ノードが何個あるか計算
			siblingCount = Utl.CountSiblings( e.Node );

			// 読み上げ
			speechText = String.Format( Msg_OnCollapse, e.Node.Text, e.Node.Index+1, siblingCount );
			AutoSpeaker.Instance.Stop();
			AutoSpeaker.Instance.Speak( speechText );
		}

		void Target_AfterExpand( object sender, TreeViewEventArgs e )
		{
			TreeView tree = (TreeView)sender;
			String speechText;

			if( tree.Focused != true )
				return;

			speechText = String.Format( Msg_OnExpand, e.Node.Text, e.Node.GetNodeCount(false) );

			AutoSpeaker.Instance.Stop();
			AutoSpeaker.Instance.Speak( speechText );
		}
		
		void Target_AfterSelect( object sender, TreeViewEventArgs e )
		{
			TreeView tree = (TreeView)sender;
			TreeNode prevSelNode;
			
			// 前回選択されていたノードの深度に応じて読み上げを変更
			prevSelNode = _PrevSelectedNodes[tree] as TreeNode;
			if( prevSelNode != null
				&& prevSelNode.Level != e.Node.Level )
			{
				// 異なる深度のノードに選択が移動した。深度を読み上げる。
				OnNonSiblingSelected( tree, e );
			}
			else
			{
				OnSiblingSelected( tree, e );
			}

			// 選択ノードを記憶
			_PrevSelectedNodes[tree] = e.Node;
		}

		void OnNonSiblingSelected( TreeView tree, TreeViewEventArgs e )
		{
			String speechText;
			int siblingCount, childCount;

			// フォーカスが当たっていないものの場合、読み上げは行わない
			if( tree.Focused != true )
			{
				return;
			}

			// 同列ノードが何個あるか、子ノードが何個あるか取得
			siblingCount = Utl.CountSiblings( e.Node );
			childCount = e.Node.GetNodeCount( false );
			
			// 子ノードの有無に応じて読み上げ
			if( 0 < childCount )
			{
				string expandState = e.Node.IsExpanded ? Msg_Expanded : Msg_Collapsed;
				speechText = String.Format( Msg_OnNonSiblingParentSelected, e.Node.Text, e.Node.Index+1, siblingCount, e.Node.Level, expandState );
			}
			else
			{
				speechText = String.Format( Msg_OnNonSiblingSelected, e.Node.Text, e.Node.Index+1, siblingCount, e.Node.Level );
			}
			AutoSpeaker.Instance.Stop();
			AutoSpeaker.Instance.Speak( speechText );
		}

		void OnSiblingSelected( TreeView tree, TreeViewEventArgs e )
		{
			String speechText;
			int siblingCount, childCount;

			// フォーカスが当たっていないものの場合、読み上げは行わない
			if( tree.Focused != true )
			{
				return;
			}

			// 同列ノードが何個あるか、子ノードが何個あるか取得
			siblingCount = Utl.CountSiblings( e.Node );
			childCount = e.Node.GetNodeCount( false );
			
			// 子ノードの有無に応じて読み上げ
			if( 0 < childCount )
			{
				string expandState = e.Node.IsExpanded ? Msg_Expanded : Msg_Collapsed;
				speechText = String.Format( Msg_OnSiblingParentSelected, e.Node.Text, e.Node.Index+1, siblingCount, expandState );
			}
			else
			{
				speechText = String.Format( Msg_OnSiblingSelected, e.Node.Text, e.Node.Index+1, siblingCount );
			}
			AutoSpeaker.Instance.Stop();
			AutoSpeaker.Instance.Speak( speechText );
		}
		#endregion

		#region Utilities
		class Utl
		{
			public static int CountSiblings( TreeNode node )
			{
				int siblingCount;
				TreeNode n;

				// 親ノードがあれば、親ノードのプロパティを参照
				if( node.Parent != null )
				{
					return node.Parent.GetNodeCount( false );
				}

				// 親ノードが無いので手動でたぐり、末弟のインデックスを取得
				n = node;
				do
				{
					siblingCount = n.Index;
					n = n.NextNode;
				}
				while( n != null );

				return siblingCount + 1;
			}
		}
		#endregion
	}
}
