﻿// file: TextBoxReader.cs
// brief: control reader for TextBox
// encoding: UTF-8
// update: 2008-09-14
//=========================================================
using System;
using System.Collections;
using System.Windows.Forms;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// テキストボックスを音声読み上げする内部スクリーンリーダー
	/// </summary>
	// XPReader は MSAA の通知に対応していないのか？イベントに反応しないので、
	// このクラスでイベントが発生するタイミングで外部制御する
	public class TextBoxReader : IControlReader<TextBox>
	{
		ArrayList _Controls = new ArrayList();
		EditUtl _EditUtl = new EditUtl();

		#region Init / Dispose
		/// <summary>
		/// リソースを解放
		/// </summary>
		public void Dispose()
		{
			Clear();
		}
		#endregion

		#region IControlReader interface
		/// <summary>
		/// 読み上げ対象のコントロールを追加
		/// </summary>
		public void Add( TextBox control )
		{
			control.KeyDown += Target_KeyDown;

			_Controls.Add( control );
		}

		/// <summary>
		/// 読み上げ対象をすべて登録解除
		/// </summary>
		public void Clear()
		{
			foreach( TextBox textBox in _Controls )
			{
				textBox.KeyDown -= Target_KeyDown;
			}
			_Controls.Clear();
		}
		#endregion

		#region Event handlers for target controls
		void Target_KeyDown( object sender, KeyEventArgs e )
		{
			TextBox textBox = (TextBox)sender;
			if( textBox.Focused != true )
				return;

			if( XpReaderSpeaker.IsAlive() == false )
				return;

			try
			{
				ISpeaker speaker = AutoSpeaker.Instance;
				string speechText = "";

				// 押されたキーに応じた文面を生成
				if( e.KeyCode == Keys.Right )
				{
					speechText = CreateSpeechTextOnRightKey( textBox, e );
				}
				else if( e.KeyCode == Keys.Left )
				{
					speechText = CreateSpeechTextOnLeftKey( textBox, e );
				}

				// 生成した文面を読み上げる
				speaker.Stop();
				speaker.Speak( speechText );
			}
			catch // （中身の実装が甘いのでまれに例外が飛ぶ。手抜きで全部キャッチ。）
			{}
		}

		string CreateSpeechTextOnRightKey( TextBox textBox, KeyEventArgs e )
		{
			string speechText = String.Empty;

			if( !e.Shift && !e.Control )
			{
				speechText = textBox.Text[ textBox.SelectionStart+1 ].ToString();
			}
			else if( e.Shift && !e.Control )
			{
				int caretIndex = _EditUtl.GetCaretIndex( textBox );
				if( _EditUtl.IsSelectingToEnd(textBox) )
				{
					speechText = textBox.Text[ caretIndex ].ToString();
				}
				else
				{
					speechText = textBox.Text[ caretIndex + 1 ].ToString();
				}
			}
			//else if( !e.Shift && !e.Control ){}

			return speechText;
		}

		string CreateSpeechTextOnLeftKey( TextBox textBox, KeyEventArgs e )
		{
			string speechText = String.Empty;

			if( !e.Shift && !e.Control )
			{
				int caretIndex = _EditUtl.GetCaretIndex( textBox );
				speechText = textBox.Text[ caretIndex - 1 ].ToString();
			}
			else if( e.Shift && !e.Control )
			{
				int caretIndex = _EditUtl.GetCaretIndex( textBox );
				if( 0 < textBox.SelectionLength
					&& _EditUtl.IsSelectingToEnd(textBox) )
				{
					speechText = textBox.Text[ caretIndex - 2 ].ToString();
				}
				else
				{
					speechText = textBox.Text[ caretIndex - 1 ].ToString();
				}
			}
			//else if( !e.Shift && e.Control ){}

			return speechText;
		}
		#endregion
	}
}
