﻿// file: CheckBoxReader.cs
// brief: internal screen reader for System.Windows.Forms.CheckBox
// encoding: UTF-8
// update: 2008-09-14
//=========================================================
using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// チェックボックスを音声読み上げする内部スクリーンリーダー
	/// </summary>
	public class CheckBoxReader : IControlReader<CheckBox>
	{
		List<CheckBox> _Controls = new List<CheckBox>();

		#region Init / Dispose
		/// <summary>
		/// リソースを解放
		/// </summary>
		public void Dispose()
		{
			Clear();
		}
		#endregion

		#region IControlReader interface
		/// <summary>
		/// 読み上げ対象のコントロールを追加。
		/// </summary>
		public void Add( CheckBox control )
		{
			control.CheckedChanged += Target_CheckedChanged;
			
			_Controls.Add( control );
		}

		/// <summary>
		/// 読み上げ対象をすべて登録解除。
		/// </summary>
		public void Clear()
		{
			foreach( CheckBox checkBox in _Controls )
			{
				checkBox.CheckedChanged -= Target_CheckedChanged;
			}
			_Controls.Clear();
		}
		#endregion

		#region Event handlers for target controls
		void Target_CheckedChanged( object sender, EventArgs e )
		{
			CheckBox control = (CheckBox)sender;
			if( control.Focused != true )
				return;

			if( XpReaderSpeaker.IsAlive() )
			{
				ISpeaker speaker;
				string label;
				string mnemonic;
				
				speaker = XpReaderSpeaker.Instance;
				Utl.ParseMnemonic( control.Text, out label, out mnemonic );

				speaker.Stop();

				if( control.Checked )
					speaker.Speak( "チェック" );
				else
					speaker.Speak( "チェック無し" );
				
				speaker.Wait();
				speaker.Speak( label );
			}
		}
		#endregion

		#region Utilities
		class Utl
		{
			/// <summary>
			/// ラベル文字列をラベルとニーモニックに分離
			/// </summary>
			public static void ParseMnemonic( string text, out string label, out string mnemonic )
			{
				Match match;

				// 末尾が (&?) ？
				match = Regex.Match( text, @"\(&(.)\)$" );
				if( match.Success )
				{
					label = text.Substring( 0, text.Length - 4 );
					mnemonic = match.Groups[1].ToString();

					return;
				}

				// 単独の & ？
				match = Regex.Match( text, "&([^&])" );
				if( match.Success )
				{
					int mnemonicPos = match.Groups[1].Index;
					label = text.Substring( 0, mnemonicPos )
						+ text.Substring( mnemonicPos + 1, text.Length - mnemonicPos - 1 );
					mnemonic = match.Groups[1].ToString();

					return;
				}

				label = text;
				mnemonic = null;
			}
		}
		#endregion
	}
}
