﻿// file: AzukiEditReader.cs
// brief: internal screen reader for Sgry.AiBTools.AzukiEdit
// encoding: UTF-8
// update: 2008-10-05
//=========================================================
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using Sgry.Azuki;
using Debug = System.Diagnostics.Debug;
using Conditional = System.Diagnostics.ConditionalAttribute;

namespace Sgry.AiBTools.AT
{
	using Gui;

	/// <summary>
	/// Azuki を音声読み上げする内部スクリーンリーダー
	/// </summary>
	public class AzukiEditReader : IControlReader<AzukiEdit>
	{
		#region Fields
		List<AzukiEdit> _Controls = new List<AzukiEdit>( 4 );
		List<int> _PrevCaretIndexes = new List<int>( 4 );
		List<ActionProc> _PrevActions = new List<ActionProc>( 4 );
		AzukiEdit _Azuki;
		bool _SpeaksAfterCaretOnLineChange = true;
		bool _SpeaksChangedPartOnSelect = true;
		string _Msg_Eof, _Msg_Eol, _Msg_Hof;
		string _Msg_Selected, _Msg_Deselected, _Msg_EmptyLine, _Msg_WhiteSpaceLine, _Msg_LineHead, _Msg_LineEnd;
		Dictionary<char, string> _SpecialChars = new Dictionary<char,string>();
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成
		/// </summary>
		public AzukiEditReader()
		{
			Localizer localizer = new Localizer();
			localizer.LoadResourceFile( "CommonUtl" );

			_Msg_EmptyLine = localizer.TryGetString( "AzukiEditReader._Msg_EmptyLine", "line end" );
			_Msg_Eol = localizer.TryGetString( "AzukiEditReader._Msg_Eol", "line end" );
			_Msg_Eof = localizer.TryGetString( "AzukiEditReader._Msg_Eof", "file end" );
			_Msg_Hof = localizer.TryGetString( "AzukiEditReader._Msg_Hof", "file head" );
			_Msg_Selected = localizer.TryGetString( "AzukiEditReader._Msg_Selected", "selected" );
			_Msg_Deselected = localizer.TryGetString( "AzukiEditReader._Msg_Deselected", "deselected" );
			_Msg_EmptyLine = localizer.TryGetString( "AzukiEditReader._Msg_EmptyLine", "empty line" );
			_Msg_WhiteSpaceLine = localizer.TryGetString( "AzukiEditReader._Msg_WhiteSpaceLine", "white space line" );
			_Msg_LineHead = localizer.TryGetString( "AzukiEditReader._Msg_LineHead", "head" );
			_Msg_LineEnd = localizer.TryGetString( "AzukiEditReader._Msg_LineEnd", "end" );

			_SpecialChars['\r'] = _Msg_Eol;
			_SpecialChars['\n'] = _Msg_Eol;
			_SpecialChars['\x20'] = localizer.TryGetString( "AzukiEditReader._Msg_HwSpace", "space" ); // 半角スペース
			_SpecialChars['\x3000'] = localizer.TryGetString( "AzukiEditReader._Msg_FwSpace", "zen kaku space" ); // 全角スペース
		}

		/// <summary>
		/// リソースを解放
		/// </summary>
		public void Dispose()
		{
			Clear();
		}
		#endregion

		#region Type specific properties
		/// <summary>
		/// 選択変更時の読み上げ方法を取得または設定します。
		/// true の場合は選択変更時に選択状態が変更された部分を読み上げます。
		/// false にした場合は通常通りカーソル位置の文字を読み上げます。
		/// </summary>
		public bool SpeaksChangedPartOnSelect
		{
			get{ return _SpeaksChangedPartOnSelect; }
			set{ _SpeaksChangedPartOnSelect = value; }
		}

		/// <summary>
		/// キャレットを上下に移動した際に、キャレット以降を読み上げるかどうか。
		/// false にした場合は新たにキャレットが進入した行全体を読み上げる。
		/// </summary>
		public bool SpeaksAfterCaretOnLineChange
		{
			get{ return _SpeaksAfterCaretOnLineChange; }
			set{ _SpeaksAfterCaretOnLineChange = value; }
		}
		#endregion

		#region IControlReader Interface
		/// <summary>
		/// 読み上げ対象のコントロールを追加。
		/// </summary>
		public void Add( AzukiEdit control )
		{
			_Controls.Add( control );
			_PrevCaretIndexes.Add( control.GetCaretIndex() );
			_PrevActions.Add( null );
			control.PreviewKeyDown += Control_PreviewKeyDown;
			control.GotFocus += Control_GotFocus;
			control.Leave += Control_Leave;
			control.SelectionChanged += Control_SelectionChanged;
		}

		/// <summary>
		/// 読み上げ対象のコントロールを一つ除外。
		/// </summary>
		public void Remove( AzukiEdit control )
		{
			int index = _Controls.IndexOf( control );
			control.PreviewKeyDown -= Control_PreviewKeyDown;
			control.GotFocus -= Control_GotFocus;
			control.Leave -= Control_Leave;
			control.SelectionChanged -= Control_SelectionChanged;
			_Controls.RemoveAt( index );
			_PrevCaretIndexes.RemoveAt( index );
			_PrevActions.RemoveAt( index );
			if( _Azuki == control )
			{
				_Azuki = null;
			}
		}

		/// <summary>
		/// 読み上げ対象をすべて登録解除。
		/// </summary>
		public void Clear()
		{
			for( int i=0; i<_Controls.Count; i++ )
			{
				Remove( _Controls[ _Controls.Count-1 ] );
			}
		}
		#endregion

		#region Event handlers for target control
		void Control_GotFocus( object sender, EventArgs e )
		{
			Debug.Assert( _Controls.Contains((AzukiEdit)sender) );
			_Azuki = (AzukiEdit)sender;
		}

		void Control_Leave( object sender, EventArgs e )
		{
			Debug.Assert( _Controls.Contains((AzukiEdit)sender) );
			_Azuki = null;
		}

		void Control_SelectionChanged( object sender, EventArgs e )
		{
			AzukiEdit azuki = (AzukiEdit)sender;
			int index = _Controls.IndexOf( azuki );
			int prevCaretIndex = _PrevCaretIndexes[ index ];
			ActionProc action = _PrevActions[ index ];

			if( action != null )
			{
				DoSpeak( azuki, action, prevCaretIndex );
			}

			_PrevCaretIndexes[ index ] = azuki.CaretIndex;
		}

		void Control_PreviewKeyDown( object sender, PreviewKeyDownEventArgs e )
		{
			AzukiEdit azuki = (AzukiEdit)sender;
			int index = _Controls.IndexOf( azuki );
			_PrevActions[ index ] = _Azuki.GetKeyBind( e.KeyData );
		}
		#endregion

		#region Reading logic (these do not stop AutoSpeaker)
		void DoSpeak( AzukiEdit azuki, ActionProc action, int prevCaretIndex )
		{
			if( action == Actions.MoveToNextWord
				|| action == Actions.MoveToPrevWord )
			{
				AutoSpeaker.Instance.Stop();
				ReadWordAtCaret();
			}
			else if( action == Actions.MoveRight
				|| action == Actions.MoveLeft )
			{
				AutoSpeaker.Instance.Stop();
				ReadCharAtCaret();
			}
			else if( action == Actions.MoveUp
				|| action == Actions.MoveDown )
			{
				AutoSpeaker.Instance.Stop();
				ReadLineAtCaret();
			}
			else if( action == Actions.MoveToLineHead )
			{
				AutoSpeaker.Instance.Stop();
				ReadWordAtCaret();
				AutoSpeaker.Instance.Speak( _Msg_LineHead );
			}
			else if( action == Actions.MoveToLineHeadSmart )
			{
				AutoSpeaker.Instance.Stop();
				if( _Azuki.GetLineHeadIndexFromCharIndex(_Azuki.CaretIndex) < _Azuki.CaretIndex )
				{
Win32.Beep( 400, 50 );
				}
				ReadWordAtCaret();
				AutoSpeaker.Instance.Speak( _Msg_LineHead );
			}
			else if( action == Actions.MoveToLineEnd )
			{
				AutoSpeaker.Instance.Stop();
				AutoSpeaker.Instance.Speak( _Msg_LineEnd );
				ReadCharAtCaret();
			}
			else if( action == Actions.MoveToFileHead )
			{
				AutoSpeaker.Instance.Stop();
				AutoSpeaker.Instance.Speak( _Msg_Hof );
			}
			else if( action == Actions.MoveToFileEnd )
			{
				AutoSpeaker.Instance.Stop();
				AutoSpeaker.Instance.Speak( _Msg_Eof );
			}
			else if( action == Actions.SelectToRight )
			{
				AutoSpeaker.Instance.Stop();

				// 選択・選択解除の対象を読み上げ
				if( SpeaksChangedPartOnSelect )
					ReadCharBeforeCaret();
				else
					ReadCharAtCaret();

				// 選択を拡張したのか縮小したのか読み上げる
				if( _Azuki.SelectingToEnd )
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				else
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
			}
			else if( action == Actions.SelectToLeft )
			{
				AutoSpeaker.Instance.Stop();

				// 選択・選択解除の対象を読み上げ
				ReadCharAtCaret();

				// 選択を拡張したのか縮小したのか読み上げる
				if( _Azuki.SelectingToEnd )
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
				else
					AutoSpeaker.Instance.Speak( _Msg_Selected );
			}
			else if( action == Actions.SelectToNextWord )
			{
				OnSelectToNextWord( azuki, prevCaretIndex );
			}
			else if( action == Actions.SelectToPrevWord )
			{
				OnSelectToPrevWord( azuki, prevCaretIndex );
			}
		}
		
		void OnSelectToNextWord( AzukiEdit azuki, int prevCaretIndex )
		{
			if( azuki.CaretIndex < prevCaretIndex )
				return;

			AutoSpeaker.Instance.Stop();

			if( SpeaksChangedPartOnSelect == false )
			{
				ReadWordAtCaret();
				if( _Azuki.SelectingToEnd )
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				else
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
			}
			else
			{
				string word;
				int anchor = azuki.SelectionAnchor;
				int caret = azuki.CaretIndex;

				// 選択状態の変更に応じて読み上げ
				if( prevCaretIndex < caret && caret < anchor )
				{
					// 先頭方向への選択が縮小。
					// 選択解除部分を読み上げ
					word = _Azuki.GetTextInRange( prevCaretIndex, caret );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
				}
				else if( prevCaretIndex < anchor && anchor < caret )
				{
					// 選択方向が反転。
					// 選択解除部分を読み上げ
					word = _Azuki.GetTextInRange( prevCaretIndex, anchor );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
					AutoSpeaker.Instance.Wait();

					// 新規選択部分を読み上げ
					word = _Azuki.GetTextInRange( anchor, caret );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				}
				else if( anchor <= prevCaretIndex && prevCaretIndex <= caret )
				{
					// 末尾方向に選択が拡張。
					// 新規選択部分を読み上げ
					word = _Azuki.GetTextInRange( prevCaretIndex, caret );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				}
			}
		}

		void OnSelectToPrevWord( AzukiEdit azuki, int prevCaretIndex )
		{
			if( prevCaretIndex < azuki.CaretIndex )
				return;

			AutoSpeaker.Instance.Stop();

			if( SpeaksChangedPartOnSelect == false )
			{
				ReadWordAtCaret();
				if( _Azuki.SelectingToEnd )
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
				else
					AutoSpeaker.Instance.Speak( _Msg_Selected );
			}
			else
			{
				string word;
				int anchor = azuki.SelectionAnchor;
				int caret = azuki.CaretIndex;

				// 選択状態の変更に応じて読み上げ
				if( anchor < caret && caret < prevCaretIndex )
				{
					// 末尾方向への選択が縮小。
					// 選択解除部分を読み上げ
					word = _Azuki.GetTextInRange( caret, prevCaretIndex );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
				}
				else if( caret < anchor && anchor < prevCaretIndex )
				{
					// 選択方向が反転。
					// 選択解除部分を読み上げ
					word = _Azuki.GetTextInRange( anchor, prevCaretIndex );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Deselected );
					AutoSpeaker.Instance.Wait();

					// 新規選択部分を読み上げ
					word = _Azuki.GetTextInRange( caret, anchor );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				}
				else if( caret <= prevCaretIndex && prevCaretIndex <= anchor )
				{
					// 先頭方向に選択が拡張。
					// 新規選択部分を読み上げ
					word = _Azuki.GetTextInRange( caret, prevCaretIndex );
					AutoSpeaker.Instance.Speak( word );
					AutoSpeaker.Instance.Speak( _Msg_Selected );
				}
			}
		}

		void ReadCharAtCaret()
		{
			Debug.Assert( _Azuki != null );
			char ch;
			string message;

			// get caret position
			int caretIndex = _Azuki.GetCaretIndex();
			if( _Azuki.TextLength <= caretIndex )
			{
				// caret is at the EOF
				AutoSpeaker.Instance.Speak( _Msg_Eof );
				return;
			}

			// get char which the caret is on
			ch = _Azuki.GetCharAt( caretIndex );
			message = ch.ToString();
			if( _SpecialChars.ContainsKey(ch) )
			{
				message = _SpecialChars[ch];
			}

			// speak
			AutoSpeaker.Instance.Speak( message );
		}

		void ReadCharBeforeCaret()
		{
			Debug.Assert( _Azuki != null );

			// get caret position
			int caretIndex = _Azuki.GetCaretIndex();
			if( 0 == caretIndex )
			{
				return;
			}

			// get char which the caret is on
			char ch = _Azuki.GetCharAt( caretIndex-1 );

			// speak
			AutoSpeaker.Instance.Speak( ch.ToString() );
		}

		void ReadWordAtCaret()
		{
			Debug.Assert( _Azuki != null );

			string message = null;
			int caretIndex = _Azuki.GetCaretIndex();

			// try get word at caret
			message = _Azuki.GetWordAt( caretIndex );
			if( message == String.Empty )
			{
				return;
			}

			// speak
			AutoSpeaker.Instance.Speak( message );
		}

		void ReadLineAtCaret()
		{
			Debug.Assert( _Azuki != null );

			int caretIndex;
			int caretLineIndex;
			string lineContent;
			string message;

			// get line content
			caretIndex = _Azuki.GetCaretIndex();
			caretLineIndex = _Azuki.GetLineIndexFromCharIndex( caretIndex );
			lineContent = _Azuki.GetLineContent( caretLineIndex );
			if( lineContent == String.Empty )
			{
				message = _Msg_EmptyLine;
			}
			else if( lineContent.Trim() == String.Empty )
			{
				message = _Msg_WhiteSpaceLine;
			}
			else if( SpeaksAfterCaretOnLineChange )
			{
				int lineHeadIndex = _Azuki.GetLineHeadIndex( caretLineIndex );
				int caretIndexInLine = caretIndex - lineHeadIndex;
				string latterPart = lineContent.Substring( caretIndexInLine );
				message = latterPart;
			}
			else
			{
				message = lineContent;
			}

			AutoSpeaker.Instance.Speak( message );
		}
		#endregion
	}
}
