// file: NbsMessageBox.cs
// brief: message boxes which can use NbsEngine
// encoding: UTF-8
// update: 2008-09-14
//=========================================================
using System;
using System.Windows.Forms;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// NBS エンジンを利用した点字表示を行うメッセージボックス
	/// </summary>
	public static class NbsMessageBox
	{
		#region Show Methods
		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="message">表示するメッセージ</param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( string message )
		{
			return Show( null, Application.ProductName, message, MessageBoxButtons.OK, MessageBoxIcon.None );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="message">表示するメッセージ</param>
		/// <param name="buttons">メッセージボックスに表示するボタンのパターン</param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( string message, MessageBoxButtons buttons )
		{
			return Show( null, Application.ProductName, message, buttons, MessageBoxIcon.None );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="message">表示するメッセージ</param>
		/// <param name="buttons">メッセージボックスに表示するボタンのパターン</param>
		/// <param name="icon">
		/// メッセージボックスの種類を指定します。
		/// アイコンのグラフィックは表示されませんが、発生するメッセージ音がこの引数によって変化します。
		/// </param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( string message, MessageBoxButtons buttons, MessageBoxIcon icon )
		{
			return Show( null, Application.ProductName, message, buttons, icon );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="owner">指定したウィンドウの前に、メッセージボックスが表示されます。</param>
		/// <param name="message">表示するメッセージ</param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( IWin32Window owner, string message )
		{
			return Show( owner, Application.ProductName, message, MessageBoxButtons.OK, MessageBoxIcon.None );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="owner">指定したウィンドウの前に、メッセージボックスが表示されます。</param>
		/// <param name="caption">メッセージボックスのキャプション</param>
		/// <param name="message">表示するメッセージ</param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( IWin32Window owner, string caption, string message )
		{
			return Show( owner, caption, message, MessageBoxButtons.OK, MessageBoxIcon.None );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="owner">指定したウィンドウの前に、メッセージボックスが表示されます。</param>
		/// <param name="message">表示するメッセージ</param>
		/// <param name="buttons">メッセージボックスに表示するボタンのパターン</param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( IWin32Window owner, string message, MessageBoxButtons buttons )
		{
			return Show( owner, Application.ProductName, message, buttons, MessageBoxIcon.None );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="owner">指定したウィンドウの前に、メッセージボックスが表示されます。</param>
		/// <param name="message">表示するメッセージ</param>
		/// <param name="buttons">メッセージボックスに表示するボタンのパターン</param>
		/// <param name="icon">
		/// メッセージボックスの種類を指定します。
		/// アイコンのグラフィックは表示されませんが、発生するメッセージ音がこの引数によって変化します。
		/// </param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( IWin32Window owner, string message, MessageBoxButtons buttons, MessageBoxIcon icon )
		{
			return Show( owner, Application.ProductName, message, buttons, icon );
		}

		/// <summary>
		/// 指定したテキストを表示するメッセージボックスを表示します。
		/// </summary>
		/// <param name="owner">指定したウィンドウの前に、メッセージボックスが表示されます。</param>
		/// <param name="caption">メッセージボックスのキャプション</param>
		/// <param name="message">表示するメッセージ</param>
		/// <param name="buttons">メッセージボックスに表示するボタンのパターン</param>
		/// <param name="icon">
		/// メッセージボックスの種類を指定します。
		/// アイコンのグラフィックは表示されませんが、発生するメッセージ音がこの引数によって変化します。
		/// </param>
		/// <returns>メッセージボックスに対するユーザの応答</returns>
		public static DialogResult Show( IWin32Window owner, string caption, string message, MessageBoxButtons buttons, MessageBoxIcon icon )
		{
			DialogResult result = DialogResult.Cancel;
			
			using( NbsMessageBoxForm form = new NbsMessageBoxForm(owner, caption, message, buttons, icon) )
			{
				result = form.ShowDialog( owner );
			}

			return result;
		}
		#endregion
	}
}

/*
= Form をカスタムしてダイアログを模擬した場合の読み上げ挙動] =
== 凡例 ==
・<OwnerTitle> ... 親ウィンドウのタイトル
・<Title> ... 模擬ダイアログのタイトル
・<Message> ... 模擬ダイアログの最初のラベル
・<AccName> ... MSAA の AccessibleName プロパティ
・<Button> ... 初期選択されているボタン名

== PC-Talker ==
　(表示直後)  <OwnerTitle> アクティブ メッセージ <Message> エンターは <Button>
　(他アプリから切り替え時) メッセージ <Message> エンターは <Button>

== XP Reader ==
　(表示直後)  <Title> オープン
　(他アプリから切り替え時) <Title>

== JAWS ==
　(表示直後)  <OwnerTitle> <Title> <Button> ボタン 選択するにはスペースキーを使用します。
　(他アプリから切り替え時) <OwnerTitle> <Title> <Button> ボタン 選択するにはスペースキーを使用します。

== 対策 ==
XP Reader は <Message> を読み上げてくれないので、
ダイアログがアクティブになったら問答無用で
	メッセージ <Title>
と発音させる。
PC-Talker の外部制御では
PC-Talker 本体の記号読み上げ設定等が反映されない。
JAWS も動作が怪しい（読み上げない時がまれにある）ので、
どのスクリーンリーダーでも外部制御で読み上げを実行する。

*/
