// file: NbsException.cs
// brief: Exceptions for NBS Engine.
// encoding: UTF-8
// update: 2008-11-09
//=========================================================
using System;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// NbsEngine で発生する例外の基底クラス
	/// </summary>
	public class NbsException : Exception
	{
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public NbsException()
		{}

		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public NbsException( string message )
			: base( message )
		{}
	}

	/// <summary>
	/// NBS エンジンの初期化に失敗
	/// </summary>
	public class AttachFailedException : NbsException
	{
		string _Message;

		/// <summary>
		/// 機種設定が不正な場合のエラー値。
		/// </summary>
		public const int InvalidDeviceConfigError = 0xfff1;

		/// <summary>
		/// サーバへの接続に失敗した場合のエラー値。
		/// </summary>
		public const int FailedToAttachToServerError = 0xfff2;

		/// <summary>
		/// 新しいインスタンスを初期化します。
		/// </summary>
		/// <param name="errorCode">PinCtrl.DLL から返ってきたエラーコード</param>
		public AttachFailedException( int errorCode )
		{
			Localizer localizer = new Localizer();
			localizer.LoadResourceFile( "CommonUtl" );
			
			switch( errorCode )
			{
				case 0x00001:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_PARAM_ERROR", "Invalid parameter." );
					break;
				case 0x00002:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_HWND_ERROR", "Handle of the owner window is invalid." );
					break;
				case 0x00003:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_WINDOW_ERROR", "Failed to attach owner window." );
					break;
				case 0x00004:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_PININIT_ERROR", "Failed to initialize." );
					break;
				case 0x10001:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_OPEN_ERROR", "Failed to open yomi-file." );
					break;
				case 0x10002:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_ALLOC_ERROR", "No enough memory." );
					break;
				case 0x10100:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_FILE_NORMAL", "Normal yomi-file is invalid." );
					break;
				case 0x10200:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_FILE_6TEN", "Six-dot yomi-file is invalid." );
					break;
				case 0x10300:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_FILE_6TENHUGOU", "Six-dot symbol-yomi-file is invalid." );
					break;
				case 0x10400:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_FILE_8TEN", "Eight-dot yomi-file is invalid." );
					break;
				case 0x10500:
					_Message = localizer.TryGetString( "NbsEngine.PINCTRL_YOMI_FILE_8TENHUGOU", "Eight-dot symbol-yomi-file is invalid." );
					break;
				case InvalidDeviceConfigError:
					_Message = localizer.TryGetString( "NbsEngine.InvalidDeviceConfigError", "Device configuration is invalid." );
					break;
				case FailedToAttachToServerError:
					_Message = localizer.TryGetString( "NbsEngine.FailedToAttachToServerError", "Failed to attach to braille device server." );
					break;
				default:
					_Message = localizer.TryGetString( "NbsEngine.UnknownError", "An error by unknown reason occured." );
					break;
			}
		}

		/// <summary>
		/// 新しいインスタンスを初期化します。
		/// </summary>
		public AttachFailedException( string message )
		{
			_Message = message;
		}

		/// <summary>
		/// 発生した例外を説明するメッセージを取得
		/// </summary>
		public override string Message
		{
			get{ return _Message; }
		}
	}
}
