﻿// file: NbsEngine.cs
// brief: abstraction layer for NBS engine
// encoding: UTF-8
//=========================================================
//#define IGNORE_ATTACH_ERROR
//#define TRACE_OUTPUT
//#define TRACE_WM
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Encoding = System.Text.Encoding;
using Encoder = System.Text.Encoder;
using System.Diagnostics;
using SysTimer = System.Timers.Timer;
using ElapsedEventArgs = System.Timers.ElapsedEventArgs;
using ElapsedEventHandler = System.Timers.ElapsedEventHandler;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// NBS エンジンを管理します。
	/// </summary>
	/// <remarks>
	/// まず、アプリケーションの起動直後（またはメインフォームのロード時）に
	/// NbsEngine.Init() を呼び、唯一のインスタンスを生成します。
	/// その後はアプリケーションから NbsEngine.Instance プロパティを利用して
	/// 唯一のインスタンスへアクセスします。
	/// また、最後にアプリケーションの終了直前（またはメインフォームが閉じた後）に
	/// NbsEngine.Dispose() で後処理を実行してください。
	/// Singleton デザインパターンにしている理由は、
	/// 将来的に複数ディスプレイ同時出力への対応や
	/// 別（or新バージョン）のエンジンへの対応を行う場合に
	/// ポリモーフィズムで楽に切り替え可能とするため。
	/// </remarks>
	public sealed class NbsEngine
	{
		#region Fields
		/// <summary>
		/// USBポートを表す特殊なCOMポート番号
		/// </summary>
		public const int UsbPortID = UInt16.MaxValue;
		const int TRUE = 1;
		const int FALSE = 2;
		static NbsEngine _Instance = null;
		static Version _Version = null;
		static bool _Installed;
		bool _Attached = false;
		bool _Locked = false;
		int _StatusWidth = 0;
		uint _CacheCount = 0;
		byte[] _CachedData;
		Int32 _CachedCursorStartPos;
		Int32 _CachedCursorEndPos;
		byte[] _CachedStatusNabccStr = null;
		Int32 _CachedOutMode;
		Int32 _CachedEndMode;

		DeviceConfig _DeviceConfig = null;
		BrailleConfig _BrailleConfig = new BrailleConfig();
		MessageSink _MessageSink;
#if DEBUG
		static bool _IsDisposed = false;
#endif
		#endregion

		#region Properties
		/// <summary>
		/// NbsEngine の唯一のインスタンスを取得
		/// </summary>
		public static NbsEngine Instance
		{
			get
			{
#				if DEBUG
				Debug.Assert( !_IsDisposed, "NbsEngine was already disposed." );
#				endif

				if( _Instance == null )
				{
					_Instance = new NbsEngine();
				}
				return _Instance;
			}
		}

		/// <summary>
		/// NBS エンジンがインストールされているかどうかを取得します。
		/// </summary>
		public static bool Installed
		{
			get
			{
				if( NbsEngine.Instance == null ) // インスタンス生成できない＝インストールされていない。改善したい構造だ。
				{
					return false;
				}
				return _Installed;
			}
		}

		/// <summary>
		/// 点字ディスプレイサーバと接続中である事を表すフラグ
		/// </summary>
		public bool IsAttached
		{
			get{ return _Attached; }
		}
		
		/// <summary>
		/// 点字ディスプレイ使用権をこのプロセスが取得しているかどうかを取得します。
		/// </summary>
		public bool IsLocked
		{
			get{ return _Locked; }
		}
		#endregion

		#region Init / Dispose
		/// <summary>
		/// NbsEngine オブジェクトを初期化
		/// </summary>
		NbsEngine()
		{
			Int32 verNum;

			_Installed = false;
			try
			{
				// バージョンを取得
				verNum = PinCtrlGetVersion();
				_Version = new Version( (verNum >> 16) & 0x0f, verNum & 0x0f );

				// 例外発生せずに取得できたらエンジンはインストール済み
				_Installed = true;

				// create message/event translation object
				_MessageSink = new MessageSink( this );
			}
			catch( DllNotFoundException )
			{}
		}

#		if DEBUG
		/// <summary>
		/// インスタンスが破棄されるときのアクション
		/// </summary>
		~NbsEngine()
		{
			Debug.Assert( _IsDisposed, "NbsEngine not disposed yet!" );
		}
#		endif

		/// <summary>
		/// NbsEngine のリソースを解放
		/// </summary>
		public static void Dispose()
		{
			if( _Instance != null )
			{
				_Instance.DetachFromServer();
				if( _Instance._MessageSink != null )
				{
					_Instance._MessageSink.Dispose();
					_Instance._MessageSink = null;
				}
				_Instance = null;
#				if DEBUG
				_IsDisposed = true;
#				endif
			}
		}
		#endregion // Init / Dispose

		#region NBSエンジンの使用準備と終了
		/// <summary>
		/// 使用中のコンピュータで利用可能なCOMポート番号を取得
		/// </summary>
		/// <returns>使用可能な COM ポート番号を数値として格納した配列。
		/// 使用可能なポートが無い場合は空の配列が返る。</returns>
		/// <exception cref="DllNotFoundException">NBS エンジンの PinCtrl.dll が見つからない</exception>
		/// <example>
		/// System.Console.WriteLine( "Available COM ports are:" );
		/// foreach( int comPortNum in NbsEngine.GetAvailableComPorts() )
		/// {
		///     System.Console.WriteLine( String.Format("    COM{0}",comPortNum) );
		/// }
		/// </example>
		public static int[] GetAvailableComPorts()
		{
			int		rc; // result code
			int		availableComPortCount;
			Int32[]	comPortNumberTable;
			
			if( !_Installed )
				return new Int32[0];

			unsafe
			{
				// 使用可能な COM ポートの数を取得
				availableComPortCount = PinCtrlEnumComPort( (Int32*)0, 0 );
				if( availableComPortCount == 0 )
				{
					return new Int32[0]; // ポートが存在しない
				}

				// 使用可能 COM ポートに何があるか取得
				comPortNumberTable = new Int32[ availableComPortCount ];
				fixed( Int32* p = comPortNumberTable )
				{
					rc = PinCtrlEnumComPort( p, availableComPortCount );
				}
			}
			if( rc < 0 )
			{
				return new Int32[0];
			}
			
			return comPortNumberTable;
		}

		/// <summary>
		/// 点字ディスプレイの機種設定を取得または設定します。
		/// </summary>
		public DeviceConfig DeviceConfig
		{
			get{ return _DeviceConfig; }
			set{ _DeviceConfig = value; }
		}

		/// <summary>
		/// 点字ディスプレイサーバに接続。
		/// </summary>
		/// <exception cref="AttachFailedException">サーバに接続できなかった</exception>
		public void AttachToServer()
		{
			if( !_Installed || _Attached || _DeviceConfig == null )
				return;

			int	rc; // result code

			try
			{
				if( _DeviceConfig.Device == BrailleDevice.None )
					throw new AttachFailedException( AttachFailedException.InvalidDeviceConfigError );

				// エンジンを初期化（ディスプレイサーバを起動）
				rc = PinCtrlInit( _MessageSink.Handle, _DeviceConfig.Device.ID,
						_BrailleConfig.CursorMode, _BrailleConfig.StatusWidth,
						_DeviceConfig.ComPort, _DeviceConfig.PortSpeed );
				if( rc != 0 )
				{
					throw new AttachFailedException( rc );
				}

				// 標準の点字体系を適用
				BrailleConfig = _BrailleConfig;

				// 点字ディスプレイサーバに接続
				rc = PinCtrlWinPinAttach();
				if( rc == 0 )
				{
					throw new AttachFailedException( AttachFailedException.FailedToAttachToServerError );
				}
			}
			catch( AttachFailedException ex )
			{
				ex.GetHashCode(); // (suppressing warning for unused object)
#				if !( IGNORE_ATTACH_ERROR )
				throw ex;
#				endif
			}

			// フラグをたてる
			_Attached = true;
		}
		

		/// <summary>
		/// 点字ディスプレイサーバから切断
		/// </summary>
		public void DetachFromServer()
		{
			if( !_Installed )
				return;

			int rc; // result code
			
			// サーバからプロセスを切り離す
			rc = PinCtrlWinPinDetach( 0 );
			if( rc == 0 )
			{
				return;
			}

			// フラグをおろす
			_Attached = false;

			// WinPinから切り離す
			PinCtrlEnd();
		}
		#endregion // NBSエンジンの使用準備と終了

		#region サーバーの操作
		/// <summary>
		/// サーバから WM_WINPIN_RECEIVE メッセージがアプリケーションに送られてきた時、
		/// メッセージを DLL に受け渡すために呼ばなければならないメソッド。
		/// 選択範囲は「以上以下」で指定します。
		/// </summary>
		/// <param name="wParam">メッセージの WPARAM パラメータ</param>
		/// <param name="selectionStart">カーソル位置、あるいは選択開始位置の全文字列中におけるインデックス</param>
		/// <param name="selectionEnd">選択終了位置のインデックス（何も選択していなければstartと同値）</param>
		public void SendDataToDll( IntPtr wParam, int selectionStart, int selectionEnd )
		{
			if( !_Installed )
				return;

			PinCtrlPinKeySet( (byte)wParam.ToInt32(), selectionStart, selectionEnd );
		}

		/// <summary>
		/// 点字ディスプレイの使用権をこのアプリケーションにロック。
		/// ロックしている間はこのアプリケーションが点字ディスプレイを占有する。
		/// </summary>
		/// <exception cref="LockFailedException">使用権の取得に失敗。</exception>
		public void Lock()
		{
			if( !_Installed )
				return;

			Int32 rc; // result code
			
			rc = PinCtrlWinPinLock( true );
			if( rc == 0 )
			{
				throw new LockFailedException(); // failed
			}
			
			_Locked = true;
		}

		/// <summary>
		/// 点字ディスプレイの使用権を解放。
		/// 他のアプリケーションが点字ディスプレイを使えるようにする。
		/// </summary>
		/// <exception cref="LockFailedException">使用権の解放に失敗。</exception>
		public void Unlock()
		{
			if( !_Installed )
				return;
			
			Int32 rc;
			
			// アタッチ済みなら何もしない
			if( _Attached == false )
			{
				return;
			}

			// 占有権を解放
			rc = PinCtrlWinPinLock( false );
			if( rc == 0 )
			{
				throw new LockFailedException(); // failed
			}
			
			_Locked = false;
		}
		#endregion // サーバーの操作

		#region 表示に関する操作と設定
		/// <summary>
		/// ディスプレイへの点字表示を抑制します。
		/// 呼び出し後はすべての描画命令がキャッシュされ、
		/// EndUpdate で抑制を解除すると
		/// 最後の描画命令が実際に実行されます。
		/// </summary>
		/// <seealso cref="EndUpdate"/>
		public void BeginUpdate()
		{
			_CacheCount++;
		}

		/// <summary>
		/// ディスプレイへの点字表示の抑制を解除し、
		/// 最後に行われた描画命令を実行します。
		/// </summary>
		public void EndUpdate()
		{
			_CacheCount--;
			UpdateDisplay();
		}

		/// <summary>
		/// 点字ディスプレイに文字列を表示。
		/// 選択範囲は「以上未満」で指定する。
		/// </summary>
		/// <param name="str">表示する文字列</param>
		/// <param name="selBegin">選択範囲の開始インデックス</param>
		/// <param name="selEnd">選択範囲の終了インデックス</param>
		/// <param name="statusNabccStr">ステータス領域に表示するNABCC文字列</param>
		/// <param name="endMark">文字列の終端に表示する記号</param>
		public void Write( string str, int selBegin, int selEnd, byte[] statusNabccStr, LineEndMark endMark )
		{
			if( !_Installed || !_Attached )
				return;

			byte[]	bytes;
			byte[]	bytesBefore, bytesOnCursor, bytesAfter;
			string	strBefore, strOnCursor, strAfter;

			// もし選択開始位置と終了位置が矛盾しているなら
			// カーソル無しで表示（表示位置は選択開始位置）
			if( selEnd < selBegin )
			{
				Write( ToSjisEncoding(str), selBegin, selBegin, statusNabccStr, endMark );
				return;
			}

			// カーソルのインデックス位置を計算するため、
			// 文字コード変換する前に三分割する
			strBefore = str.Substring( 0, selBegin );
			strOnCursor = str.Substring( selBegin, selEnd - selBegin );
			strAfter = str.Substring( selEnd, str.Length - selEnd );

			// エンコード
			bytesBefore = ToSjisEncoding( strBefore );
			bytesOnCursor = ToSjisEncoding( strOnCursor );
			bytesAfter = ToSjisEncoding( strAfter );
			bytes = Utl.Connect( bytesBefore, bytesOnCursor, bytesAfter );

			// 点字ディスプレイに表示
			Write(
				bytes,
				bytesBefore.Length,
				bytesBefore.Length + bytesOnCursor.Length,
				statusNabccStr,
				endMark
			);
		}

		/// <summary>
		/// 点字ディスプレイに文字列を表示。
		/// 選択範囲は「以上未満」で指定する。
		/// </summary>
		/// <param name="str">表示する文字列（システム標準のエンコーディング）</param>
		/// <param name="selBegin">選択範囲の開始インデックス</param>
		/// <param name="selEnd">選択範囲の終了インデックス</param>
		/// <param name="statusNabccStr">ステータス領域に表示するNABCC文字列</param>
		/// <param name="endMark">文字列の終端に表示する記号</param>
		public void Write( byte[] str, int selBegin, int selEnd, byte[] statusNabccStr, LineEndMark endMark )
		{
			if( selBegin < 0 || str.Length < selBegin )
				throw new ArgumentOutOfRangeException( "selBegin" );
			if( selEnd < selBegin || str.Length < selEnd )
				throw new ArgumentOutOfRangeException( "selEnd" );
			if( !_Installed || !_Attached )
				return;

			// outMode には 0 (単純文字列出力) か 3(編集行など) を与える仕様だが、
			// 0 だと点字表示体系の設定が有効にならない事があったため、3 に固定した。
			const int outMode = 3;
			int curFirst, curLast;

			// カーソル表示範囲を「以上未満」から「１始まりの以上以下」に変更
			curFirst = 1 + selBegin;
			curLast = 1 + selEnd;
			if( curLast < curFirst )
			{
				curLast = curFirst; // 空範囲指定時は長さ１の範囲が指定されたと再解釈
			}

			// 点字ディスプレイに表示するデータをキャッシュする
			TraceOutput( str, selBegin, selEnd );
			lock( this )
			{
				_CachedData = str;
				_CachedCursorStartPos = curFirst;
				_CachedCursorEndPos = curLast;
				_CachedStatusNabccStr = statusNabccStr;
				_CachedOutMode = outMode;
				_CachedEndMode = (Int32)endMark;
			}

			// 描画（描画抑制されていなければ）
			UpdateDisplay();
		}

		/// <summary>
		/// 最後の描画命令を実行（描画抑制されていなければ）
		/// </summary>
		public void UpdateDisplay()
		{
			// 描画抑制されていれば何もしない
			if( 0 < _CacheCount || _CachedData == null )
				return;

			// 描画
			lock( this )
			{
				PinCtrlWrite( _CachedData, _CachedData.Length, _CachedStatusNabccStr,
					_CachedCursorStartPos, _CachedCursorEndPos,
					_CachedOutMode, _CachedEndMode );
			}
		}

		/// <summary>
		/// 点字ディスプレイの表示をクリア
		/// </summary>
		public void ClearDisplay()
		{
			if( !_Installed )
				return;

			PinCtrlFlush();
		}

		/// <summary>
		/// 点字ディスプレイのカーソル表示方法
		/// </summary>
		public CursorMode CursorMode
		{
			get
			{
				if( !_Installed )
					return CursorMode.None;
				return (CursorMode)PinCtrlGetPinCurSw();
			}
			set
			{
				if( !_Installed )
					return;

				PinCtrlSetPinCurSw( (int)value );
			}
		}

		/// <summary>
		/// ステータス表示の幅（0:非表示, 正数:左端に表示, 負数:右端に表示）
		/// </summary>
		public int StatusWidth
		{
			get{ return _StatusWidth; }
			set
			{
				int nativeValue;

				if( value < 0 )
				{
					_StatusWidth = -5;
					nativeValue = 2; // 右端
				}
				else if( 0 == value )
				{
					_StatusWidth = 0;
					nativeValue = 0; // 非表示
				}
				else
				{
					_StatusWidth = +5;
					nativeValue = 1; // 左端
				}

				PinCtrlSetPinStatus( nativeValue );
			}
		}

		/// <summary>
		/// 表示に使う点字体系などの設定を取得あるいは設定します。
		/// </summary>
		public BrailleConfig BrailleConfig
		{
			get{ return _BrailleConfig; }
			set
			{
				_BrailleConfig = value;
				
				if( !_Installed )
					return;

				// 日本点字体系の設定を適用
				if( value.JapaneseCoding == JapaneseCoding.KanTenji )
				{
					PinCtrlSetPinCharAttr( 1 );
					PinCtrlSetPinCharDetail( 0 );
					PinCtrlSetPinTenKanji( 1 );
				}
				else if( value.JapaneseCoding == JapaneseCoding.RokuTenKanji )
				{
					PinCtrlSetPinCharAttr( 1 );
					PinCtrlSetPinCharDetail( 0 );
					PinCtrlSetPinTenKanji( 0 );
				}
				else if( value.JapaneseCoding == JapaneseCoding.Shousai )
				{
					PinCtrlSetPinCharDetail( 1 );
				}
				else// if( value.Japanese == JapaneseCoding.Kana )
				{
					PinCtrlSetPinCharAttr( 0 );
					PinCtrlSetPinCharDetail( 0 );
				}

				// 英数字点字体系の設定を適用
				if( value.EnglishCoding == EnglishCoding.Grade1 )
				{
					PinCtrlSetPinEiji( 0 );
				}
				else if( value.EnglishCoding == EnglishCoding.Grade2 )
				{
					PinCtrlSetPinEiji( 5 );
				}
				else if( value.EnglishCoding == EnglishCoding.NABCC )
				{
					PinCtrlSetPinEiji( 4 );
				}
				else if( value.EnglishCoding == EnglishCoding.JouhouKomoji )
				{
					PinCtrlSetPinEiji( 2 );
				}
				else if( value.EnglishCoding == EnglishCoding.JouhouOhmoji )
				{
					PinCtrlSetPinEiji( 3 );
				}
				else// if( value.English == EnglishBrailleSystem.JouhouNatural )
				{
					PinCtrlSetPinEiji( 1 );
				}

				// 点訳モードの設定を適用
				if( value.IsTenyakuMode  )
				{
					PinCtrlSetPinTenyakuMode( 1 );
				}
				else
				{
					PinCtrlSetPinTenyakuMode( 0 );
				}

				// 点訳モードの最大幅を設定
				if( 1 <= value.TenyakuMaxWidth && value.TenyakuMaxWidth <= 64 )
				{
					PinCtrlSetPinTenyakuMaxCol( value.TenyakuMaxWidth );
				}

				// タブ文字の表示幅を設定
				PinCtrlSetPinTabWidth( value.TabWidth - 1 );

				// カーソル表示の方法を設定
				this.CursorMode = value.CursorMode;
				
				// ステータス表示の幅を設定
				this.StatusWidth = value.StatusWidth;
			}
		}
		#endregion // 点字表示設定

		#region WM_WINPIN に関するイベント
		/// <summary>
		/// WinPinMessageReceived を処理するメソッドを表します。
		/// </summary>
		/// <param name="sender">イベントのソース</param>
		/// <param name="e">発生したイベントについての情報</param>
		public delegate void WinPinMessageHandler( object sender, WinPinMessageEventArgs e );
		
		/// <summary>
		/// 点字ディスプレイサーバから
		/// DLLへデータを転送するよう要求された時に発生します。
		/// </summary>
		public event WinPinMessageHandler WinPinMessageReceived;
		
		/// <summary>
		/// WinPinMessageReceived イベントに関する情報を提供します。
		/// </summary>
		public class WinPinMessageEventArgs : EventArgs
		{
			IntPtr _WParam;

			/// <summary>
			/// 新しいインスタンスを初期化します。
			/// </summary>
			/// <param name="WParam">点字ディスプレイサーバから送信されてきた WPARAM パラメータ</param>
			public WinPinMessageEventArgs( IntPtr WParam )
			{
				_WParam = WParam;
			}

			/// <summary>
			/// WinPinMessageReceivedイベント発生時にDLLへ転送しなければならないデータ
			/// </summary>
			public IntPtr WParam
			{
				get { return _WParam; }
			}
		}

		void InvokeWinPinMessageReceived( IntPtr WParam )
		{
			if( WinPinMessageReceived != null )
			{
				WinPinMessageReceived( this, new WinPinMessageEventArgs(WParam) );
			}
		}
		#endregion

		#region タッチカーソルキーのイベント
		/// <summary>
		/// TouchCursorPressed イベントを処理するメソッドを表します。
		/// </summary>
		/// <param name="sender">イベントのソース</param>
		/// <param name="e">発生したイベントについての情報</param>
		public delegate void TouchCursorPressedHandler( object sender, TouchCursorPressedEventArgs e );

		/// <summary>
		/// 点字ディスプレイのタッチカーソルキーが押された時か、
		/// シフトキーを押された時に発生します。
		/// </summary>
		public event TouchCursorPressedHandler TouchCursorPressed;

		/// <summary>
		/// TouchCursorPressed イベントに関する情報を提供します。
		/// </summary>
		public class TouchCursorPressedEventArgs : EventArgs
		{
			int _Index;
			bool _IsShiftDown, _IsControlDown, _IsAltDown;

			/// <summary>
			/// TouchCursorPressedEventArgs クラスの新しいインスタンスを初期化します。
			/// </summary>
			/// <param name="index">押されたタッチカーソルキーの位置（半角文字単位のインデックス）</param>
			/// <param name="isShiftDown">Shift キーが同時に押されたかどうか</param>
			/// <param name="isControlDown">Ctrl キーが同時に押されたかどうか</param>
			/// <param name="isAltDown">Alt キーが同時に押されたかどうか</param>
			public TouchCursorPressedEventArgs( int index, bool isShiftDown, bool isControlDown, bool isAltDown )
			{
				_Index = index;
				_IsShiftDown = isShiftDown;
				_IsControlDown = isControlDown;
				_IsAltDown = isAltDown;
			}

			/// <summary>
			/// ユーザが押したタッチカーソルの位置にある文字の位置（Shift_JISでのバイトインデックス）
			/// </summary>
			public int ByteIndex
			{
				get { return _Index; }
			}

			/// <summary>
			/// Shiftキーが同時に押されたかどうか
			/// </summary>
			public bool IsShiftDown
			{
				get{ return _IsShiftDown; }
			}

			/// <summary>
			/// Ctrlキーが同時に押されたかどうか
			/// </summary>
			public bool IsControlDown
			{
				get{ return _IsControlDown; }
			}
			
			/// <summary>
			/// Altキーが同時に押されたかどうか
			/// </summary>
			public bool IsAltDown
			{
				get{ return _IsAltDown; }
			}
		}

		/// <summary>
		/// TouchCursorPressed イベントを発生させます。
		/// </summary>
		/// <param name="index">押されたタッチカーソルキーの位置（半角文字単位のインデックス）</param>
		/// <param name="isShiftDown">Shift キーが同時に押されたかどうか</param>
		/// <param name="isControlDown">Ctrl キーが同時に押されたかどうか</param>
		/// <param name="isAltDown">Alt キーが同時に押されたかどうか</param>
		void OnTouchCursorPressed( int index, bool isShiftDown, bool isControlDown, bool isAltDown )
		{
			if( TouchCursorPressed != null )
			{
				TouchCursorPressed( this, new TouchCursorPressedEventArgs(index, isShiftDown, isControlDown, isAltDown) );
			}
		}
		#endregion

		#region 点字ディスプレイ上の方向キーに関するイベント
		/// <summary>
		/// 点字ディスプレイ上の方向キーを押された時に発生します。
		/// </summary>
		public event KeyEventHandler KeyDown;
		void OnKeyDown( KeyEventArgs e )
		{
			if( KeyDown != null )
			{
				KeyDown( this, e );
			}
		}

		/// <summary>
		/// 点字ディスプレイ上の方向キーが上がった時に発生します。
		/// </summary>
		public event KeyEventHandler KeyUp;
		void OnKeyUp( KeyEventArgs e )
		{
			if( KeyUp != null )
			{
				KeyUp( this, e );
			}
		}
		#endregion

		#region 行をまたぐシフトのイベント
		/// <summary>
		/// 点字ディスプレイから行をまたぐ左シフトの要求がきた後に発生します。
		/// </summary>
		public event EventHandler ShiftToPreviousLine;
		void OnShiftToPreviousLine()
		{
			if( ShiftToPreviousLine != null )
			{
				ShiftToPreviousLine( this, EventArgs.Empty );
			}
		}

		/// <summary>
		/// 点字ディスプレイから行をまたぐ右シフトの要求がきた後に発生します。
		/// </summary>
		public event EventHandler ShiftToNextLine;
		void OnShiftToNextLine()
		{
			if( ShiftToNextLine != null )
			{
				ShiftToNextLine( this, EventArgs.Empty );
			}
		}
		#endregion

		#region メッセージシンク
		/// <summary>
		/// エンジンから送られてくるウィンドウメッセージを受け取り、
		/// .NET のイベントに変換するためのシンク
		/// </summary>
		class MessageSink : IDisposable
		{
			IntPtr		_Handle;
			WindowProc	_MessageSinkWindowProc;
			NbsEngine	_Engine;
			
			public IntPtr Handle
			{
				get{ return _Handle; }
			}

			public MessageSink( NbsEngine engine )
			{
				const int HWND_MESSAGE_ID = -3;
				const int WS_MINIMIZE	= 0x20000000;
				const int WS_DISABLED = 0x08000000;
				const int WS_EX_NOACTIVATE	= 0x08000000;
				const string WindowClassName = "NbsEngine_CSharp_MessageSink";
				int			rc; // result code
				WNDCLASS	winc;
				IntPtr		parent;
				
				_Engine = engine;
				
				// create message-only window for 2000 or later
				if( 5 <= Environment.OSVersion.Version.Major )
				{
					parent = new IntPtr( HWND_MESSAGE_ID );
				}
				else
				{
					parent = IntPtr.Zero;
				}
				
				// prepare window class to create
				_MessageSinkWindowProc = new WindowProc( MessageSinkWindowProc );
				winc.style			= 0;
				winc.wndProc		= _MessageSinkWindowProc;
				winc.classExtra		= 0;
				winc.windowExtra	= 0;
				winc.instance		= IntPtr.Zero;
				winc.icon			= IntPtr.Zero;
				winc.cursor			= IntPtr.Zero;
				winc.backgroundBrush= IntPtr.Zero;
				winc.menuName		= null;
				winc.className		= WindowClassName;
				
				// register window class
				rc = RegisterClass( ref winc );
				if( rc == 0 )
				{
					throw new NbsException( "Failed to register window class of the message-sink." );
				}
				
				// create a window that has only message handling capability
				_Handle = CreateWindowEx( WS_EX_NOACTIVATE, WindowClassName, "", WS_DISABLED|WS_MINIMIZE, 0, 0, 0, 0, parent, IntPtr.Zero, IntPtr.Zero, IntPtr.Zero );
				if( _Handle == IntPtr.Zero )
				{
					throw new NbsException( "Failed to create message-sink window." );
				}
			}

			public void Dispose()
			{
				DestroyWindow( _Handle );
			}

			IntPtr MessageSinkWindowProc( IntPtr hWnd, Int32 message, IntPtr wparam, IntPtr lparam )
			{
				const int WM_WINPIN_RECEIVE		= 0x000070F;
				const int WM_PINCTRL_KEY		= 0x0000464;
				const int WM_PINCTRL_CHAR		= 0x0000465;
				const int WM_PINCTRL_TENJI_KEY	= 0x0000466;
				//const int WM_WINPIN_MODIFY_DIC	= 0x0000706;
				const int PINCTRL_NEXT_LINE_TOP		= 257;
				const int PINCTRL_BEFORE_LINE_END	= 258;
				const int WM_ACTIVATE	= 0x0006;

				if( message == WM_WINPIN_RECEIVE )
				{
					TraceWM( "WM_WINPIN_RECEIVE {0}", wparam.ToString("X2") );
					_Engine.InvokeWinPinMessageReceived( wparam );
				}
				else if( message == WM_PINCTRL_KEY )
				{
					TraceWM( "WM_PINCTRL_KEY" );
					Keys key = (Keys)wparam;

					if( key == Keys.Right+0x80 || key == Keys.Left+0x80 )
					{
						// タッチカーソルキーが押された
						int newColumnPos = (lparam.ToInt32() & 0x7FFFFFFF) - 1;
						TraceWM( "    TC-Key {0}", (lparam.ToInt32() & 0x7FFFFFFF) - 1 );
						_Engine.OnTouchCursorPressed(
								newColumnPos,
								Utl.IsShiftKeyDown(),
								Utl.IsControlKeyDown(),
								Utl.IsAltKeyDown()
							);
					}
					else if( key == Keys.Up || key == Keys.Down
						|| key == Keys.Left || key == Keys.Right )
					{
						// 方向キーが押された
						int count = (int)lparam;
						if((count&0x0F00) == 0 || (count&0x0F00) == 0x0100)
						{
							// キーダウン
							TraceWM( "    Dir-Key {0} down", key );
							_Engine.OnKeyDown( new KeyEventArgs(key) );
						}
						if( (count&0x0F00) == 0 || (count&0x0F00) == 0x0200 )
						{
							// キーアップ
							TraceWM( "    Dir-Key {0} up", key );
							_Engine.OnKeyUp( new KeyEventArgs(key) );
						}

						Keys keyData = (Keys)lparam;
						_Engine.OnKeyDown( new KeyEventArgs(keyData) );
					}
				}
				else if( message == WM_PINCTRL_CHAR )
				{
					TraceWM( "WM_PINCTRL_CHAR" );
				}
				else if( message == WM_PINCTRL_TENJI_KEY )
				{
					if( wparam.ToInt32() == PINCTRL_BEFORE_LINE_END )
					{
						TraceWM( "WM_PINCTRL_TENJI_KEY prev" );
						_Engine.OnShiftToPreviousLine();
					}
					else if( wparam.ToInt32() == PINCTRL_NEXT_LINE_TOP )
					{
						TraceWM( "WM_PINCTRL_TENJI_KEY next" );
						_Engine.OnShiftToNextLine();
					}
				}
				else if( message == WM_ACTIVATE )
				{
					// このコントロールがアクティブになってはいけない。
					// 直前にアクティブだったウィンドウをアクティブにしなおす
					const int WA_ACTIVE = 1;
					const int WA_CLICKACTIVE = 2;
					int low_word = (wparam.ToInt32() & 0x0000FFFF);
					if( low_word == WA_ACTIVE || low_word == WA_CLICKACTIVE )
					{
						if( lparam.ToInt32() != 0 )
						{
							IntPtr	prevWindow = lparam;
							Form.FromHandle( prevWindow ).Focus();
							return new IntPtr(1); // 親クラスの動作は実行しない
						}
					}
				}
				return new IntPtr(1);
			}

			[StructLayout(LayoutKind.Sequential)]
			struct WNDCLASS
			{
				public int			style;
				public WindowProc	wndProc;
				public int			classExtra;
				public int			windowExtra;
				public IntPtr		instance;
				public IntPtr		icon;
				public IntPtr		cursor;
				public IntPtr		backgroundBrush;
				[MarshalAs(UnmanagedType.LPWStr)]
				public string		menuName;
				[MarshalAs(UnmanagedType.LPWStr)]
				public string		className;
			}
			
			delegate IntPtr WindowProc( IntPtr hWnd, Int32 message, IntPtr wparam, IntPtr lparam );

			[DllImport("user32", EntryPoint="SendMessageW")]
			static extern IntPtr SendMessage( IntPtr hWnd, Int32 msg, IntPtr wparam, IntPtr lparam );
			[DllImport("user32", EntryPoint = "RegisterClassW")]
			static extern UInt16 RegisterClass( ref WNDCLASS wndClass );
			
			[DllImport ("user32")]
			static extern IntPtr CreateWindowEx( UInt32 exStyle, string className, string windowName,
				UInt32 style, Int32 x, Int32 y, Int32 width, Int32 height, IntPtr parent, IntPtr menu, IntPtr hInstance, IntPtr param );
			[DllImport("user32")]
			static extern Int32 DestroyWindow( IntPtr hWnd );

			class Utl
			{
				/// <summary>
				/// 直前のウィンドウメッセージ送信時にShiftキーが押されていたか判定
				/// </summary>
				/// <returns>Shiftキーが押されていたならtrue</returns>
				public static bool IsShiftKeyDown()
				{
					const int VK_SHIFT = 0x10;
					short state = GetKeyState( VK_SHIFT );
					return ( state < 0 );
				}

				/// <summary>
				/// 直前のウィンドウメッセージ送信時にCtrlキーが押されていたか判定
				/// </summary>
				/// <returns>Ctrlキーが押されていたならtrue</returns>
				public static bool IsControlKeyDown()
				{
					const int VK_CONTROL = 0x11;
					short state = GetKeyState( VK_CONTROL );
					return ( state < 0 );
				}

				/// <summary>
				/// 直前のウィンドウメッセージ送信時にAltキーが押されていたか判定
				/// </summary>
				/// <returns>Altキーが押されていたならtrue</returns>
				public static bool IsAltKeyDown()
				{
					const int VK_MENU = 0x12;
					short state = GetKeyState( VK_MENU );
					return ( state < 0 );
				}

				public static void SendKeyDownEvent( System.Windows.Forms.Keys key )
				{
					keybd_event( (byte)key, 0, 0, IntPtr.Zero );
				}
				public static void SendKeyUpEvent( System.Windows.Forms.Keys key )
				{
					keybd_event( (byte)key, 0, 2, IntPtr.Zero );
				}
				[DllImport("user32")]
				static unsafe extern void keybd_event( byte virtualKey, byte scanCode, UInt32 flags, IntPtr extraInfo );
				[DllImport("user32")]
				static extern Int16 GetKeyState( Int32 keyCode );
			}
		}
		#endregion // メッセージシンク

		#region Utilities
		/// <summary>
		/// Shift_JISでのバイトインデックスを、Unicodeでのインデックスに変換
		/// </summary>
		/// <param name="str">この文字列におけるShift_JISでのバイトインデックスをUnicodeインデックスに変換します。</param>
		/// <param name="byteIndex">Shift_JISでのバイトインデックス</param>
		/// <returns>Unicodeインデックス</returns>
		public static int SjisByteIndexToUnicodeIndex( string str, int byteIndex )
		{
			int charPos;
			int	hwPos			= 0;
			int offsetToNext	= 0;
			
			// 先頭から一文字ずつ見ていき、半角文字単位での幅を積算していく
			charPos = 0;
			while( charPos < str.Length )
			{
				char ch = str[charPos];
				
				// タブか半角文字？
				if( '\t' == ch || Utl.IsHalfWidthChar(ch) )
				{
					offsetToNext = 1;
				}
				// 全角文字？
				else
				{
					offsetToNext = 2;
				}

				// 次の文字へ
				hwPos += offsetToNext;
				charPos++;
				
				// 指定された幅を超えたなら、超える直前の文字カウントを返す
				if( byteIndex < hwPos )
				{
					return charPos - 1;
				}
			}

			// 全文字を見終わったのに、まだ指定された幅に達しない。
			// つまり指定された幅が大きすぎるという事なので、行末とする
			return str.Length;
		}

		/// <summary>
		/// Unicode 文字列を Shift_JIS 文字列に変換
		/// </summary>
		/// <param name="str">変換したい Unicode 文字列</param>
		/// <returns>変換された Shift_JIS に文字列（失敗するとnull）</returns>
		public static byte[] ToSjisEncoding( string str )
		{
			Encoder encoder;
			char[] chars;
			byte[] bytes;
			int byteCount;

			if( str == String.Empty )
			{
				return new byte[0];
			}

			// システム標準文字コードへエンコードしたら何バイトになるか計算
			encoder = Encoding.Default.GetEncoder();
			chars = str.ToCharArray();
			byteCount = encoder.GetByteCount( chars, 0, chars.Length, false );

			// エンコード
			bytes = new byte[byteCount];
			int rc = encoder.GetBytes( chars, 0, chars.Length, bytes, 0, true );
			if( rc == 0 )
			{
				return null;
			}

			return bytes;
		}

		class Utl
		{
			/// <summary>
			/// 三つのバイト列を結合
			/// </summary>
			/// <param name="source1">結合する一つめのバイト列</param>
			/// <param name="source2">結合する二つめのバイト列</param>
			/// <param name="source3">結合する三つめのバイト列</param>
			/// <returns>結合されたバイト列</returns>
			public static byte[] Connect( byte[] source1, byte[] source2, byte[] source3 )
			{
				byte[] target;

				target = new byte[source1.Length + source2.Length + source3.Length];
				for( int i = 0; i < source1.Length; i++ )
				{
					target[i]
						= source1[i];
				}
				for( int i = 0; i < source2.Length; i++ )
				{
					target[i + source1.Length]
						= source2[i];
				}
				for( int i = 0; i < source3.Length; i++ )
				{
					target[i + source1.Length + source2.Length]
						= source3[i];
				}

				return target;
			}

			/// <summary>
			/// 指定文字が半角文字かどうか判定
			/// </summary>
			/// <param name="ch">診断対象文字</param>
			/// <returns>半角だったら true</returns>
			public static bool IsHalfWidthChar( char ch )
			{
				// is in "Basic-Latin"?
				if( 0x20 <= ch && ch < 0x7F )
				{
					return true;
				}
				// is in "Half-width CJK punctuation"?
				if( 0xFF61 < ch && ch < 0xFF64 )
				{
					return true;
				}
				// is in "Half-width Katakana variants"?
				if( 0xFF65 < ch && ch < 0xFF9F )
				{
					return true;
				}
				// is in "Half-width Hangle variants"?
				if( 0xFFA0 < ch && ch < 0xFFBD )
				{
					return true;
				}
				// is in "Half-width Symbol variants"?
				if( 0xFFE8 < ch && ch < 0xFFEE )
				{
					return true;
				}

				return false;
			}

			/// <summary>
			/// 直前のウィンドウメッセージ送信時にShiftキーが押されていたか判定
			/// </summary>
			/// <returns>Shiftキーが押されていたならtrue</returns>
			public static bool IsShiftKeyDown()
			{
				const int VK_SHIFT = 0x10;
				short state = GetKeyState( VK_SHIFT );
				return ( state < 0 );
			}

			/// <summary>
			/// 直前のウィンドウメッセージ送信時にCtrlキーが押されていたか判定
			/// </summary>
			/// <returns>Ctrlキーが押されていたならtrue</returns>
			public static bool IsControlKeyDown()
			{
				const int VK_CONTROL = 0x11;
				short state = GetKeyState( VK_CONTROL );
				return ( state < 0 );
			}

			/// <summary>
			/// 直前のウィンドウメッセージ送信時にAltキーが押されていたか判定
			/// </summary>
			/// <returns>Altキーが押されていたならtrue</returns>
			public static bool IsAltKeyDown()
			{
				const int VK_MENU = 0x12;
				short state = GetKeyState( VK_MENU );
				return ( state < 0 );
			}

			[DllImport("user32.dll")]
			static extern Int16 GetKeyState( Int32 keyCode );
		}
		#endregion // Utilities
		
		#region デバッグ
		[Conditional("TRACE_OUTPUT")]
		static void TraceOutput( byte[] str, int curBegin, int curEnd )
		{
			// 出力行を表示
			Console.Error.WriteLine( Encoding.Default.GetString(str) + '|' );

			// 選択より前の部分に空白を表示
			for( int i=0; i<curBegin && i<str.Length; i++ )
			{
				if( str[i] == '\t' )
				{
					Console.Error.Write( "\t" );
				}
				else
				{
					Console.Error.Write( " " );
				}
			}

			// 選択部分の下に波線（^の列）を表示（手抜き）
			for( int i=curBegin; i<=curEnd && i<str.Length; i++ )
			{
				if( str[i] == '\t' )
				{
					Console.Error.Write( "→" );
				}
				else
				{
					Console.Error.Write( "^" );
				}
			}
			Console.Error.WriteLine( "" );
		}

		[Conditional("TRACE_WM")]
		static void TraceWM( string format, params object[] p )
		{
			Console.Error.WriteLine( String.Format(format, p) );
		}
		#endregion

		#region P/Invoke entry points
		/// <summary>
		/// NBS 点字ディスプレイサーバを初期化
		/// </summary>
		/// <param name="ownerWindow">サーバからのメッセージを受け取るウィンドウ</param>
		/// <param name="deviceId">点字ディスプレイのID</param>
		/// <param name="cursorMode">カーソル表示の方法（0:なし  1:アンダーライン表示  2:ブリンク表示）</param>
		/// <param name="statusType">ステータス表示の方法（0:なし  1:左端にステータス表示  2:右端にステータス表示）</param>
		/// <param name="port">点字ディスプレイが接続されているCOMポートの番号</param>
		/// <param name="portSpeed">ポートの通信速度</param>
		/// <returns>エラーコード</returns>
		Int32 PinCtrlInit(
			IntPtr ownerWindow, Int32 deviceId,
			CursorMode cursorMode, Int32 statusType,
			Int32 port, Int32 portSpeed )
		{
			if( 2 <= _Version.Major )
			{
				return PinCtrlInit_2_0( ownerWindow, deviceId, (int)cursorMode, statusType, port, portSpeed );
			}
			else
			{
				int cursorModeValue = (cursorMode != CursorMode.None) ? 1 : 0;
				return PinCtrlInit_1_1( ownerWindow, deviceId, cursorModeValue, port, portSpeed );
			}
		}

		/// <summary>
		/// NBS 点字ディスプレイサーバを初期化（v1.1用）
		/// </summary>
		/// <param name="ownerWindow">サーバからのメッセージを受け取るウィンドウ</param>
		/// <param name="deviceId">点字ディスプレイのID</param>
		/// <param name="showsCursor">カーソル表示の方法（0:なし  1:あり）</param>
		/// <param name="port">点字ディスプレイが接続されているCOMポートの番号</param>
		/// <param name="portSpeed">ポートの通信速度</param>
		/// <returns>エラーコード</returns>
		[DllImport("PinCtrl.dll", EntryPoint="PinCtrlInit")]
		static extern Int32 PinCtrlInit_1_1(
			IntPtr ownerWindow, Int32 deviceId,
			Int32 showsCursor,
			Int32 port, Int32 portSpeed );

		/// <summary>
		/// NBS 点字ディスプレイサーバを初期化（v2.0用）
		/// </summary>
		/// <param name="ownerWindow">サーバからのメッセージを受け取るウィンドウ</param>
		/// <param name="deviceId">点字ディスプレイのID</param>
		/// <param name="cursorType">カーソル表示の方法（0:なし  1:アンダーライン表示  2:ブリンク表示）</param>
		/// <param name="statusType">ステータス表示の方法（0:なし  1:左端にステータス表示  2:右端にステータス表示）</param>
		/// <param name="port">点字ディスプレイが接続されているCOMポートの番号</param>
		/// <param name="portSpeed">ポートの通信速度</param>
		/// <returns>エラーコード</returns>
		[DllImport("PinCtrl.dll", EntryPoint="PinCtrlInit")]
		static extern Int32 PinCtrlInit_2_0(
			IntPtr ownerWindow, Int32 deviceId,
			Int32 cursorType, Int32 statusType,
			Int32 port, Int32 portSpeed );

		/// <summary>
		/// DLL のバージョンを取得
		/// </summary>
		/// <returns>上位２バイトがメジャーバージョン、下位２バイトがマイナーバージョンの４バイト整数値</returns>
		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlGetVersion();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlEnd();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlIsWinPin();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlWinPinAttach();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlWinPinDetach( Int32 flushDisplay );

		/// <summary>
		/// 点字ディスプレイに文字列を表示（v2.0用）。
		/// カーソル表示範囲は「１始まりの以上以下」で指定。
		/// </summary>
		bool PinCtrlWrite(
			byte[] data, Int32 dataLength,
			byte[] status,
			Int32 cursorStartPos, Int32 cursorEndPos,
			Int32 outMode, Int32 endMode )
		{
			if( status == null )
				status = new byte[5];

			if( 2 <= _Version.Major )
				return PinCtrlWrite_2_0( data, dataLength, status, cursorStartPos, cursorEndPos, outMode, endMode );
			else
				return PinCtrlWrite_1_1( data, dataLength, cursorStartPos, cursorEndPos, outMode, endMode );
		}

		/// <summary>
		/// 点字ディスプレイに文字列を表示（v1.1用）。
		/// カーソル表示範囲は「１始まりの以上以下」で指定。
		/// </summary>
		[DllImport("PinCtrl.dll", EntryPoint="PinCtrlWrite")]
		static extern bool PinCtrlWrite_1_1(
			byte[] data, Int32 dataLength,
			Int32 cursorStartPos, Int32 cursorEndPos,
			Int32 outMode, Int32 endMode );

		/// <summary>
		/// 点字ディスプレイに文字列を表示（v2.0用）。
		/// カーソル表示範囲は「１始まりの以上以下」で指定。
		/// </summary>
		[DllImport("PinCtrl.dll", EntryPoint="PinCtrlWrite")]
		static extern bool PinCtrlWrite_2_0(
			byte[] data, Int32 dataLength,
			byte[] status,
			Int32 cursorStartPos, Int32 cursorEndPos,
			Int32 outMode, Int32 endMode );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlFlush();

		/// <summary>
		/// ステータス表示の方法を取得します。
		/// </summary>
		/// <returns>-2:DLLが非対応  -1:DLL未初期化  0:なし  1:左端にステータス表示  2:右端にステータス表示</returns>
		Int32 PinCtrlGetPinStatus()
		{
			if( 2 <= _Version.Major )
				return PinCtrlGetPinStatus_2_0();
			else
				return -2;
		}

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlGetPinStatus_2_0();

		/// <summary>
		/// ステータス表示の方法を設定します。
		/// </summary>
		/// <param name="statusType">ステータス表示の方法（0:なし  1:左端にステータス表示  2:右端にステータス表示）</param>
		/// <returns>-2:DLLが非対応  -1:DLL未初期化  0:失敗  1:成功</returns>
		Int32 PinCtrlSetPinStatus( Int32 statusType )
		{
			if( 2 <= _Version.Major )
				return PinCtrlSetPinStatus_2_0( statusType );
			else
				return -2;
		}

		[DllImport("PinCtrl.dll", EntryPoint="PinCtrlSetPinStatus")]
		static extern Int32 PinCtrlSetPinStatus_2_0( Int32 statusType );

		[DllImport("PinCtrl.dll")]
		unsafe static extern Int32 PinCtrlEnumComPort( Int32* outComPortNumberTable, Int32 maxTableSize );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlWinPinLock( bool doLock );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlGetPinCurSw();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinCurSw( Int32 show );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinCharAttr( Int32 isTenKanji );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinCharDetail( Int32 isDetailModeOn );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinTenKanji( Int32 isEightDotTenKanji );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinEiji( Int32 systemId );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinTabWidth( Int32 tabWidthId );

		/// <summary>点訳モードかどうかを取得</summary>
		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlGetPinTenyakuMode();

		/// <summary>点訳モードを設定</summary>
		/// <param name="isTenyakuMode">trueなら仮想モード</param>
		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinTenyakuMode( Int32 isTenyakuMode );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlGetPinTenyakuMaxCol();

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlSetPinTenyakuMaxCol( Int32 maxCol );

		/// <summary>点訳モード時に上下キーを押されたタイミングで呼び出す関数</summary>
		/// <param name="bDownKey">下方向キーが押された場合はTrue、上方向キーが押された場合はFalse</param>
		/// <returns>失敗時は0を返す</returns>
		[DllImport("PinCtrl.dll")]
		public static extern Int32 PinCtrlTenyakuKeySet( Int32 bDownKey );

		[DllImport("PinCtrl.dll")]
		static extern Int32 PinCtrlPinKeySet( byte cCh, Int32 lStartPos, Int32 lEndPos );
		#endregion // P/Invoke entry points
	}
}
