// file: DeviceConfigForm.cs
// brief: a form to ask braille display's setting in user's environment
// encoding: UTF-8
//=========================================================
using System;
using System.Windows.Forms;
using CultureInfo = System.Globalization.CultureInfo;
using Sgry.AiBTools;

namespace Sgry.AiBTools.AT
{
	using Gui;

	/// <summary>
	/// 点字ディスプレイの機種設定を行うフォーム。
	/// </summary>
	public class DeviceConfigForm : AiBDialogBase
	{
		#region Fields
		DeviceConfig	_DeviceConfig = null;
		
		string[] _Items_Combo_ComSpeed = new string[] {
				"9600", "14400", "19200",
				"38400", "57600", "115200", "128000"
			};
		#endregion

		#region Init / Dispose
		/// <summary>
		/// フォームを初期化
		/// </summary>
		/// <param name="initValue">ダイアログに表示する機種設定の初期値</param>
		public DeviceConfigForm( DeviceConfig initValue )
		{
			const string ComPortNameStringFormat = "COM{0}";
			InitializeComponent();
			LocalizeComponent();
			_Combo_DeviceName.SelectedIndexChanged += _Combo_DeviceName_SelectedIndexChanged;
			AutoScaleMode = AutoScaleMode.Font;
			Font = SystemInformation.MenuFont;
			Activated += Form_Activated;

			// 初期値をコピーして保持
			_DeviceConfig = new DeviceConfig( initValue );

			// 機種一覧コンボボックスを初期化
			ctor_Init_Combo_DeviceName();
			_Combo_DeviceName.SelectedItem = initValue.Device.ToString();

			// COM Port リストボックスを初期化
			try
			{
				_List_ComPort.Sorted = true;
				foreach( int comPortNum in NbsEngine.GetAvailableComPorts() )
				{
					string item;
					
					// 表示用の文字列を生成、追加
					item = String.Format( ComPortNameStringFormat, comPortNum  );
					_List_ComPort.Items.Add( item );
				}
				if( _DeviceConfig.Device.IsUsbDevice )
				{
					_List_ComPort.Items.Add( "USB" );
				}
			}
			catch( DllNotFoundException )
			{}

			if( 0 < _List_ComPort.Items.Count )
			{
				if( initValue.ComPort != NbsEngine.UsbPortID )
				{
					// COM 接続時は該当のポートを検索して選択
					int index = _List_ComPort.FindStringExact(
							String.Format( ComPortNameStringFormat, (initValue.ComPort + 1) )
						);
					if( 0 <= index && index < _List_ComPort.Items.Count )
					{
						_List_ComPort.SelectedIndex = index;
					}
					else
					{
						_List_ComPort.SelectedIndex = 0;
					}
				}
				else
				{
					// USB 接続時は USB を選択
					_List_ComPort.SelectedIndex = _List_ComPort.Items.Count - 1;
				}
			}

			// COM Speed コンボボックスを初期化
			_Combo_ComSpeed.Items.AddRange( _Items_Combo_ComSpeed );
			_Combo_ComSpeed.Text = initValue.PortSpeed.ToString();
		}

		void Form_Activated( object sender, EventArgs e )
		{
			_ComboBoxBrailler.UpdateDisplay();
		}
		#endregion

		#region その他
		/// <summary>
		/// ユーザが選択した機種設定
		/// </summary>
		public DeviceConfig DeviceConfig
		{
			get{ return _DeviceConfig; }
		}
		#endregion

		#region OK ボタンのイベント処理
		void OkButton_Click( object sender, EventArgs e )
		{
			DeviceNameComboBoxItem comboItem;

			// 機種名を保存
			if( _Combo_DeviceName.SelectedItem != null )
			{
				comboItem = (DeviceNameComboBoxItem)_Combo_DeviceName.SelectedItem;

				_DeviceConfig.Device = comboItem.Value;
			}
			else
			{
				_DeviceConfig.Device = BrailleDevice.None;
			}
			
			// ポート番号を保存
			try
			{
				if( _List_ComPort.Text.StartsWith("COM") )
				{
					// "COM" の後ろにある一文字だけ抽出
					int portNum = Int32.Parse( _List_ComPort.Text.Substring(3, 1) );
					_DeviceConfig.ComPort = portNum - 1;
				}
				else
				{
					_DeviceConfig.ComPort = NbsEngine.UsbPortID;
				}
			}
			catch( Exception )
			{
				_DeviceConfig.ComPort = 0;
			}
			
			// 通信速度を保存
			_DeviceConfig.PortSpeed = Int32.Parse( _Combo_ComSpeed.Text );

			this.DialogResult = DialogResult.OK;
			this.Close();
		}

		void _Combo_DeviceName_SelectedIndexChanged( object sender, EventArgs e )
		{
			DeviceNameComboBoxItem comboItem;
			BrailleDevice device = BrailleDevice.None;

			// 現在選択されている機種名を取得
			if( _Combo_DeviceName.SelectedItem != null )
			{
				comboItem = (DeviceNameComboBoxItem)_Combo_DeviceName.SelectedItem;
				device = comboItem.Value;
			}

			// USB ポートの表示・非表示を切り替える
			if( device.IsUsbDevice )
			{
				// 表示されていなければ表示
				int itemCount = _List_ComPort.Items.Count;
				if( 0 < itemCount
					&& _List_ComPort.Items[itemCount-1].ToString().StartsWith("USB") == false )
				{
					_List_ComPort.Items.Add( "USB" );
				}
			}
			else
			{
				// 表示していれば非表示に
				int itemCount = _List_ComPort.Items.Count;
				if( 0 < itemCount
					&& _List_ComPort.Items[itemCount-1].ToString().StartsWith("USB") )
				{
					_List_ComPort.Items.RemoveAt( itemCount-1 );
				}
			}
		}
		#endregion

		#region 内部処理
		void ctor_Init_Combo_DeviceName()
		{
			DeviceNameComboBoxItem item;
			DeviceNameComboBoxItem selectedItem = null;

			// デバイス選択ボックスを設定
			foreach( BrailleDevice device in BrailleDevice.AllDevices )
			{
				// コンボにこの列挙値を追加し、もし選択中の値ならついでに保存
				item = new DeviceNameComboBoxItem( device );
				_Combo_DeviceName.Items.Add( item );
				if( device == _DeviceConfig.Device )
				{
					selectedItem = item; // 選択中の値を見つけたので保存
				}
			}
			
			// 項目オブジェクトのメンバDisplayNameの値を項目名に
			_Combo_DeviceName.DisplayMember = "DisplayName";

			// 項目オブジェクトのメンバValueの値を項目の値に
			_Combo_DeviceName.ValueMember = "Value";

			// 現在選択されているデバイスをコンボ上でも選択
			_Combo_DeviceName.SelectedItem = selectedItem;
		}
		#endregion // 内部処理

		#region コンボボックスの項目クラス
		/// <summary>
		/// 点字ディスプレイデバイス名を表示するコンボボックスの項目。
		/// 環境に応じて適切な言語でその点字ディスプレイの名前を表示するために
		/// 項目を抽象化したもの。
		/// </summary>
		class DeviceNameComboBoxItem
		{
			BrailleDevice	_Value;

			public DeviceNameComboBoxItem( BrailleDevice device )
			{
				_Value = device;
			}

			/// <summary>
			/// 点字ディスプレイの名称を取得
			/// </summary>
			public string DisplayName
			{
				get{ return _Value.Name; }
			}

			/// <summary>
			/// 点字ディスプレイのIDを取得
			/// </summary>
			public BrailleDevice Value
			{
				get{ return _Value; }
			}
		}
		#endregion

		#region UI Component Initialization
		/// <summary>
		/// ダイアログで使用する点字表示クライアントを初期化します。
		/// </summary>
		protected override void InstallBraillers()
		{
			_ComboBoxBrailler.Init();
			_ComboBoxBrailler.Add( _Combo_DeviceName );
			_ComboBoxBrailler.Add( _Combo_ComSpeed );

			_ControlBrailler.Init();
			_ControlBrailler.Add( _Button_Ok );
			_ControlBrailler.Add( _Button_Cancel );
			
			_ListBoxBrailler.Init();
			_ListBoxBrailler.Add( _List_ComPort );
		}

		/// <summary>
		/// ダイアログで使用した点字表示クライアントのリソースを解放します。
		/// </summary>
		protected override void UninstallBraillers()
		{
			_ControlBrailler.Dispose();
			_ComboBoxBrailler.Dispose();
			_ListBoxBrailler.Dispose();
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			_ComboBoxReader.Add( _Combo_ComSpeed );
			_ComboBoxReader.Add( _Combo_DeviceName );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void UninstallReaders()
		{
			_ComboBoxReader.Dispose();
		}

		void InitializeComponent()
		{
			this._Label_DeviceName = new System.Windows.Forms.Label();
			this._Combo_DeviceName = new System.Windows.Forms.ComboBox();
			this._Label_ComPort = new System.Windows.Forms.Label();
			this._Combo_ComSpeed = new System.Windows.Forms.ComboBox();
			this._Label_ComSpeed = new System.Windows.Forms.Label();
			this._Button_Cancel = new System.Windows.Forms.Button();
			this._Button_Ok = new System.Windows.Forms.Button();
			this._List_ComPort = new System.Windows.Forms.ListBox();
			this.SuspendLayout();
			// 
			// _Label_DeviceName
			// 
			this._Label_DeviceName.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this._Label_DeviceName.Location = new System.Drawing.Point( 12, 12 );
			this._Label_DeviceName.Name = "_Label_DeviceName";
			this._Label_DeviceName.Size = new System.Drawing.Size( 95, 20 );
			this._Label_DeviceName.TabIndex = 0;
			this._Label_DeviceName.Text = "&Device Name";
			this._Label_DeviceName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// _Combo_DeviceName
			// 
			this._Combo_DeviceName.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this._Combo_DeviceName.Location = new System.Drawing.Point( 113, 12 );
			this._Combo_DeviceName.Name = "_Combo_DeviceName";
			this._Combo_DeviceName.Size = new System.Drawing.Size( 170, 24 );
			this._Combo_DeviceName.TabIndex = 1;
			// 
			// _Label_ComPort
			// 
			this._Label_ComPort.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this._Label_ComPort.Location = new System.Drawing.Point( 12, 40 );
			this._Label_ComPort.Name = "_Label_ComPort";
			this._Label_ComPort.Size = new System.Drawing.Size( 95, 20 );
			this._Label_ComPort.TabIndex = 2;
			this._Label_ComPort.Text = "COM &Port";
			this._Label_ComPort.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// _Combo_ComSpeed
			// 
			this._Combo_ComSpeed.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this._Combo_ComSpeed.Location = new System.Drawing.Point( 113, 100 );
			this._Combo_ComSpeed.Name = "_Combo_ComSpeed";
			this._Combo_ComSpeed.Size = new System.Drawing.Size( 170, 24 );
			this._Combo_ComSpeed.TabIndex = 5;
			// 
			// _Label_ComSpeed
			// 
			this._Label_ComSpeed.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this._Label_ComSpeed.Location = new System.Drawing.Point( 12, 100 );
			this._Label_ComSpeed.Name = "_Label_ComSpeed";
			this._Label_ComSpeed.Size = new System.Drawing.Size( 95, 20 );
			this._Label_ComSpeed.TabIndex = 4;
			this._Label_ComSpeed.Text = "COM &Speed";
			this._Label_ComSpeed.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// _Button_Cancel
			// 
			this._Button_Cancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
			this._Button_Cancel.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this._Button_Cancel.Location = new System.Drawing.Point( 192, 136 );
			this._Button_Cancel.Name = "_Button_Cancel";
			this._Button_Cancel.Size = new System.Drawing.Size( 90, 24 );
			this._Button_Cancel.TabIndex = 7;
			this._Button_Cancel.Text = "Cancel";
			// 
			// _Button_Ok
			// 
			this._Button_Ok.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this._Button_Ok.Location = new System.Drawing.Point( 96, 136 );
			this._Button_Ok.Name = "_Button_Ok";
			this._Button_Ok.Size = new System.Drawing.Size( 90, 24 );
			this._Button_Ok.TabIndex = 6;
			this._Button_Ok.Text = "&OK";
			this._Button_Ok.Click += new System.EventHandler( this.OkButton_Click );
			// 
			// _List_ComPort
			// 
			this._List_ComPort.ItemHeight = 16;
			this._List_ComPort.Location = new System.Drawing.Point( 113, 40 );
			this._List_ComPort.Name = "_List_ComPort";
			this._List_ComPort.Size = new System.Drawing.Size( 170, 52 );
			this._List_ComPort.Sorted = true;
			this._List_ComPort.TabIndex = 3;
			// 
			// DeviceConfigForm
			// 
			this.AcceptButton = this._Button_Ok;
			this.CancelButton = this._Button_Cancel;
			this.ClientSize = new System.Drawing.Size( 294, 172 );
			this.Controls.Add( this._Label_ComSpeed );
			this.Controls.Add( this._Combo_ComSpeed );
			this.Controls.Add( this._Label_ComPort );
			this.Controls.Add( this._List_ComPort );
			this.Controls.Add( this._Label_DeviceName );
			this.Controls.Add( this._Combo_DeviceName );
			this.Controls.Add( this._Button_Ok );
			this.Controls.Add( this._Button_Cancel );
			this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
			this.MinimizeBox = false;
			this.MaximizeBox = false;
			this.Name = "DeviceConfigForm";
			this.SizeGripStyle = System.Windows.Forms.SizeGripStyle.Hide;
			this.Text = "Device config of braille display";
			this.ResumeLayout( false );
		}

		void LocalizeComponent()
		{
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				this.Text = "点字ディスプレイ機種設定";
				_Label_DeviceName.Text = "機種名";
				_Label_ComPort.Text = "通信ポート";
				_Label_ComSpeed.Text = "通信スピード";
			}
		}
		#endregion

		#region UI Components
		ListBox		_List_ComPort;
		Label		_Label_DeviceName;
		ComboBox	_Combo_DeviceName;
		Label		_Label_ComPort;
		ComboBox	_Combo_ComSpeed;
		Label		_Label_ComSpeed;
		Button		_Button_Ok;
		Button		_Button_Cancel;

		ControlBrailler _ControlBrailler = new ControlBrailler();
		ComboBoxBrailler _ComboBoxBrailler = new ComboBoxBrailler();
		ListBoxBrailler _ListBoxBrailler = new ListBoxBrailler();
		ComboBoxReader _ComboBoxReader = new ComboBoxReader();
		#endregion // UI Components
	}
}
