﻿// file: BrlCvtParam.cs
// brief: Conversion parameter structure used by BrailleConverter.dll
// author: SGRY (YAMAMOTO Suguru)
// update: 2010-02-07
//=========================================================
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// ファイル形式変換のパラメータ
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public struct BrlCvtParam
	{
		#region Parameters
		/// <summary>英数字の種別</summary>
		[MarshalAs(UnmanagedType.I4)]
		public AlnumCharType	alnumCharType;
		
		/// <summary>日本語文字の種別</summary>
		[MarshalAs(UnmanagedType.I4)]
		public JaCharType		jaCharType;
		
		/// <summary>点字モード(?)</summary>
		[MarshalAs(UnmanagedType.I4)]
		public TenjiMode		tenjiMode;
		
		/// <summary>１行の最大文字数</summary>
		public Int32			maxColumn;
		
		/// <summary>１ページの行数</summary>
		public Int32			maxLine;
		
		/// <summary>ページ付けするかどうか</summary>
		public Int32			usePageNum;
		
		/// <summary>最初のページ番号（ページ付けする場合のみ有効）</summary>
		public Int32			startPageNum;
		
		/// <summary>基本モードを英語にするかどうか</summary>
		public Int32			isBaseModeEnglish;

		/// <summary>英数字を全角にするかどうか</summary>
		public Int32			useZenkakuAlnum;

		/// <summary>折れ線行の属性</summary>
		[MarshalAs(UnmanagedType.I4)]
		public FoldLineType		foldLineType;

		/// <summary>折れ線行の属性</summary>
		public unsafe fixed byte	foldLineNumAry[10];
		#endregion

		#region Property
		/// <summary>
		/// 標準の変換パラメータを取得します。
		/// </summary>
		public static BrlCvtParam Default
		{
			get
			{
				BrlCvtParam param = new BrlCvtParam();

				param.alnumCharType = AlnumCharType.JohoNatural;
				param.jaCharType = JaCharType.Kana;
				param.maxColumn = 32;
				param.maxLine = 18;
				param.startPageNum = 1;
				param.foldLineType = FoldLineType.None;
				param.isBaseModeEnglish = 1;

				return param;
			}
		}
		#endregion

		#region Serialization
		/// <summary>
		/// 設定内容をINIオブジェクトに保存します。
		/// </summary>
		public void Serialize( Ini ini, string sectionName )
		{
			StringBuilder buf = new StringBuilder( 64 );
			unsafe
			{
				ini.Set( sectionName, "alnumCharType", alnumCharType );
				ini.Set( sectionName, "jaCharType", jaCharType );
				ini.Set( sectionName, "tenjiMode", tenjiMode );
				ini.Set( sectionName, "maxColumn", maxColumn );
				ini.Set( sectionName, "maxLine", maxLine );
				ini.Set( sectionName, "usePageNum", usePageNum );
				ini.Set( sectionName, "startPageNum", startPageNum );
				ini.Set( sectionName, "isBaseModeEnglish", isBaseModeEnglish );
				ini.Set( sectionName, "useZenkakuAlnum", useZenkakuAlnum );
				ini.Set( sectionName, "foldLineType", foldLineType );
				fixed( byte* pFoldLineNumAry = foldLineNumAry )
				{
					buf.Append( pFoldLineNumAry[0].ToString() );
					for( int i=1; i<10; i++ )
					{
						buf.Append( ", " + pFoldLineNumAry[i].ToString() );
					}
				}
				ini.Set( sectionName, "foldLineNumAry", buf.ToString() );
			}
		}

		/// <summary>
		/// 設定内容をINIオブジェクトから復元します。
		/// </summary>
		public static BrlCvtParam Deserialize( Ini ini, string sectionName )
		{
			BrlCvtParam param = BrlCvtParam.Default;
			string numbersStr;

			try
			{
				param.alnumCharType = ini.Get( sectionName, "alnumCharType", param.alnumCharType );
				param.jaCharType = ini.Get( sectionName, "jaCharType", param.jaCharType );
				param.tenjiMode = ini.Get( sectionName, "tenjiMode", param.tenjiMode );
				param.maxColumn = ini.Get( sectionName, "maxColumn", param.maxColumn );
				param.maxLine = ini.Get( sectionName, "maxLine", param.maxLine );
				param.usePageNum = ini.Get( sectionName, "usePageNum", param.usePageNum );
				param.startPageNum = ini.Get( sectionName, "startPageNum", param.startPageNum );
				param.isBaseModeEnglish = ini.Get( sectionName, "isBaseModeEnglish", param.isBaseModeEnglish );
				param.useZenkakuAlnum = ini.Get( sectionName, "useZenkakuAlnum", param.useZenkakuAlnum );
				param.foldLineType = ini.Get( sectionName, "foldLineType", param.foldLineType );
				numbersStr = ini.Get( sectionName, "foldLineNumAry", "" );
				unsafe
				{
					SortedList<byte, byte> numbers = new SortedList<byte, byte>(32);
					
					// 小さい順に並べ替えつつ入力された数値列を取得
					foreach( string numStr in numbersStr.Split(',') )
					{
						byte num;
						bool ok;

						ok = Byte.TryParse( numStr.Trim(), out num );
						if( ok && 0 < num && numbers.ContainsKey(num) == false )
						{
							numbers.Add( num, num );
						}
					}
					
					// 構造体に設定
					for( int i=0; i<10; i++ )
					{
						if( numbers.Count <= i )
							break;

						param.foldLineNumAry[i] = numbers.Values[i];
					}
				}
			}
			catch
			{}
			
			return param;
		}
		#endregion
	}

	#region ファイル形式
	/// <summary>
	/// 点字ファイル形式
	/// </summary>
	public enum FileFormat : int
	{
		/// <summary>かな漢字テキストファイル</summary>
		KANJI_TEXT	= 0,
		
		/// <summary>ブレイルスター３</summary>
		BS3			= 1,
		
		/// <summary>ブレイルスター２</summary>
		BS2			= 2,
		
		/// <summary>IBM-BES</summary>
		BES			= 3,
		
		/// <summary>BET</summary>
		BET			= 4,
		
		/// <summary>BASE</summary>
		BASE		= 5,
		
		/// <summary>KGS BM</summary>
		BM			= 6,
		
		/// <summary>かなテキストファイル</summary>
		KANA_TEXT	= 7,
		
		/// <summary>NABCC</summary>
		NABCC		= 8,
		
		/// <summary>書式付きNABCC</summary>
		EX_NABCC	= 9,
		
		/// <summary>ブレイルライト</summary>
		BLITE		= 10,
		
		/// <summary>書式付きブレイルライト</summary>
		EX_BLITE	= 11,
		
		/// <summary>ブレイルスター</summary>
		BS1			= 12,
		
		/// <summary>コータクン</summary>
		KOUTAKUN	= 13,
		
		/// <summary>点字プリンター</summary>
		TPRINT		= 14
	}
	#endregion

	#region 文字種
	/// <summary>
	/// 英数字の点字種別
	/// </summary>
	public enum AlnumCharType : int
	{
		/// <summary>１級</summary>
		Grade1		= 0,
		
		/// <summary>情報処理ナチュラル</summary>
		JohoNatural	= 1,
		
		/// <summary>情報処理小文字</summary>
		JohoSmall	= 2,
		
		/// <summary>情報処理大文字</summary>
		JohoLarge	= 3,
		
		/// <summary>NABCC</summary>
		NABCC		= 4,
		
		/// <summary>２級（略字）</summary>
		Grade2		= 5
	}

	/// <summary>
	/// 日本文字の点字種別
	/// </summary>
	public enum JaCharType : int
	{
		/// <summary>かな分かち書き</summary>
		Kana		= 0,
		
		/// <summary>六点漢字</summary>
		SixDots		= 1,
		
		/// <summary>漢点字</summary>
		Kantenji	= 2,
		
		/// <summary>UJBCC</summary>
		UJBCC		= 3
	}
	#endregion

	#region その他の変換パラメータ
	/// <summary>
	/// 点字モード（？？）
	/// </summary>
	public enum TenjiMode : int
	{
		/// <summary>点字</summary>
		Tenji			= 0,

		/// <summary>かな点字</summary>
		KanaTenji		= 1,

		/// <summary>カナ（BSW形式）</summary>
		KanaBSW			= 2,

		/// <summary>カナ（半角カナテキスト）</summary>
		KanaHankaku		= 3,

		/// <summary>カタカナ（全角）</summary>
		KanaKatakana	= 4,

		/// <summary>ひらがな</summary>
		KanaHiragana	= 5
	}

	/// <summary>
	/// 折り返し線
	/// </summary>
	public enum FoldLineType : int
	{
		/// <summary>付けない</summary>
		None	= 0,

		/// <summary>全面</summary>
		Both	= 1,

		/// <summary>奇数面</summary>
		Odd		= 2,

		/// <summary>偶数面</summary>
		Even	= 3,
	}
	#endregion
}
