﻿// file: TextEditorBrailler.cs
// brief: NBS engine client for ITextEditor
// encoding: UTF-8
//=========================================================
using System;
using System.Text;
using System.Collections.Generic;
using System.Windows.Forms;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AT
{
	using AT;
	using Gui;

	/// <summary>
	/// ITextEditor を備えるエディタ部品を点字表示するクラス。
	/// </summary>
	public class TextEditorBrailler : IControlBrailler<ITextEditor>
	{
		static TextEditorBrailler _Instance = null;
		List<ITextEditor> _Controls = new List<ITextEditor>( 8 );
		ITextEditor _ActiveControl = null;
		int _CaretByteIndex = 0;

		#region Init / Dispose
		TextEditorBrailler()
		{
			// NBS エンジンのイベントハンドラを登録
			NbsEngine.Instance.WinPinMessageReceived += NbsEngine_WinPinMessageReceived;
			NbsEngine.Instance.TouchCursorPressed += NbsEngine_TouchCursorPressed;
			NbsEngine.Instance.ShiftToNextLine += NbsEngine_ShiftToNextLine;
			NbsEngine.Instance.ShiftToPreviousLine += NbsEngine_ShiftToPreviousLine;
			NbsEngine.Instance.KeyDown += NbsEngine_KeyDown;
		}

		/// <summary>
		/// リソースを解放します。
		/// </summary>
		public void Dispose()
		{
			Clear();

			// NBS エンジンのイベントハンドラを削除
			NbsEngine.Instance.WinPinMessageReceived -= NbsEngine_WinPinMessageReceived;
			NbsEngine.Instance.TouchCursorPressed -= NbsEngine_TouchCursorPressed;
			NbsEngine.Instance.ShiftToNextLine -= NbsEngine_ShiftToNextLine;
			NbsEngine.Instance.ShiftToPreviousLine -= NbsEngine_ShiftToPreviousLine;
			NbsEngine.Instance.KeyDown -= NbsEngine_KeyDown;
		}
		#endregion

		#region Properties
		/// <summary>
		/// 唯一のインスタンスを取得します。
		/// </summary>
		public static TextEditorBrailler Inst
		{
			get
			{
				if( _Instance == null )
				{
					_Instance = new TextEditorBrailler();
				}

				return _Instance;
			}
		}

		/// <summary>
		/// 現在アクティブになっている表示対象のコントロールを取得します。
		/// </summary>
		public ITextEditor ActiveControl
		{
			get{ return _ActiveControl; }
		}
		#endregion

		#region IControlBrailler<T>
		/// <summary>
		/// 点字表示するコントロールを追加します。
		/// </summary>
		public void Add( ITextEditor control )
		{
			if( control == null )
				throw new ArgumentNullException();

			_Controls.Add( control );

			// すべてのコントロールに共通なイベントハンドラを削除
			control.GotFocus += Control_GotFocus;
			control.Leave += Control_Leave;
			control.KeyUp += Control_KeyUp;
			control.KeyDown += Control_KeyDown;
			control.SelectionChanged += Control_SelectionChanged;
		}

		/// <summary>
		/// コントロールを点字表示の対象から外します。
		/// </summary>
		public void Remove( ITextEditor control )
		{
			control.GotFocus -= Control_GotFocus;
			control.Leave -= Control_Leave;
			control.KeyUp -= Control_KeyUp;
			control.KeyDown -= Control_KeyDown;
			control.SelectionChanged -= Control_SelectionChanged;
		}

		/// <summary>
		/// 登録済みの全コントロールを点字表示の対象から外します。
		/// </summary>
		public void Clear()
		{
			for( int i=0; i<_Controls.Count; i++ )
			{
				Remove( _Controls[ _Controls.Count-1 ] );
			}
		}
		#endregion

		#region コントロールからのイベント処理
		void Control_GotFocus( object sender, EventArgs e )
		{
			// コントロールが前回表示時と異なれば仮想カーソル位置をリセット
			if( _ActiveControl != (ITextEditor)sender )
			{
				_CaretByteIndex = 0;
			}

			// アクティブコントロールに設定
			_ActiveControl = (ITextEditor)sender;

			// 再表示
			UpdateDisplay();
		}

		void Control_Leave( object sender, EventArgs e )
		{
			_CaretByteIndex = 0;
			_ActiveControl = null;
		}

		void Control_KeyDown( object sender, KeyEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 点訳モード時は上下キーで表示窓をシフトする
			if( NbsEngine.Instance != null
				&& NbsEngine.Instance.IsAttached
				&& NbsEngine.Instance.BrailleConfig.IsTenyakuMode )
			{
				if( e.KeyCode == Keys.Down )
				{
					NbsEngine.PinCtrlTenyakuKeySet( 1 );
					e.Handled = true;
				}
				else if( e.KeyCode == Keys.Up )
				{
					NbsEngine.PinCtrlTenyakuKeySet( 0 );
					e.Handled = true;
				}
			}
		}

		void Control_KeyUp( object sender, KeyEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 文字削除時はキャレットが移動しない一方、
			// 表示を更新する必要がある。そのためキーコードで文字削除を検出
			if( e.KeyCode == Keys.Delete )
			{
				UpdateDisplay();
			}
		}

		void Control_SelectionChanged( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			UpdateDisplay();
		}
		#endregion

		#region NBSエンジンからのイベント処理
		/// <summary>
		/// 点字ディスプレイサーバから WM_WINPIN メッセージが送られた時の動作。
		/// </summary>
		void NbsEngine_WinPinMessageReceived( object sender, NbsEngine.WinPinMessageEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			int start, end;

			// 選択状態を取得
			GetSelectionRange( out start, out end );

			// サーバから送られてきたデータを DLL に転送
			NbsEngine.Instance.SendDataToDll( e.WParam, start, end );
		}

		/// <summary>
		/// 点字ディスプレイのタッチカーソルキーが押された時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_TouchCursorPressed( object sender, NbsEngine.TouchCursorPressedEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			_CaretByteIndex = e.ByteIndex;

			// 具象クラスへ処理を委譲
			OnTouchCursorPressed( e );
		}

		/// <summary>
		/// NBS エンジンから次の行へシフトする要求がきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_ShiftToNextLine( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnShiftToNextLine();
		}

		/// <summary>
		/// NBS エンジンから前の行へシフトする要求がきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_ShiftToPreviousLine( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnShiftToPreviousLine();
		}

		/// <summary>
		/// NBS エンジンからキー押し下げイベントが送られてきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_KeyDown( object sender, KeyEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnKeyDown( e );
		}
		#endregion // NBSエンジンからのイベント処理

		#region Template Methods
		/// <summary>
		/// 現在の状態を点字ディスプレイに再出力
		/// </summary>
		public void UpdateDisplay()
		{
			if( _ActiveControl == null
				|| _ActiveControl.Focused != true )
			{
				return;
			}

			ITextEditor	editor = _ActiveControl;
			int caretLineIndex, caretColumnIndex, caretLineHeadIndex;
			int selStart, selEnd;
			string caretLineContent;
			LineEndMark lineEndMark;
			byte[] statusBytes = null;

			// キャレット位置とキャレット行の内容を取得
			editor.GetCaretIndex( out caretLineIndex, out caretColumnIndex );
			caretLineContent = editor.GetLineContent( caretLineIndex );
			caretLineHeadIndex = editor.GetLineHeadIndex( caretLineIndex );

			// 行番号をステータス表示
			{
				int statusMaxLen = Math.Abs( NbsEngine.Instance.StatusWidth );

				string lineNum = (caretLineIndex + 1).ToString();
				if( lineNum.Length <= statusMaxLen )
				{
					string statusStr = lineNum;
					/*string columnNum = (caretColumnIndex + 1).ToString();
					if( lineNum.Length + columnNum.Length + 1 <= statusMaxLen )
					{
						statusStr += " " + columnNum;
					}*/
					statusBytes = new byte[ statusStr.Length ];
					Array.Copy( Encoding.ASCII.GetBytes(statusStr), statusBytes, statusStr.Length );
				}
			}

			// 選択状態を取得
			if( editor.SelectingToEnd )
			{
				Utl.GetSelectionState_WhenSelectingToEnd( editor, caretColumnIndex, out selStart, out selEnd );
			}
			else
			{
				Utl.GetSelectionState_WhenNotSelectingToEnd( editor, caretColumnIndex, caretLineContent.Length, out selStart, out selEnd );
			}

			// 現在行が物理改行された行かどうか判定
			lineEndMark = Utl.GetLineEndType( editor, caretLineHeadIndex, caretLineContent.Length );

			// キャレット行を出力
			NbsEngine.Instance.Write(
					caretLineContent,
					selStart,
					selEnd,
					statusBytes,
					lineEndMark
				);
		}

		/// <summary>
		/// 現在表示中テキストにおける選択範囲を取得します。
		/// </summary>
		void GetSelectionRange( out int begin, out int end )
		{
			_ActiveControl.GetSelection( out begin, out end );
		}
		
		/// <summary>
		/// 点字ディスプレイのタッチカーソルが押された直後に呼ばれます。
		/// </summary>
		void OnTouchCursorPressed( NbsEngine.TouchCursorPressedEventArgs e )
		{
			ITextEditor	editor = _ActiveControl;
			int caretIndex, lineIndex, lineHeadIndex, columnIndex, index;
			string caretLineContent;
			
			// 現在キャレットがある行の先頭にある文字のインデックスを取得
			caretIndex = editor.GetCaretIndex();
			lineIndex = editor.GetLineIndexFromCharIndex( caretIndex );
			lineHeadIndex = editor.GetLineHeadIndex( lineIndex );

			// タッチカーソルが押されたのが何文字目か計算
			caretLineContent = editor.GetLineContent( lineIndex );
			columnIndex = NbsEngine.SjisByteIndexToUnicodeIndex( caretLineContent, e.ByteIndex );
			
			// タッチカーソルが押された文字の、テキスト全体でのインデックスを取得
			index = lineHeadIndex + columnIndex;

			if( e.IsShiftDown )
			{
				// タッチカーソルが押された位置まで選択を拡張
				editor.SetSelection( editor.SelectionAnchor, index );
			}
			else
			{
				editor.SetSelection( index, index );
			}
			//DO_NOT//UpdateDisplay(); // 選択変更イベントがディスプレイを更新するので不要
		}
		
		/// <summary>
		/// 点字ディスプレイの操作が次の行へシフトしなければならなくなった直後に呼ばれます。
		/// </summary>
		void OnShiftToNextLine()
		{
			ITextEditor	editor = _ActiveControl;
			int line, column;

			editor.GetCaretIndex( out line, out column );
			
			// キャレットが最終行になければ
			// 次行の先頭にキャレットを移動
			if( line != editor.GetLineCount() - 1 )
			{
				editor.SetCaretIndex( line + 1, 0 );
			}
			else
			{
				// キャレットが最終行にあるので
				// ファイル末尾にキャレットを移動
				int index = editor.TextLength;
				_ActiveControl.SetSelection( index, index );
			}
			//DO_NOT//UpdateDisplay();
		}

		/// <summary>
		/// 点字ディスプレイの操作が前の行へシフトしなければならなくなった直後に呼ばれます。
		/// </summary>
		void OnShiftToPreviousLine()
		{
			int line, column;
			int lineLength;
			
			_ActiveControl.GetCaretIndex( out line, out column );

			if( line != 0 )
			{
				// キャレットの位置は先頭行でない。
				// 前行の末尾にキャレットを移動
				lineLength = _ActiveControl.GetLineLength( line - 1 );
				_ActiveControl.SetCaretIndex( line - 1, Math.Max(0, lineLength-1) );
			}
			else
			{
				// キャレットが先頭行にある。
				// ファイル先頭にキャレットを移動
				_ActiveControl.SetSelection( 0, 0 );
			}
			//DO_NOT//UpdateDisplay();
		}

		/// <summary>
		/// 点字ディスプレイのキーが押された直後に呼ばれます。
		/// なおタッチカーソルキーとシフトキーが押された直後には呼ばれません。
		/// </summary>
		void OnKeyDown( KeyEventArgs e )
		{
			Utl.SendKeyDownEvent( e.KeyCode );
			Utl.SendKeyUpEvent( e.KeyCode );
			//DO_NOT//UpdateDisplay();
		}
		#endregion

		#region Utilities
		/// <summary>
		/// イベントを処理できる状態にあるかどうかを判定します。
		/// </summary>
		protected bool CanHandleEvent()
		{
			if( _ActiveControl == null )
			{
				return false;
			}
			
			if( _ActiveControl is Control
				&& (_ActiveControl as Control).Focused != true )
			{
				return false;
			}

			return true;
		}
		#endregion

		#region Utilities
		class Utl
		{
			public static LineEndMark GetLineEndType( ITextEditor editor, int lineHeadIndex, int lineLength )
			{
				// 行末の次の文字が改行コードか判定
				try
				{
					int lineEndIndex = lineHeadIndex + lineLength;
					if( editor.TextLength <= lineEndIndex )
					{
						return LineEndMark.PageEnd;
					}
					else if( editor.GetCharAt(lineEndIndex) == '\r' )
					{
						return LineEndMark.HardLineBreak;
					}
					else
					{
						return LineEndMark.SoftLineBreak;
					}
				}
				catch( ArgumentOutOfRangeException )
				{
					// 後続を取得できなかった＝テキスト末尾
					return LineEndMark.PageEnd;
				}
			}

			public static void GetSelectionState_WhenSelectingToEnd( ITextEditor editor, int caretColumnIndex, out int selectionStart, out int selectionEnd )
			{
				int selStart, selEnd, selLength;
				int selStartInFirstSelectionLine;

				// 状態を取得
				{
					int selStartLineIndex, selStartLineHeadIndex;

					editor.GetSelection( out selStart, out selEnd );
					selLength = selEnd - selStart;
					selStartLineIndex = editor.GetLineIndexFromCharIndex( selStart );
					selStartLineHeadIndex = editor.GetLineHeadIndex( selStartLineIndex );
					selStartInFirstSelectionLine = selStart - selStartLineHeadIndex;
				}

				//---- 選択開始位置を設定 ----
				// 選択が行をまたいでいるなら行頭に
				if( caretColumnIndex < selLength )
				{
					selectionStart = 0;
				}
				else
				{
					selectionStart = selStartInFirstSelectionLine;
				}

				//---- 選択終了位置を設定 ----
				if( 0 < selLength )
				{
					// 選択がある。
					// 選択中はキャレット位置にカーソル表示を行わない。
					// でないとキャレット位置まで選択中と勘違いしてしまう
					selectionEnd = caretColumnIndex - 1;
				}
				else
				{
					// 選択が無い
					selectionEnd = caretColumnIndex;
				}
			}

			// ファイル先頭に向かって選択されている場合の、
			// 点字ディスプレイに表示されるテキストの選択範囲を決定
			public static void GetSelectionState_WhenNotSelectingToEnd( ITextEditor editor, int caretColumnIndex, int caretLineLength, out int selectionStart, out int selectionEnd )
			{
				int selLength;
				int selEndInLastSelectionLine;

				{
					int selBeginInWhole;
					int selEndInWhole;
					int selEndLineIndex, selEndLineHeadIndex;

					editor.GetSelection( out selBeginInWhole, out selEndInWhole );
					selEndLineIndex = editor.GetLineIndexFromCharIndex( selEndInWhole );
					selEndLineHeadIndex = editor.GetLineHeadIndex( selEndLineIndex );
					selEndInLastSelectionLine = selEndInWhole - selEndLineHeadIndex;
					selLength = selEndInWhole - selBeginInWhole;
				}

				// 選択が行をまたぐなら行末後に
				selectionStart = caretColumnIndex;
				if( selEndInLastSelectionLine < selLength )
				{
					selectionEnd = caretLineLength - 1;
				}
				else
				{
					selectionEnd = selEndInLastSelectionLine - 1;
				}
			}

			/// <summary>
			/// 「キーが下がった」というイベントを発生させる
			/// </summary>
			public static void SendKeyDownEvent( System.Windows.Forms.Keys key )
			{
				keybd_event( (byte)key, 0, 0, IntPtr.Zero );
			}

			/// <summary>
			/// 「キーが上がった」というイベントを発生させる
			/// </summary>
			public static void SendKeyUpEvent( System.Windows.Forms.Keys key )
			{
				keybd_event( (byte)key, 0, 2, IntPtr.Zero );
			}

			[System.Runtime.InteropServices.DllImport("user32")]
			static extern void keybd_event( byte virtualKeyCode, byte scanCode, UInt32 flags, IntPtr extraInfo );
			[System.Runtime.InteropServices.DllImport("user32")]
			static extern UInt32 MapVirtualKey( UInt32 code, UInt32 mapType );
		}
		#endregion
	}
}
