// file: ControlBraillerBase.cs
// brief: base implementation of control braillers
// encoding: UTF-8
// update: 2008-12-20
//=========================================================
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// コントロールを点字表示する基本的機能を備えたベースクラスです。
	/// </summary>
	/// <typeparam name="T">点字表示するコントロールの型。</typeparam>
	public abstract class ControlBraillerBase<T> : IControlBrailler<T> where T : Control
	{
		#region Fields
		bool _Initialized = false;

		/// <summary>点字表示するすべてのコントロール。</summary>
		protected List<T> _Controls = new List<T>( 2 );
		
		/// <summary>現在アクティブになっている表示対象のコントロール。</summary>
		protected T _ActiveControl = null;

		/// <summary>表示中テキストにおける仮想カーソルの位置（バイトインデックス）。</summary>
		protected int _CaretByteIndex = 0;
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 点字表示オブジェクトを初期化します。
		/// </summary>
		public void Init()
		{
			try
			{
				// NBS エンジンのイベントハンドラを登録
				NbsEngine.Instance.WinPinMessageReceived += NbsEngine_WinPinMessageReceived;
				NbsEngine.Instance.TouchCursorPressed += NbsEngine_TouchCursorPressed;
				NbsEngine.Instance.ShiftToNextLine += NbsEngine_ShiftToNextLine;
				NbsEngine.Instance.ShiftToPreviousLine += NbsEngine_ShiftToPreviousLine;
				NbsEngine.Instance.KeyDown += NbsEngine_KeyDown;
				
				_Initialized = true;
			}
			catch( NullReferenceException ex )
			{
				// NBSEngine が初期化されていないので、例外を投げる
				throw new InvalidOperationException( "NbsEngine not initialized yet.", ex );
			}
		}

		/// <summary>
		/// リソースを解放します。
		/// </summary>
		public virtual void Dispose()
		{
			// NBS エンジンのイベントハンドラを削除
			try
			{
				NbsEngine.Instance.WinPinMessageReceived -= NbsEngine_WinPinMessageReceived;
				NbsEngine.Instance.TouchCursorPressed -= NbsEngine_TouchCursorPressed;
				NbsEngine.Instance.ShiftToNextLine -= NbsEngine_ShiftToNextLine;
				NbsEngine.Instance.ShiftToPreviousLine -= NbsEngine_ShiftToPreviousLine;
				NbsEngine.Instance.KeyDown -= NbsEngine_KeyDown;
			}
			catch( NullReferenceException )
			{}

			Clear();
		}
		#endregion

		/// <summary>
		/// 現在アクティブになっている表示対象のコントロールを取得します。
		/// </summary>
		public T ActiveControl
		{
			get{ return _ActiveControl; }
		}

		#region IControlBrailler<T>
		/// <summary>
		/// 点字表示するコントロールを追加します。
		/// </summary>
		public virtual void Add( T control )
		{
			if( !_Initialized )
				throw new InvalidOperationException( "Init was not called yet" );
			if( control == null )
				throw new ArgumentNullException();

			_Controls.Add( control );

			// すべてのコントロールに共通なイベントハンドラを削除
			control.GotFocus += Control_GotFocus;
			control.Leave += Control_Leave;

			// 特化のイベントハンドラを登録
			InstallSpecialEventHandlers( control );
		}

		/// <summary>
		/// コントロールを点字表示の対象から外します。
		/// </summary>
		public virtual void Remove( T control )
		{
			// すべてのコントロールに共通なイベントハンドラを削除
			control.GotFocus -= Control_GotFocus;
			control.Leave -= Control_Leave;

			// コントロール固有のイベントハンドラを削除
			// （派生クラスへ委譲）
			UninstallSpecialEventHandlers( control );
		}

		/// <summary>
		/// 登録済みの全コントロールを点字表示の対象から外します。
		/// </summary>
		public virtual void Clear()
		{
			for( int i=0; i<_Controls.Count; i++ )
			{
				Remove( _Controls[ _Controls.Count-1 ] );
			}
		}
		#endregion

		#region コントロールからのイベント処理
		/// <summary>
		/// コントロールがフォーカスを得た直後の動作
		/// </summary>
		protected virtual void Control_GotFocus( object sender, EventArgs e )
		{
			// コントロールが前回表示時と異なれば仮想カーソル位置をリセット
			if( _ActiveControl != (T)sender )
			{
				_CaretByteIndex = 0;
			}

			// アクティブコントロールに設定
			_ActiveControl = (T)sender;

			// 再表示
			UpdateDisplay();
		}

		/// <summary>
		/// コントロールがフォーカスを失った直後の動作
		/// </summary>
		protected virtual void Control_Leave( object sender, EventArgs e )
		{
			_CaretByteIndex = 0;
			_ActiveControl = null;
		}
		#endregion

		#region NBSエンジンからのイベント処理
		/// <summary>
		/// 点字ディスプレイサーバから WM_WINPIN メッセージが送られた時の動作。
		/// </summary>
		void NbsEngine_WinPinMessageReceived( object sender, NbsEngine.WinPinMessageEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			int start, end;

			// 選択状態を取得
			GetSelectionRange( out start, out end );

			// サーバから送られてきたデータを DLL に転送
			NbsEngine.Instance.SendDataToDll( e.WParam, start, end );
		}

		/// <summary>
		/// 点字ディスプレイのタッチカーソルキーが押された時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_TouchCursorPressed( object sender, NbsEngine.TouchCursorPressedEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			_CaretByteIndex = e.ByteIndex;

			// 具象クラスへ処理を委譲
			OnTouchCursorPressed( e );
		}

		/// <summary>
		/// NBS エンジンから次の行へシフトする要求がきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_ShiftToNextLine( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnShiftToNextLine();
		}

		/// <summary>
		/// NBS エンジンから前の行へシフトする要求がきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_ShiftToPreviousLine( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnShiftToPreviousLine();
		}

		/// <summary>
		/// NBS エンジンからキー押し下げイベントが送られてきた時の動作。
		/// 再表示等は具象クラスで行う必要がある。
		/// </summary>
		protected void NbsEngine_KeyDown( object sender, KeyEventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			// 具象クラスへ処理を委譲
			OnKeyDown( e );
		}
		#endregion // NBSエンジンからのイベント処理

		/// <summary>
		/// 点字ディスプレイの表示を更新します。
		/// </summary>
		public void UpdateDisplay()
		{
			if( CanHandleEvent() )
			{
				OnUpdateDisplay();
			}
		}

		#region Template Methods
		/// <summary>点字ディスプレイの表示を更新します。</summary>
		public abstract void OnUpdateDisplay();

		/// <summary>対象コントロール固有のイベントハンドラをインストールします。</summary>
		protected abstract void InstallSpecialEventHandlers( T control );

		/// <summary>対象コントロール固有のイベントハンドラをアンインストールします。</summary>
		protected abstract void UninstallSpecialEventHandlers( T control );

		/// <summary>現在表示中テキストにおける選択範囲を取得します。</summary>
		protected abstract void GetSelectionRange( out int begin, out int end );
		
		/// <summary>点字ディスプレイのタッチカーソルが押された直後に呼ばれます。</summary>
		protected abstract void OnTouchCursorPressed( NbsEngine.TouchCursorPressedEventArgs e );
		
		/// <summary>点字ディスプレイの操作が次の行へシフトしなければならなくなった直後に呼ばれます。</summary>
		protected abstract void OnShiftToNextLine();

		/// <summary>点字ディスプレイの操作が前の行へシフトしなければならなくなった直後に呼ばれます。</summary>
		protected abstract void OnShiftToPreviousLine();

		/// <summary>
		/// 点字ディスプレイのキーが押された直後に呼ばれます。
		/// なおタッチカーソルキーとシフトキーが押された直後には呼ばれません。
		/// </summary>
		protected abstract void OnKeyDown( KeyEventArgs e );
		#endregion

		#region Utilities
		/// <summary>
		/// イベントを処理できる状態にあるかどうかを判定します。
		/// </summary>
		protected bool CanHandleEvent()
		{
			if( !NbsEngine.Instance.IsAttached || !NbsEngine.Instance.IsLocked )
			{
				return false;
			}
			if( _ActiveControl == null || _ActiveControl.Focused != true )
			{
				return false;
			}

			return true;
		}

		/// <summary>
		/// コントロールのラベル文字列から余分な &amp; 記号を削除する。
		/// </summary>
		protected string RemoveAmpersands( string label )
		{
			StringBuilder newLabel = new StringBuilder( label.Length );

			for( int i=0; i<label.Length; i++ )
			{
				if( label[i] == '&' )
				{
					i++;
					if( i < label.Length )
					{
						newLabel.Append( Char.ToLower(label[i]) ); // 点字では大文字符が邪魔と推測
					}
				}
				else
				{
					newLabel.Append( label[i] );
				}
			}

			return newLabel.ToString();
		}
		#endregion
	}
}
