// file: ControlBrailler.cs
// brief: control brailler for Control with label
// encoding: UTF-8
// update: 2010-05-09
//=========================================================
using System;
using System.Collections;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;

namespace Sgry.AiBTools.AT
{
	/// <summary>
	/// 一般的な UI 要素のテキストを点字ディスプレイに表示する監視オブジェクト
	/// </summary>
	public class ControlBrailler : ControlBraillerBase<Control>
	{
		#region ControlBraillerBase<T> の Template Method 実装
		/// <summary>
		/// 現在の状態を点字ディスプレイに再出力。
		/// </summary>
		public override void OnUpdateDisplay()
		{
			// もしTextにニーモニックがあればそれを削除
			string text = RemoveAmpersands( ActiveControl.Text );

			// 点字ディスプレイに表示
			NbsEngine.Instance.CursorMode = CursorMode.None;
			NbsEngine.Instance.Write(
					NbsEngine.ToSjisEncoding(text),
					_CaretByteIndex,
					_CaretByteIndex,
					null,
					LineEndMark.None
				);
		}

		/// <summary>
		/// 対象コントロール固有のイベントハンドラをインストールします。
		/// </summary>
		protected override void InstallSpecialEventHandlers( Control control )
		{
			control.TextChanged += Control_TextChanged;
		}

		/// <summary>
		/// 対象コントロール固有のイベントハンドラをアンインストールします。
		/// </summary>
		protected override void UninstallSpecialEventHandlers( Control control )
		{
			control.TextChanged -= Control_TextChanged;
		}

		/// <summary>
		/// 現在表示中テキストにおける選択範囲を取得します。
		/// </summary>
		protected override void GetSelectionRange( out int begin, out int end )
		{
			begin = end = _CaretByteIndex;
		}

		/// <summary>
		/// 点字ディスプレイのタッチカーソルが押された直後に呼ばれます。
		/// </summary>
		protected override void OnTouchCursorPressed( NbsEngine.TouchCursorPressedEventArgs e )
		{
			UpdateDisplay();
		}

		/// <summary>
		/// 点字ディスプレイの操作が次の行へシフトしなければならなくなった直後に呼ばれます。
		/// </summary>
		protected override void OnShiftToNextLine()
		{}

		/// <summary>
		/// 点字ディスプレイの操作が前の行へシフトしなければならなくなった直後に呼ばれます。
		/// </summary>
		protected override void OnShiftToPreviousLine()
		{}

		/// <summary>
		/// 点字ディスプレイのキーが押された直後に呼ばれます。
		/// なおタッチカーソルキーとシフトキーが押された直後には呼ばれません。
		/// </summary>
		protected override void OnKeyDown( KeyEventArgs e )
		{}
		#endregion

		#region コントロール固有のイベント処理
		void Control_TextChanged( object sender, EventArgs e )
		{
			if( !CanHandleEvent() )
				return;

			UpdateDisplay();
		}
		#endregion
	}
}
