// file: BrailleConfigForm.cs
// brief: dialog window to ask user prefernce of the braille system to use
// encoding: UTF-8
// update: 2012-03-25
//=========================================================
using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using Sgry.AiBTools;
using CultureInfo = System.Globalization.CultureInfo;

namespace Sgry.AiBTools.AT
{
	using Gui;

	/// <summary>
	/// 点字表示設定を行うフォーム
	/// </summary>
	public partial class BrailleConfigForm : AiBDialogBase
	{
		#region Fields
		BrailleConfig	_BrailleConfig = null;

		ControlBrailler _ControlBrailler = new ControlBrailler();
		ComboBoxBrailler _ComboBrailler = new ComboBoxBrailler();
		ComboBoxReader _ComboBoxReader = new ComboBoxReader();
		string _Msg_InvalidTenyakuMaxWidth = "Number of cells to use in Ten-yaku mode must be greater than " + BrailleConfig.MinTenyakuMaxWidth + ".";
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 点字表示の設定用ダイアログを初期化
		/// </summary>
		/// <param name="initValue">ダイアログに表示する点字設定の初期値</param>
		public BrailleConfigForm( BrailleConfig initValue )
		{
			InitializeComponent();
			LocalizeComponents();

			_OkButton.DialogResult = DialogResult.OK;
			_CancelButton.DialogResult = DialogResult.Cancel;
			Activated += Form_Activated;
			AcceptButton = _OkButton;
			CancelButton = _CancelButton;
			AutoScaleMode = AutoScaleMode.Font;
			Font = SystemInformation.MenuFont;
			int screenHeight = Screen.GetWorkingArea(this).Height;
			if( screenHeight*0.95 < this.Height )
			{
				this.Height = (int)( screenHeight*0.95 );
			}

			//---- 現在の設定を UI に反映 ----//

			// 英数字点字体系
			switch( initValue.EnglishCoding )
			{
				case EnglishCoding.Grade1:
					_Radio_English_Grade1.Checked = true;
					break;
				case EnglishCoding.Grade2:
					_Radio_English_Grade2.Checked = true;
					break;
				case EnglishCoding.NABCC:
					_Radio_English_Nabcc.Checked = true;
					break;
				case EnglishCoding.JouhouKomoji:
					_Radio_English_JouhouKomoji.Checked = true;
					break;
				case EnglishCoding.JouhouOhmoji:
					_Radio_English_JouhouOhmoji.Checked = true;
					break;
				case EnglishCoding.JouhouNatural:
				default:
					_Radio_English_JouhouNatural.Checked = true;
					break;
			}
			// 日本点字体系
			switch( initValue.JapaneseCoding )
			{
				case JapaneseCoding.KanTenji:
					_Radio_Japanese_KanTenji.Checked = true;
					break;
				case JapaneseCoding.RokuTenKanji:
					_Radio_Japanese_RokuTenKanji.Checked = true;
					break;
				case JapaneseCoding.Shousai:
					_Radio_Japanese_Shousai.Checked = true;
					break;
				default:
				case JapaneseCoding.Kana:
					_Radio_Japanese_Kana.Checked = true;
					break;
			}

			// 点訳モードを使うかどうかを保存
			_Check_Tenyaku.Checked = initValue.IsTenyakuMode;

			// 点訳モードの折り返し幅を保存
			_TextBox_TenyakuMaxWidth.Text = initValue.TenyakuMaxWidth.ToString();

			// カーソル表示
			_Combo_ShowCursor.SelectedIndex = (int)initValue.CursorMode;

			// タブ幅
			_Combo_TabWidth.SelectedIndex = 0;
			try
			{
				_Combo_TabWidth.SelectedIndex = initValue.TabWidth - 1;
			}
			catch( ArgumentException )
			{}

			// カーソル表示
			if( initValue.StatusWidth < 0 )
				_Combo_StatusPosition.SelectedIndex = 2;
			else if( 0 == initValue.StatusWidth )
				_Combo_StatusPosition.SelectedIndex = 0;
			else
				_Combo_StatusPosition.SelectedIndex = 1;
		}

		void Form_Activated( object sender, EventArgs e )
		{
			_ControlBrailler.UpdateDisplay();
		}
		#endregion

		/// <summary>
		/// フォームでユーザが入力した点字表示設定
		/// </summary>
		public BrailleConfig BrailleConfig
		{
			get{ return _BrailleConfig; }
		}

		#region GUI のイベント処理
		void BrailleConfigForm_Closing( object sender, CancelEventArgs e )
		{
			if( DialogResult != DialogResult.OK )
				return;

			int width;
			e.Cancel = true;

			// 点訳モード時の折り返しセル数が有効な数値か確認
			if( Int32.TryParse(_TextBox_TenyakuMaxWidth.Text, out width) == false
				|| width < BrailleConfig.MinTenyakuMaxWidth )
			{
				NbsMessageBox.Show( _Msg_InvalidTenyakuMaxWidth,
									MessageBoxButtons.OK,
									MessageBoxIcon.Warning );
				_TextBox_TenyakuMaxWidth.Select();
				_TextBox_TenyakuMaxWidth.SelectAll();
				return;
			}

			e.Cancel = false;
		}

		void BrailleConfigForm_Closed( object sender, EventArgs e )
		{
			if( DialogResult != DialogResult.OK )
				return;

			int statusWidth;

			this._BrailleConfig = new BrailleConfig();

			// 英数字の点字表記体系を保存
			if( _Radio_English_Grade1.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.Grade1;
			}
			else if( _Radio_English_Grade2.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.Grade2;
			}
			else if( _Radio_English_JouhouKomoji.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.JouhouKomoji;
			}
			else if( _Radio_English_JouhouOhmoji.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.JouhouOhmoji;
			}
			else if( _Radio_English_Nabcc.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.NABCC;
			}
			else// if( _Radio_English_JouhouNatural.Checked )
			{
				_BrailleConfig.EnglishCoding = EnglishCoding.JouhouNatural;
			}

			// 日本文字の点字表記体系を保存
			if( _Radio_Japanese_KanTenji.Checked )
			{
				_BrailleConfig.JapaneseCoding = JapaneseCoding.KanTenji;
			}
			else if( _Radio_Japanese_RokuTenKanji.Checked )
			{
				_BrailleConfig.JapaneseCoding = JapaneseCoding.RokuTenKanji;
			}
			else if( _Radio_Japanese_Shousai.Checked )
			{
				_BrailleConfig.JapaneseCoding = JapaneseCoding.Shousai;
			}
			else// if( _Radio_Japanese_Kana.Checked )
			{
				_BrailleConfig.JapaneseCoding = JapaneseCoding.Kana;
			}

			// 点訳モードを使うかどうかを保存
			_BrailleConfig.IsTenyakuMode = _Check_Tenyaku.Checked;

			// 点訳モードの折り返し幅を保存
			_BrailleConfig.TenyakuMaxWidth = Int32.Parse( _TextBox_TenyakuMaxWidth.Text );

			// カーソルを表示するかを保存
			_BrailleConfig.CursorMode = (CursorMode)_Combo_ShowCursor.SelectedIndex;

			// タブ表示幅を保存
			_BrailleConfig.TabWidth = _Combo_TabWidth.SelectedIndex + 1;

			// ステータス表示の幅を保存
			if( _Combo_StatusPosition.SelectedIndex == 1 )
				statusWidth = 5;
			else if( _Combo_StatusPosition.SelectedIndex == 2 )
				statusWidth = -5;
			else
				statusWidth = 0;
			_BrailleConfig.StatusWidth = statusWidth;
		}
		#endregion

		#region UI component initialization
		/// <summary>
		/// ダイアログで使用する点字表示クライアントを初期化します。
		/// </summary>
		protected override void InstallBraillers()
		{
			_ControlBrailler.Init();
			_ControlBrailler.Add( _Radio_English_Grade1 );
			_ControlBrailler.Add( _Radio_English_Grade2 );
			_ControlBrailler.Add( _Radio_English_Nabcc );
			_ControlBrailler.Add( _Radio_English_JouhouNatural );
			_ControlBrailler.Add( _Radio_English_JouhouKomoji );
			_ControlBrailler.Add( _Radio_English_JouhouOhmoji );
			_ControlBrailler.Add( _Radio_Japanese_Kana );
			_ControlBrailler.Add( _Radio_Japanese_KanTenji );
			_ControlBrailler.Add( _Radio_Japanese_RokuTenKanji );
			_ControlBrailler.Add( _Radio_Japanese_Shousai );
			_ControlBrailler.Add( _OkButton );
			_ControlBrailler.Add( _CancelButton );

			_ComboBrailler.Init();
			_ComboBrailler.Add( _Combo_ShowCursor );
			_ComboBrailler.Add( _Combo_TabWidth );
			_ComboBrailler.Add( _Combo_StatusPosition );
		}

		/// <summary>
		/// ダイアログで使用した点字表示クライアントのリソースを解放します。
		/// </summary>
		protected override void UninstallBraillers()
		{
			_ComboBrailler.Dispose();
			_ControlBrailler.Dispose();
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			_ComboBoxReader.Add( _Combo_TabWidth );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void UninstallReaders()
		{
			_ComboBoxReader.Dispose();
		}

		/// <summary>
		/// 文字列をローカライズ
		/// </summary>
		void LocalizeComponents()
		{
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				Text = "点字表示設定";
				_Group_English.Text = "英数字(&E)";
				_Group_Japanese.Text = "日本語文字(&J)";
				_Radio_English_Grade1.Text = "一級(&1)";
				_Radio_English_Grade2.Text = "二級(&2)";
				_Radio_English_Nabcc.Text = "NABCC(&B)";
				_Radio_English_JouhouNatural.Text = "ナチュラル表記（情報） (&N)";
				_Radio_English_JouhouKomoji.Text = "小文字基本表記（情報） (&L)";
				_Radio_English_JouhouOhmoji.Text = "大文字基本表記（情報） (&U)";
				_Radio_Japanese_Kana.Text = "かな(&K)";
				_Radio_Japanese_KanTenji.Text = "漢点字(&8)";
				_Radio_Japanese_RokuTenKanji.Text = "六点漢字(&6)";
				_Radio_Japanese_Shousai.Text = "詳細読み(&S)";
				_Group_Tenyaku.Text = "点訳モード";
				_Check_Tenyaku.Text = "点訳モードを使用する(&Y)";
				_Label_TenyakuMaxWidth.Text = "表示に使う最大セル数(&W)";
				_Group_Others.Text = "その他(&O)";
				_Label_ShowCursor.Text = "カーソルの表示方法(&C)";
				_Combo_ShowCursor.Items[0] = "表示しない";
				_Combo_ShowCursor.Items[1] = "アンダーライン";
				_Combo_ShowCursor.Items[2] = "点滅";
				_Label_TabWidth.Text = "タブ幅(&T)";
				_Label_StatusPosition.Text = "ステータス表示(&A)";
				_Combo_StatusPosition.Items[0] = "表示しない";
				_Combo_StatusPosition.Items[1] = "左端に表示";
				_Combo_StatusPosition.Items[2] = "右端に表示";

				_Msg_InvalidTenyakuMaxWidth = "表示に使う最大セル数には" + BrailleConfig.MinTenyakuMaxWidth + "以上の数値を指定してください。";
			}
		}
		#endregion
	}
}
