// file: CompileCommand.cs
// brief: base class for internal command of AiB Terminal.
// encoding: UTF-8
// update: 2009-02-22
//=========================================================
using System;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using Win32Exception = System.ComponentModel.Win32Exception;

namespace Sgry.AiBTools.AiBTerminal.Command
{
	using AT;
	using Compilation;

	class CompileCommand
	{
		protected Localizer _Localizer = new Localizer();
		protected AiBTerminalForm _MainForm;
		protected Compiler _Compiler;

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		/// <exception cref="FileNotFoundException">コンパイラが見つからないか、実行できない致命的エラーがある</exception>
		public CompileCommand( AiBTerminalForm mainForm, Compiler compiler )
		{
			// パラメータを保存
			_MainForm = mainForm;
			_Compiler = compiler;

			// 実行可能なコンパイラか確認
			if( Utl.IsExecutable(_Compiler.ExecutableName) == false )
			{
				throw new FileNotFoundException();
			}

			// 言語リソースをロード
			_Localizer.LoadResourceFile( "CommonUtl" );
		}
		#endregion

		/// <summary>
		/// コンパイラコマンドを実行します。
		/// </summary>
		public virtual void Exec( AiBTerminalForm mainForm )
		{
			// コンパイラを実行
			using( Form dialog = new NowCompilingForm(_Compiler) )
			{
				dialog.ShowDialog( mainForm );
			}

			// 結果を表示
			if( _Compiler.Succeeded )
			{
				OnCompileSucceeded();
			}
			else if( _Compiler.Failed )
			{
				OnCompileError( _Compiler );
			}
			else
			{
				OnOtherErrors( _Compiler ); // その他のエラー
			}
		}

		/// <summary>
		/// コンパイル成功時の動作
		/// </summary>
		protected virtual void OnCompileSucceeded()
		{
			// 「成功した」とメッセージを表示
			string message = _Localizer.TryGetString( "Compiler.Msg_CompileSucceeded", "Successfully compiled." );
			NbsMessageBox.Show( _MainForm, message );
		}

		/// <summary>
		/// コンパイル エラーが発生した時の動作
		/// </summary>
		protected virtual void OnCompileError( Compiler compiler )
		{
			ErrorListForm errorListForm;
			
			// エラーを一覧表示
			using( errorListForm = new ErrorListForm(compiler) )
			{
				Win32.MessageBeep_Information();
				errorListForm.ShowDialog( _MainForm );
			}
		}

		/// <summary>
		/// その他のエラーが発生した時の動作
		/// </summary>
		protected virtual void OnOtherErrors( Compiler compiler )
		{
			// そのまま表示
			using( OutputForm form = new OutputForm(compiler.Log) )
			{
				form.Text = Application.ProductName;
				Win32.MessageBeep_Information();
				form.ShowDialog( _MainForm );
			}
		}

		#region Utilities
		protected static class Utl
		{
			/// <summary>
			/// 指定したプログラムが実行可能か確認します。
			/// </summary>
			public static bool IsExecutable( string exeFileName )
			{
				Process p = new Process();

				try
				{
					// とりあえず実行するという荒業で判定
					p.StartInfo.WindowStyle = ProcessWindowStyle.Hidden;
					p.StartInfo.CreateNoWindow = true;
					p.StartInfo.FileName = exeFileName;
					p.Start();
				}
				catch( Exception )
				{
					return false;
				}
				finally
				{
					try
					{
						p.Kill();
						p.Dispose();
					}
					catch
					{}
				}

				return true;
			}
		}
		#endregion
	}
}
