// file: AppLogic.cs
// brief: application logic of AiB Terminal
// encoding: UTF-8
//=========================================================
using System;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Timers;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AiBTerminal
{
	using AiBTerminal.Command;
	using AT;
	using Compilation;

	/// <summary>
	/// アプリケーションロジックを統括する。
	/// 子プロセスの出力監視については MonitoringLogic クラスが受け持つ。
	/// </summary>
	class AppLogic
	{
		static Localizer _Localizer = new Localizer();
		static string _AuditoryIconPath = null;
		static string Msg_CompilerNotFound = "{0} was not found. Please ensure that PATH environment variable contains the path of the directory containing {0}.";
		MonitoringLogic _MonitoringLogic;
		AiBTerminalForm _MainForm;

		public static Localizer Localizer
		{
			get{ return _Localizer; }
		}

		/// <summary>
		/// 新しいインスタンスを生成
		/// </summary>
		public AppLogic( AiBTerminalForm form )
		{
			_MainForm = form;
			_MainForm.Closing += Form_Closing;
			_MainForm.Closed += Form_Closed;
			_MainForm.Disposed += Form_Disposed;

			_MonitoringLogic = new MonitoringLogic( form );
			_MonitoringLogic.ProcessExited += ChildProcess_Exited;

			form.OutputWindow.SetExternalWmHookProc( OutputWindow_WmHook );
		}

		/// <summary>
		/// Ctrl+Break イベントを子プロセスに送信
		/// </summary>
		public void SendCtrlBreak()
		{
			_MonitoringLogic.SendCtrlBreak();
		}

		/// <summary>
		/// フォームが閉じられる直前の確認動作。
		/// 本当に終了していいのかユーザに問い合わせる。
		/// </summary>
		void Form_Closing( object sender, CancelEventArgs e )
		{
			DialogResult	reply;
			string			message;
			
			// 子プロセスが終了しているなら確認するまでもない
			if( _MonitoringLogic.ChildProcessHasExited )
			{
				return;
			}

			// ask user that it's okay to close or not
			message = AppLogic.Localizer.TryGetString( "AiBTerminalForm.Msg_ConfirmExit", "The program is still running. Are you sure to exit?" );
			reply = NbsMessageBox.Show( _MainForm, message, MessageBoxButtons.OKCancel );
			if( reply == DialogResult.Cancel )
			{
				e.Cancel = true; // cancel closing
				return;
			}
		}

		/// <summary>
		/// フォームが閉じられた後の動作。
		/// フォーム消滅前に行う後処理を実行。
		/// </summary>
		void Form_Closed( object sender, EventArgs e )
		{
			// save settings
			_MainForm.Config.SavePreferences( _MainForm, _MainForm._BrailleConfig, _MainForm._DeviceConfig );

			// dispose staffs around child process
			_MonitoringLogic.Dispose();

			// dispose staffs around braille display
			TextEditorBrailler.Inst.Dispose();
			_MainForm.EndUseBrailleDisplay();
		}

		void Form_Disposed( object sender, EventArgs e )
		{
			NbsEngine.Dispose();
		}

		/// <summary>
		/// ユーザの入力が確定した時の動作。
		/// </summary>
		/// <param name="text">ユーザが入力した文字列</param>
		/// <param name="withNewLineCode">改行コードを付けずに入力を送信するかどうか</param>
		public void InputBox_InputComitted( string text, bool noEolCode )
		{
			if( _MonitoringLogic.ChildProcessHasExited )
			{
				return;
			}

			// play audio icon
			PlayAuditoryIcon( "InputSubmitted.wav" );

			// 現在の末尾位置を記憶
			string output = _MainForm.OutputText;
			_MainForm.SetLastEndPos( output.LastIndexOf("\r") + 1 );

			// 特殊コマンドを処理
			if( Utl.ExecuteSpecialCommand(_MainForm, text) )
			{
				// 特殊コマンドを実行したので、
				// 子プロセスへはコマンドを送らず空文字を送る
				text = String.Empty;
			}

			NbsEngine.Instance.BeginUpdate();

			// 子プロセスへ送信
			if( noEolCode )
			{
				_MonitoringLogic.WriteToBuffer( text );
				_MonitoringLogic.WriteToChildProcess( text );
			}
			else
			{
				_MonitoringLogic.WriteToBuffer( text + Environment.NewLine );
				_MonitoringLogic.WriteToChildProcess( text + Environment.NewLine );
			}

			try
			{
				//DO_NOT//_MainForm.OutputWindow.Focus();
				
				// 出力の監視タイマーを再起動
				_MonitoringLogic.ResetTimer();
			}
			catch( ObjectDisposedException )
			{
				// 子プロセスへたとえば"exit"を送信した場合、メソッドが終わる前に
				// FormとそのメンバがDisposeされる事がある。これを無視。
				// 2007-03-04追記。スレッドセーフな処理に再実装して以降、再現性が無くなっている。不要かも。
				Debug.Fail("ObjectDisposedException thrown in AppLogic.InputBox_InputComitted");
			}
			NbsEngine.Instance.EndUpdate();
		}

		int OutputWindow_WmHook( IntPtr hWnd, UInt32 msg, IntPtr wParam, IntPtr lParam, out IntPtr lResult )
		{
			// IME 関連の処理をすべて横取り
			if( Win32.WM_IME_STARTCOMPOSITION <= msg && msg <= Win32.WM_IME_COMPOSITION )
			{
				NbsEngine.Instance.BeginUpdate();

				// 各文字の確定ごとにイベントを発行
				if( msg == Win32.WM_IME_COMPOSITION )
				{
					_MainForm.InputWindow.Focus();
					int end = _MainForm.InputWindow.TextLength;
					_MainForm.InputWindow.SetSelection( end, end );
				}

				NbsEngine.Instance.EndUpdate();
				lResult = IntPtr.Zero;
				return 1;
			}

			// 一般的な可読文字の入力ならば入力ウィンドウに遷移
			if( msg == Win32.WM_CHAR )
			{
				char ch = (char)wParam.ToInt32();
				if( ' ' <= ch && ch <= '~' )
				{
					NbsEngine.Instance.BeginUpdate();

					_MainForm.InputWindow.Focus();
					_MainForm.InputWindow.Text = ch.ToString();
					int end = _MainForm.InputWindow.TextLength;
					_MainForm.InputWindow.SetSelection( end, end );

					if( AutoSpeaker.Instance is PcTalkerSpeaker )
					{
						AutoSpeaker.Instance.Stop();
						AutoSpeaker.Instance.Speak( ch.ToString() );
					}

					NbsEngine.Instance.EndUpdate();
				}

				lResult = IntPtr.Zero;
				return 1;
			}
			
			// Ctrl, Enter を処理
			if( msg == Win32.WM_KEYDOWN )
			{
				Keys key = (Keys)wParam.ToInt32();
				if( key == Keys.ControlKey )
				{
					if( XpReaderSpeaker.IsAlive() )
						XpReaderSpeaker.Instance.Stop();
					lResult = IntPtr.Zero;
					return 1;
				}
				if( key == Keys.Enter )
				{
					NbsEngine.Instance.BeginUpdate();
					_MainForm.SetLastEndPos( _MainForm.OutputText.LastIndexOf("\r") + 1 );
					_MonitoringLogic.WriteToBuffer( "\r" );
					_MonitoringLogic.WriteToChildProcess( "\r\n" );
					_MainForm.ResetCaretPosition();
					NbsEngine.Instance.EndUpdate();
					lResult = IntPtr.Zero;
					return 1;
				}
			}

			lResult = IntPtr.Zero;
			return 0;
		}


		/// <summary>
		/// 出力ウィンドウで文字が入力された時の動作。
		/// 自動的に入力ウィンドウにフォーカスを移動し、
		/// その文字を入力ウィンドウに設定する。
		/// また、Enter が入力された場合は改行コードを子プロセスに送信する。
		/// </summary>
		void OutputTextBox_KeyPress( object sender, KeyPressEventArgs e )
		{
			char ch = e.KeyChar;

			NbsEngine.Instance.BeginUpdate();

			// 一般的な可読文字
			if( ' ' <= ch && ch <= '~' )
			{
				Debug.Fail("Now there must be no chance to be executed...");
				_MainForm.InputWindow.Focus();
				_MainForm.InputWindow.Text = ch.ToString();
				int end = _MainForm.InputWindow.TextLength;
				_MainForm.InputWindow.SetSelection( end, end );
			}
			// Enter キー
			else if( ch == '\r' || ch == '\n' )
			{
				Debug.Fail("Now there must be no chance to be executed...");
				_MainForm.SetLastEndPos( _MainForm.OutputText.LastIndexOf("\r") + 1 );
				_MonitoringLogic.WriteToBuffer( "\r" );
				_MonitoringLogic.WriteToChildProcess( "\r\n" );
				_MainForm.ResetCaretPosition();
			}

			NbsEngine.Instance.EndUpdate();
		}

		/// <summary>
		/// 子プロセスが終了した直後の動作。
		/// アプリケーションを終了するためにメインウィンドウを閉じる
		/// </summary>
		void ChildProcess_Exited( object sender, EventArgs e )
		{
			// 子プロセスが終了すればアプリケーションも終了
			if( _MainForm.Config.CloseOnExit )
			{
				_MainForm.Close();
			}
		}

		/// <summary>
		/// AiB Editを起動します。
		/// </summary>
		public void ExecAibedit( string argument )
		{
			try
			{
				string path = Path.GetDirectoryName( Application.ExecutablePath );
				path = Path.Combine( path, "AiBEditWin.exe" );
				System.Diagnostics.Process.Start( path, argument );
			}
			catch( Exception ex )
			{
				// メッセージを表示
				NbsMessageBox.Show( _MainForm, ex.Message,
									MessageBoxButtons.OK, MessageBoxIcon.Error );
			}
		}

		#region 音アイコン
		/// <summary>
		/// 音アイコンを再生
		/// </summary>
		public static void PlayAuditoryIcon( string fileName )
		{
			Win32.PlaySound( Path.Combine(AuditoryIconPath, fileName), true );
		}

		/// <summary>
		/// 音アイコンファイルを配置したディレクトリのパス
		/// </summary>
		static string AuditoryIconPath
		{
			get
			{
				if( _AuditoryIconPath == null )
				{
					string appDir = Path.GetDirectoryName( Application.ExecutablePath );
					_AuditoryIconPath = Path.Combine( appDir, "icons" );
				}
				return _AuditoryIconPath;
			}
		}
		#endregion

		class Utl
		{
			public static bool ExecuteSpecialCommand( AiBTerminalForm mainForm, string commandLine )
			{
				string exeFileName = "";
				string argument = "";
				Compiler compiler;
				CompileCommand command;
				
				// コマンドライン文字列を分割
				Match match = Regex.Match( commandLine, @"\s*([^\s]+)(\s+.*)?" );
				if( 2 <= match.Groups.Count )
				{
					exeFileName = match.Groups[1].ToString();
				}
				if( 3 <= match.Groups.Count )
				{
					argument = match.Groups[2].ToString();
				}

				// 特殊コマンドの対象かどうかを判定
				if( exeFileName.StartsWith("_javac") )
				{
					compiler = new JavacCompiler( argument, mainForm.InputWindow.CurrentDirectory );
				}
				else if( exeFileName.StartsWith("_csc") )
				{
					compiler = new CscCompiler( argument, mainForm.InputWindow.CurrentDirectory );
				}
				else
				{
					return false;
				}

				// 特殊コマンドを入力したように画面表示
				mainForm.AppendOutputText( commandLine + Environment.NewLine );

				try
				{
					// コマンド名に .exe が無いならば追加（プロセス起動時は厳密なファイル名が必要なので）
					if( exeFileName.EndsWith(".exe") == false)
						exeFileName += ".exe";

					// コマンドを実行
					command = new CompileCommand( mainForm, compiler );
					command.Exec( mainForm );
				}
				catch( FileNotFoundException )
				{
					string format = Localizer.TryGetString( "AppLogic.Msg_CompilerNotFound", Msg_CompilerNotFound );
					string msg = String.Format( format, compiler.ExecutableName );
					NbsMessageBox.Show( mainForm, msg );
				}
				catch( Exception ex )
				{
					NbsMessageBox.Show( mainForm, ex.Message );
				}

				return true;
			}
		}
	}
}
