// file: SaveFileDialog.cs
// brief: "Save file as" dialog
// encoding: UTF-8
// update: 2010-01-23
//=========================================================
using System;
using System.IO;
using System.Windows.Forms;
using System.Drawing;
using CultureInfo = System.Globalization.CultureInfo;

namespace Sgry.AiBTools.AiBEdit
{
	using AT;
	using Gui;

	/// <summary>
	/// ファイル保存ダイアログ
	/// </summary>
	class SaveFileDialog : OpenFileDialog
	{
		#region Property
		/// <summary>
		/// ダイアログ上で選択された改行コード
		/// </summary>
		public EolCode EolCode
		{
			get
			{
				switch( _EolCodeComboBox.SelectedIndex )
				{
					case 1: return EolCode.CR;
					case 2: return EolCode.LF;
					default: return EolCode.CRLF;
				}
			}
			set
			{
				switch( value )
				{
					case EolCode.CR:
						_EolCodeComboBox.SelectedIndex = 1;
						break;
					case EolCode.LF:
						_EolCodeComboBox.SelectedIndex = 2;
						break;
					default:
						_EolCodeComboBox.SelectedIndex = 0;
						break;
				}
			}
		}
		#endregion

		#region Dialog Customization
		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			base.InstallReaders();

			_ComboBoxReader.Add( _EolCodeComboBox );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void InstallBraillers()
		{
			base.InstallBraillers();

			_ComboBoxBrailler.Add( _EolCodeComboBox );
		}

		protected override bool ValidateFileName( string path )
		{
			DialogResult	result;
			string			format;

			// 同名ファイルが存在しなければ何もしない
			if( File.Exists(path) != true )
			{
				return true;
			}

			// 同名ファイルが存在するので、上書きするか確認する
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				format = "{0} は既に存在します。上書きしますか？";
			}
			else
			{
				format = "{0} already exists. overwrite?";
			}
			result = NbsMessageBox.Show( this,
					String.Format(format, Path.GetFileName(path)),
					MessageBoxButtons.YesNo
				);
			if( result != DialogResult.Yes )
			{
				return false; // 上書きキャンセルされた
			}

			return true;
		}

		protected override void SetupCustomPanel()
		{
			base.SetupCustomPanel();

			_CustomPanel.Controls.AddRange( new Control[]{
				_EolCodeLabel, _EolCodeComboBox
			} );

			// 
			// _EncodingComboBox
			// 
			_EncodingComboBox.Items[0] = "same encoding";
			//
			// _EolCodeLabel
			//
			_EolCodeLabel.Anchor = AnchorStyles.Top | AnchorStyles.Left;
			_EolCodeLabel.Location = new Point( 0, 28 );
			_EolCodeLabel.Name = "_EolCodeLabel";
			_EolCodeLabel.Size = new Size( 135, 20 );
			_EolCodeLabel.Text = "&EOL code";
			//
			// _EolCodeComboBox
			//
			_EolCodeComboBox.Anchor = AnchorStyles.Top | AnchorStyles.Left;
			_EolCodeComboBox.Location = new Point( 136, 25 );
			_EolCodeComboBox.Name = "_EolCodeComboBox";
			_EolCodeComboBox.DropDownStyle = ComboBoxStyle.DropDownList;
			_EolCodeComboBox.Items.AddRange( new object[] {
			"CRLF",
			"CR",
			"LF"} );
			_EolCodeComboBox.SelectedIndex = 0;

			_CustomPanel.Height = _EolCodeComboBox.Bottom;
		}

		protected override void LocalizeComponent()
		{
			base.LocalizeComponent();

			Localizer.TryGetString( "AiBFileDialog._EolCodeLabel.Text", _EolCodeLabel );

			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				_OkButton.Text = "保存(&S)";
				_CancelButton.Text = "キャンセル";
				Text = "名前を付けて保存";
			}
			else
			{
				_OkButton.Text = "&Save";
				Text = "Save file as";
			}
		}
		#endregion

		#region UI Components
		Label				_EolCodeLabel		= new Label();
		ComboBox			_EolCodeComboBox	= new ComboBox();
		#endregion
	}
}
