// file: RubyOutliner.cs
// brief: Outline parser for Ruby
// update: 2008-09-13
//=========================================================
using System;
using System.Text;
using System.Text.RegularExpressions;
using System.Collections;
using System.Windows.Forms;

namespace Sgry.AiBTools.AiBEdit
{
	/// <summary>
	/// Ruby pAEgC͊
	/// </summary>
	class RubyOutliner : Outliner
	{
		/// <summary>
		/// CX^X𐶐
		/// </summary>
		/// <param name="dialog">V{\AEgC_CAO</param>
		/// <param name="source">͂镶</param>
		/// <param name="caretIndex">݂̃Lbgʒu</param>
		public RubyOutliner( OutlineDialog dialog, string source, int caretIndex )
			: base( dialog, source, caretIndex )
		{}

		/// <summary>
		/// V{𒊏oăc[Rg[ɕ\
		/// </summary>
		public override void Parse()
		{
			// parse source code
			try
			{
				Parse( _Dialog.TreeView, Strip(_Source), _CaretIndex );
			}
#			if DEBUG
			catch( Exception ex )
			{
				throw ex;
			}
#			else
			catch {}
#			endif

			// if no identifier found, display dummy node with message
			if( _Dialog.TreeView.Nodes.Count == 0 )
			{
				_DummyNode.Tag = _CaretIndex;
				_Dialog.TreeView.Nodes.Add( _DummyNode );
			}
		}

		#region ̓WbN
		static void Parse( TreeView treeView, string source, int caretIndex )
		{
			Stack		endOwnerIsContainer = new Stack();
			TreeNode	currentParent = null, initSelNode = null;
			int			pos;
			string		token;

			// read each tokens and if it is a symbol to be displayed in the outline, add it to the tree
			pos = 0;
			token = Utl.WordAt( source, pos );
			while( token != null )
			{
				//--- analyze this token ---
				if( token == "module" )
				{
					AddContainerAt( treeView, ref currentParent, ref initSelNode, caretIndex, source, Utl.NextWordPos(source, pos) );
					endOwnerIsContainer.Push( true );
				}
				else if( token == "class" )
				{
					AddContainerAt( treeView, ref currentParent, ref initSelNode, caretIndex, source, Utl.NextWordPos(source, pos) );
					endOwnerIsContainer.Push( true );
				}
				else if( token == "def" )
				{
					AddNonContainerAt( treeView, ref currentParent, ref initSelNode, caretIndex, source, Utl.NextWordPos(source, pos) );
					endOwnerIsContainer.Push( false );
				}
				else if( token == "if" || token == "while" || token == "unless" )
				{
					if( IsModifierIfWhile(source, pos) == false )
						endOwnerIsContainer.Push( false );
				}
				else if( token == "for" || token == "do" || token == "begin" || token == "case" || token == "until" )
				{
					endOwnerIsContainer.Push( false );
				}
				else if( token == "end" )
				{
					bool ownerIsContainer = (bool)endOwnerIsContainer.Pop();
					if( ownerIsContainer )
					{
						currentParent = currentParent.Parent;
					}
				}

				// goto next token
				token = Utl.NextNonWhiteSpaceToken( source, ref pos );
			}

			// select initial selection node
			if( initSelNode != null )
			{
				Sgry.Win32.TreeView_SelectItem( treeView.Handle, initSelNode.Handle );
			}
		}

		/// <summary>
		/// if/unless/while Cqǂ𔻒
		/// </summary>
		static bool IsModifierIfWhile( string source, int pos )
		{
			if( pos <= 0 )
			{
				return false; // if/while OɕȂ̂if/whileCqł͂Ȃ
			}

			// get previous non white space char
			int prevChPos = Utl.PrevNonWhiteSpacePos( source, pos - 1 );
			int lineHeadPos = Utl.CurrentLineHeadPos( source, pos );

			if( prevChPos < lineHeadPos )
			{
				return false; // s if/while; ܂蕁ʂif/whileB
			}
			else if( source[prevChPos] == '='
				|| (source[prevChPos-1] == '=' && source[prevChPos] == '>') )
			{
				return false; // ̉Eӂ if B C ̎OZqƓӖif/else/endB
			}

			return true;
		}

		static void AddContainerAt( TreeView treeView, ref TreeNode currentNode,
				ref TreeNode initSelNode, int caretIndex,
				string source, int index )
		{
			TreeNode node = new TreeNode();

			// make node
			node.Text = Utl.WordAt( source, index );
			node.Tag = index;
			
			// but wait, it can be a singleton class. if so, add the object name
			if( node.Text == "<<" )
			{
				int nextWordPos = Utl.NextWordPos( source, index );
				if( nextWordPos != -1 )
				{
					node.Text += " " + Utl.WordAt( source, nextWordPos );
				}
			}

			// add node 
			if( currentNode == null )
			{
				treeView.Nodes.Add( node );
			}
			else
			{
				currentNode.Nodes.Add( node );
			}

			// if seek position didnt go over where caret is, keep this node as candidate for initial selection
			if( Utl.CurrentLineHeadPos(source, index) <= caretIndex )
			{
				initSelNode = node;
			}

			currentNode = node;
		}

		static void AddNonContainerAt( TreeView treeView, ref TreeNode currentNode,
				ref TreeNode initSelNode, int caretIndex,
				string source, int index )
		{
			TreeNode node = new TreeNode();

			// make node
			node.Text = GetMethodNameAt( source, index );
			node.Tag = index;

			// add node 
			if( currentNode == null )
			{
				treeView.Nodes.Add( node );
			}
			else
			{
				currentNode.Nodes.Add( node );
			}

			// if seek position didnt go over where caret is, keep this node as candidate for initial selection
			if( Utl.CurrentLineHeadPos(source, index) <= caretIndex )
			{
				initSelNode = node;
			}
		}

		static string GetMethodNameAt( string source, int index )
		{
			int endPos;

			// try extract token between here and where first '(', ';' or whitespaces is appeared
			for( endPos=index; endPos<source.Length; endPos++ )
			{
				char ch = source[endPos];
				if( Char.IsWhiteSpace(ch)
					|| ch == '('
					|| ch == ';' )
				{
					return source.Substring( index, endPos - index );
				}
			}

			// the token just after the 'def' is the last token. may not be isValid syntax but try parse.
			return source.Substring( index );
		}
		#endregion

		#region Rg̏WbN
		/// <summary>
		/// Ruby\[XRg菜
		/// </summary>
		/// <param name="src">Rg菜\[X</param>
		/// <returns>Rg菜ꂽ\[X</returns>
		public static string Strip( string src )
		{
			StringBuilder buffer = new StringBuilder( src.Length );

			for( int i=0; i<src.Length; i++ )
			{
				//--- strip here document ---
				if( src[i] == '<'
					&& i+1 < src.Length && src[i+1] == '<'
					&& i+2 < src.Length && Char.IsWhiteSpace(src[i+2]) == false )
				{
					// possibly here document. check
					int hdEndPos = GetHereDocumentEndAt( src, i );
					if( hdEndPos != -1 )
					{
						i = Utl.EmptyToPattern( src, i, hdEndPos, ref buffer );
					}
				}
				//--- strip string literals ---
				// single quote string literal begin?
				else if( src[i] == '\'' )
				{
					i = Utl.EmptyToPattern( src, i, "\'", ref buffer );
				}
				// string literal begin?
				else if( src[i] == '\"' )
				{
					i = Utl.EmptyToPattern( src, i, "\"", ref buffer );
				}
				// command output begin?
				else if( src[i] == '`' )
				{
					i = Utl.EmptyToPattern( src, i, "`", ref buffer );
				}
				// % expression begin?
				else if( src[i] == '%' )
				{
					char p_next = src[i+1];
					if( !Char.IsLetterOrDigit(p_next) )
					{
						i = Utl.EmptyToPattern( src, i, p_next.ToString(), ref buffer );
					}
					else
					{
						if( p_next == 'q' || p_next == 'Q' || p_next == 'w' || p_next == 'W'
							|| p_next == 'x' || p_next == 'r' || p_next == 's' )
							i = Utl.EmptyToPattern( src, i, src[i+2].ToString(), ref buffer );
					}
				}
				//--- comments ---
				// one-line-comment begin?
				else if( src[i] == '#' )
				{
					// get next line head pos
					int nextLineHeadPos = Utl.NextLineHeadPos( src, i+1 );
					if( nextLineHeadPos == -1 )
					{
						break; // final line not terminated by EOL
					}

					// replace between here to there with whitespaces
					for( int j=0; j<nextLineHeadPos - i - 1; j++ )
					{
						buffer.Append( ' ' );
					}
					i = nextLineHeadPos - 1;
				}

				// add this char to comment-stripped buffer
				buffer.Append( src[i] );
			}

			return buffer.ToString();
		}
		
		static int GetHereDocumentEndAt( string src, int index )
		{
			bool isIndentHd = false;
			Regex hdIdPattern, hdEndPattern;
			Match hdIdMatchRes, hdEndMatchRes;
			string hdId;
			int hdEndIdPos;
			int hdEndLineHeadPos;

			if( src.Length < index+2 )
			{
				return -1; // too short. never be here document.
			}

			// validate here doc begin pattern
			hdIdPattern = new Regex( @"<<(-?)([^\r|^\n]+)" );
			hdIdMatchRes = hdIdPattern.Match( src, index );
			if( hdIdMatchRes.Success != true )
			{
				return -1;
			}

			// check if the here doc terminator would be indented
			if( hdIdMatchRes.Groups[1].ToString() != String.Empty )
			{
				isIndentHd = true;
			}

			// find terminator of the here doc
			hdId = hdIdMatchRes.Groups[2].ToString();
			hdEndIdPos = src.IndexOf( hdId, index + 2 + hdId.Length ); // 2=="<<".Length
			if( hdEndIdPos == -1 )
			{
				return -1;
			}
			
			// validate the terminator
			hdEndLineHeadPos = Utl.CurrentLineHeadPos( src, hdEndIdPos );
			if( isIndentHd )
				hdEndPattern = new Regex( @"\s*" + hdId + @"[\r|\n]" );
			else
				hdEndPattern = new Regex( hdId + @"[\r|\n]" );
			hdEndMatchRes = hdEndPattern.Match( src, hdEndLineHeadPos );
			if( hdEndMatchRes.Success != true )
			{
				return -1;
			}

			return hdEndIdPos + hdId.Length;
		}
		#endregion
	}
}
