// file: OutlineDialog.cs
// brief: outline dialog window
// encoding: UTF-8
// update: 2008-12-20
//=========================================================
using System;
using System.Windows.Forms;

namespace Sgry.AiBTools.AiBEdit
{
	using Gui;
	using AT;

	/// <summary>
	/// 文書のアウトラインを表示するダイアログ。
	/// </summary>
	/// 各アウトラインパーサは TreeView プロパティで得られるツリービューにノードを追加する。
	/// また、ノードの Tag にそのシンボルがある位置をソースにおけるインデックスで与えておく。
	/// このダイアログはそれを前提に動作している。
	class OutlineDialog : AiBDialogBase
	{
		int _SelectedSymbolIndex;

		#region Init / Dispose
		/// <summary>
		/// インスタンスを生成
		/// </summary>
		public OutlineDialog()
		{
			// initialize form itself
			Text = "Outline";
			FormBorderStyle = FormBorderStyle.SizableToolWindow;
			StartPosition = FormStartPosition.CenterParent;
			Font = SystemInformation.MenuFont;
			Activated += Form_Activated;

			// initialize tree view
			_TreeView.AccessibleRole = AccessibleRole.Outline;
			_TreeView.Dock = DockStyle.Fill;
			_TreeView.PathSeparator = "@";
			_TreeView.KeyDown += new KeyEventHandler( TreeView_KeyDown );
			_TreeView.DoubleClick += new EventHandler( TreeView_DoubleClick );
			Controls.Add( _TreeView );
		}

		void Form_Activated( object sender, EventArgs e )
		{
			_TreeViewBrailler.UpdateDisplay();
		}
		#endregion

		#region Properties
		/// <summary>
		/// アウトライン上でユーザが選択したシンボルがある、テキスト全体でのインデックスを取得します。
		/// </summary>
		public int SelectedSymbolIndex
		{
			get{ return _SelectedSymbolIndex; }
			set{ _SelectedSymbolIndex = value; }
		}

		/// <summary>
		/// アウトラインを表示するツリービュー
		/// </summary>
		public TreeView TreeView
		{
			get{ return _TreeView; }
		}
		#endregion

		#region UI behavior customization
		/// <summary>
		/// オーナーウィンドウの中央に適当なサイズでモーダル表示する
		/// </summary>
		public new DialogResult ShowDialog( IWin32Window owner )
		{
			// 作者の個人的観点から適当と思うサイズを決定
			Width = 640;
			if( SystemInformation.WorkingArea.Height < 600 )
				Height = SystemInformation.WorkingArea.Height;
			else
				Height = 600;

			// ダイアログを表示
			return base.ShowDialog( owner );
		}

		/// <summary>
		/// Ctrl+W と Escape で閉じられるように
		/// </summary>
		protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
		{
			const int WM_KEYDOWN = 0x100;
			
			// close window when Ctrl+W pressed
			if( msg.Msg == WM_KEYDOWN )
			{
				if( keyData == (Keys.Control | Keys.W) ) // Ctrl+W
					this.Close();
				if( keyData == (Keys.Escape) ) // Escape
					this.Close();
			}

			return base.ProcessCmdKey( ref msg, keyData );
		}
		#endregion

		#region Tree view control
		void TreeView_DoubleClick( object sender, EventArgs e )
		{
			Decide();
		}

		void TreeView_KeyDown( object sender, KeyEventArgs e )
		{
			if( e.KeyCode == Keys.Enter )
			{
				Decide();
			}
		}

		void Decide()
		{
			if( _TreeView.SelectedNode.Tag == null )
			{
				// may be un-jumpable symbol.
				// in C++ source, there can be no class def/decl but may exist member functions.
				// in such case, dummy node of the owner class of the function would be generated.
				// such class node has no proper location to jump to.
				Sgry.Win32.MessageBeep_Notify();
				return;
			}

			try
			{
				SelectedSymbolIndex = (Int32)_TreeView.SelectedNode.Tag;
			}
			catch( InvalidCastException )
			{
				Sgry.Win32.MessageBeep_Notify();
				return;
			}

			this.DialogResult = DialogResult.OK;
			this.Close();
		}
		#endregion

		#region UI Component Initializations
		/// <summary>
		/// ダイアログで使用する点字表示クライアントを初期化します。
		/// </summary>
		protected override void InstallBraillers()
		{
			_TreeViewBrailler.Init();
			_TreeViewBrailler.Add( _TreeView );
		}

		/// <summary>
		/// ダイアログで使用した点字表示クライアントのリソースを解放します。
		/// </summary>
		protected override void UninstallBraillers()
		{
			_TreeViewBrailler.Dispose();
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			_TreeViewReader.Add( _TreeView );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void UninstallReaders()
		{
			_TreeViewReader.Dispose();
		}
		#endregion

		#region UI Components
		TreeView _TreeView	= new TreeView();
		TreeViewBrailler _TreeViewBrailler = new TreeViewBrailler();
		TreeViewReader	_TreeViewReader = new TreeViewReader();
		#endregion
	}
}
