// file: OpenFileDialog.cs
// brief: "Open file" dialog
// encoding: UTF-8
// update: 2010-01-24
//=========================================================
using System;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;
using CultureInfo = System.Globalization.CultureInfo;

namespace Sgry.AiBTools.AiBEdit
{
	using AT;
	using Gui;

	/// <summary>
	/// ファイルオープン ダイアログ
	/// </summary>
	class OpenFileDialog : AiBFileDialog
	{
		bool	_WithBom = true;

		#region Property
		/// <summary>
		/// ダイアログ上で選択されたエンコーディング
		/// </summary>
		public Encoding Encoding
		{
			get
			{
				int selectedIndex = _EncodingComboBox.SelectedIndex;
				if( selectedIndex == 1 )
				{
					return Encoding.GetEncoding("shift_jis");
				}
				else if( selectedIndex == 2 )
				{
					return Encoding.GetEncoding("iso-2022-jp");
				}
				else if( selectedIndex == 3 )
				{
					return Encoding.GetEncoding("euc-jp");
				}
				else if( selectedIndex == 4 )
				{
					return Encoding.UTF8;
				}
				else if( selectedIndex == 5 )
				{
					return Encoding.UTF8;
				}
				else if( selectedIndex == 6 )
				{
					return Encoding.Unicode;
				}
				else if( selectedIndex == 7 )
				{
					return Encoding.BigEndianUnicode;
				}
				else
				{
					return null;
				}
			}
			set
			{
				if( value.WebName == "shift_jis" )
				{
					_EncodingComboBox.SelectedIndex = 1;
				}
				else if( value.WebName == "iso-2022-jp" )
				{
					_EncodingComboBox.SelectedIndex = 2;
				}
				else if( value.WebName == "euc-jp" )
				{
					_EncodingComboBox.SelectedIndex = 3;
				}
				else if( value == Encoding.UTF8 && _WithBom )
				{
					_EncodingComboBox.SelectedIndex = 4;
				}
				else if( value == Encoding.UTF8 && _WithBom == false )
				{
					_EncodingComboBox.SelectedIndex = 5;
				}
				else if( value == Encoding.Unicode )
				{
					_EncodingComboBox.SelectedIndex = 6;
				}
				else if( value == Encoding.BigEndianUnicode )
				{
					_EncodingComboBox.SelectedIndex = 7;
				}
				else
				{
					_EncodingComboBox.SelectedIndex = 0;
				}
			}
		}

		/// <summary>
		/// ダイアログ上で選択された BOM の有無
		/// </summary>
		public bool WithBom
		{
			get
			{
				if( _EncodingComboBox.SelectedIndex == 5
					|| _EncodingComboBox.SelectedIndex == 6
					|| _EncodingComboBox.SelectedIndex == 7 )
				{
					return true;
				}
				return false;
			}
			set
			{
				_WithBom = value;
				if( this.Encoding == Encoding.UTF8 && _WithBom == false )
				{
					_EncodingComboBox.SelectedIndex = 4;
				}
				else if( this.Encoding == Encoding.UTF8 && _WithBom )
				{
					_EncodingComboBox.SelectedIndex = 5;
				}
			}
		}
		#endregion

		#region Dialog Customization
		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダを初期化します。
		/// </summary>
		protected override void InstallReaders()
		{
			base.InstallReaders();

			_ComboBoxReader.Add( _EncodingComboBox );
		}

		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void InstallBraillers()
		{
			base.InstallBraillers();

			_ComboBoxBrailler.Add( _EncodingComboBox );
		}

		/// <summary>
		/// ダイアログの内容を確認して、その内容で決定して良いか検証。
		/// </summary>
		/// <param name="fullPath">選択されたファイルのパス</param>
		/// <returns>その内容で決定して良い場合はtrue、良くない場合はfalse</returns>
		protected override bool ValidateFileName( string fullPath )
		{
			const int big_file_size = 4000000; // 4MB = 4000 * 1000
			DialogResult	result;
			string			format;

			// if its not exists, confirm to newly create or not
			if( File.Exists(fullPath) == false )
			{
				if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
					format = "{0} は存在しません。新しく作成しますか？";
				else
					format = "{0} does not exists. Create it?";

				result = NbsMessageBox.Show( this,
						String.Format(format, Path.GetFileName(fullPath)),
						MessageBoxButtons.YesNo
					);
				if( result != DialogResult.Yes )
				{
					return false; // cancel to create file newly
				}
			}
			// if its a large file, confirm its ok to open
			else if( big_file_size < new FileInfo(fullPath).Length )
			{
				if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
					format = "{0} はサイズが大きいので開くのに時間がかかります。本当に開きますか？";
				else
					format = "{0} is a big file so it would take a long time to be opened. Are you sure to open it?";

				result = NbsMessageBox.Show( this,
						String.Format(format, Path.GetFileName(fullPath)),
						MessageBoxButtons.YesNo
					);
				if( result != DialogResult.Yes )
				{
					return false; // cancel to open the big file
				}
			}

			return true;
		}

		protected override void SetupCustomPanel()
		{
			_CustomPanel.Controls.AddRange( new Control[]{
				_EncodingLabel, _EncodingComboBox
			} );
			// 
			// _EncodingLabel
			// 
			_EncodingLabel.Anchor = AnchorStyles.Top | AnchorStyles.Left;
			_EncodingLabel.Location = new Point( 0, 3 );
			_EncodingLabel.Name = "_EncodingLabel";
			_EncodingLabel.Size = new Size( 135, 20 );
			_EncodingLabel.Text = "En&coding";
			// 
			// _EncodingComboBox
			// 
			_EncodingComboBox.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
			_EncodingComboBox.DropDownStyle = ComboBoxStyle.DropDownList;
			_EncodingComboBox.Items.AddRange( new string[]{
				"auto detect",
				"Shift_JIS",
				"JIS",
				"EUC-JP",
				"UTF-8",
				"UTF-8 (with BOM)",
				"UTF-16",
				"UTF-16 (Big Endian)"
			});
			_EncodingComboBox.Location = new Point( 136, 0 );
			_EncodingComboBox.Name = "_EncodingComboBox";
			_EncodingComboBox.Size = new Size( 324, 20 );
			_EncodingComboBox.SelectedIndex = 0;

			_CustomPanel.Height = _EncodingComboBox.Bottom;
		}

		protected override void LocalizeComponent()
		{
			base.LocalizeComponent();

			Localizer.TryGetString( "AiBFileDialog._EncodingLabel.Text", _EncodingLabel );

			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				_OkButton.Text = "開く(&O)";
				_CancelButton.Text = "キャンセル";
				Text = "ファイルを開く";
			}
			else
			{
				_OkButton.Text = "&Open";
				Text = "Open file";
			}
		}
		#endregion

		#region UI Components
		Label				_EncodingLabel		= new Label();
		protected ComboBox	_EncodingComboBox	= new ComboBox();
		#endregion
	}
}
