﻿// file: ExportFileOptionDialog.cs
// brief: dialog to let users specify options for exporting braille file.
// author: SGRY (YAMAMOTO Suguru)
// update: 2010-02-07
//=========================================================
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Diagnostics;

namespace Sgry.AiBTools.AiBEdit
{
	using Sgry.AiBTools.AT;
	using Sgry.AiBTools.Gui;

	partial class ExportFileOptionDialog : AiBDialogBase
	{
		#region Fields
		ControlBrailler _ControlBrailler = new ControlBrailler();
		TextBoxBrailler _TextBoxBrailler = new TextBoxBrailler();
		ComboBoxBrailler _ComboBoxBrailler = new ComboBoxBrailler();
		TextBoxReader _TextBoxReader = new TextBoxReader();
		CheckBoxReader _CheckBoxReader = new CheckBoxReader();
		ComboBoxReader _ComboBoxReader = new ComboBoxReader();
		#endregion

		#region Resources
		string _Text = "Conversion option";
		string _Msg_MustBeInteger = "{0} must be between {1} and {2}.";
		#endregion

		#region Init / Dispose
		/// <summary>
		/// 新しいインスタンスを生成します。
		/// </summary>
		public ExportFileOptionDialog()
		{
			InitializeComponent();

			// ローカライズ
			AppLogic.Localizer.Localize( this );
			this.Text = _Text;
			this.Font = SystemInformation.MenuFont;

			// イベントハンドラを設定
			_UsePageNumCheck.CheckedChanged += delegate {
				UpdateControlAvailability();
			};

			// GUI の有効無効を更新
			UpdateControlAvailability();
		}
		#endregion

		#region Property
		/// <summary>
		/// 点字変換パラメータを取得します。
		/// </summary>
		public BrlCvtParam ConversionParam
		{
			get{  unsafe{ return GetConversionParam(); }  }
			set{  unsafe{ SetConversionParam( value ); }  }
		}

		/// <summary>
		/// ダイアログをこのファイル形式に合わせて最適化します。
		/// </summary>
		public FileFormat FileFormat
		{
			set
			{
				// ブレイルスター３形式の場合だけ「表示文字」を有効に
				if( value == FileFormat.BS3 )
				{
					_DispCharTypeLabel.Enabled
						= _DispCharTypeCombo.Enabled = true;
				}
				else
				{
					_DispCharTypeLabel.Enabled
						= _DispCharTypeCombo.Enabled = false;
				}
			}
		}

		unsafe BrlCvtParam GetConversionParam()
		{
			BrlCvtParam value = BrlCvtParam.Default;
			
			try
			{
				value.maxColumn = Int32.Parse( _CplEdit.Text );
				value.maxLine = Int32.Parse( _LppEdit.Text );
				value.usePageNum = (_UsePageNumCheck.Checked ? 1 : 0);
				value.startPageNum = Int32.Parse( _FirstPageNumEdit.Text );
				switch( _EnglishCharTypeCombo.SelectedIndex )
				{
					case 1:
						value.alnumCharType = AlnumCharType.JohoSmall;
						break;
					case 2:
						value.alnumCharType = AlnumCharType.JohoLarge;
						break;
					case 3:
						value.alnumCharType = AlnumCharType.Grade1;
						break;
					case 4:
						value.alnumCharType = AlnumCharType.Grade2;
						break;
					case 5:
						value.alnumCharType = AlnumCharType.NABCC;
						break;
					default:
						value.alnumCharType = AlnumCharType.JohoNatural;
						break;
				}
				value.jaCharType = (JaCharType)_JapaneseCharTypeCombo.SelectedIndex;
				value.tenjiMode = (TenjiMode)_DispCharTypeCombo.SelectedIndex;
			}
			catch( FormatException ex )
			{
				NbsMessageBox.Show(
					this, ex.Message,
					MessageBoxButtons.OK, MessageBoxIcon.Error
				);
			}

			return value;
		}

		unsafe void SetConversionParam( BrlCvtParam value )
		{
			_CplEdit.Text = value.maxColumn.ToString();
			_LppEdit.Text = value.maxLine.ToString();
			_UsePageNumCheck.Checked = (value.usePageNum != 0);
			_FirstPageNumEdit.Text = value.startPageNum.ToString();
			switch( value.alnumCharType )
			{
				case AlnumCharType.JohoSmall:
					_EnglishCharTypeCombo.SelectedIndex = 1;
					break;
				case AlnumCharType.JohoLarge:
					_EnglishCharTypeCombo.SelectedIndex = 2;
					break;
				case AlnumCharType.Grade1:
					_EnglishCharTypeCombo.SelectedIndex = 3;
					break;
				case AlnumCharType.Grade2:
					_EnglishCharTypeCombo.SelectedIndex = 4;
					break;
				case AlnumCharType.NABCC:
					_EnglishCharTypeCombo.SelectedIndex = 5;
					break;
				default:
					_EnglishCharTypeCombo.SelectedIndex = 0;
					break;
			}
			_JapaneseCharTypeCombo.SelectedIndex = (int)value.jaCharType;
			_DispCharTypeCombo.SelectedIndex = (int)value.tenjiMode;
		}
		#endregion

		#region AT
		protected override void InstallBraillers()
		{
			_ControlBrailler.Init();
			_TextBoxBrailler.Init();
			_ComboBoxBrailler.Init();

			_TextBoxBrailler.Add( _CplEdit );
			_TextBoxBrailler.Add( _LppEdit );
			_ControlBrailler.Add( _UsePageNumCheck );
			_TextBoxBrailler.Add( _FirstPageNumEdit );
			_ComboBoxBrailler.Add( _EnglishCharTypeCombo );
			_ComboBoxBrailler.Add( _JapaneseCharTypeCombo );
			_ComboBoxBrailler.Add( _DispCharTypeCombo );
			_ControlBrailler.Add( _OkButton );
			_ControlBrailler.Add( _CancelButton );
		}

		protected override void InstallReaders()
		{
			_TextBoxReader.Add( _CplEdit );
			_TextBoxReader.Add( _LppEdit );
			_CheckBoxReader.Add( _UsePageNumCheck );
			_TextBoxReader.Add( _FirstPageNumEdit );
			_ComboBoxReader.Add( _EnglishCharTypeCombo );
			_ComboBoxReader.Add( _JapaneseCharTypeCombo );
			_ComboBoxReader.Add( _DispCharTypeCombo );
		}

		protected override void UninstallBraillers()
		{
			_ControlBrailler.Dispose();
			_TextBoxBrailler.Dispose();
			_ComboBoxBrailler.Dispose();
		}

		protected override void UninstallReaders()
		{
			_TextBoxReader.Dispose();
			_CheckBoxReader.Dispose();
			_ComboBoxReader.Dispose();
		}
		#endregion

		#region Event Handlers
		void _OkButton_Click( object sender, EventArgs e )
		{
			if( IsValidInput() == false )
			{
				return;
			}

			this.DialogResult = DialogResult.OK;
			this.Close();
		}

		void _FoldingLineNumbersEdit_GotFocus( object sender, System.EventArgs e )
		{
			((TextBox)sender).SelectAll();
		}

		void UpdateControlAvailability()
		{
			_FirstPageNumEdit.Enabled = _UsePageNumCheck.Checked;
		}

		bool IsValidInput()
		{
			try
			{
				ValidateInput_Int( _CplEdit, _CplLabel, 15, 45 );
				ValidateInput_Int( _LppEdit, _LppLabel, 4, 64 );
				ValidateInput_Int( _FirstPageNumEdit, _FirstPageNumLabel, 1, 10000 );

				return true;
			}
			catch( Exception ex )
			{
				MessageBoxIcon icon;

				// メッセージの警告レベルを決定
				if( ex is ApplicationException )
				{
					icon = MessageBoxIcon.Warning;
				}
				else
				{
					icon = MessageBoxIcon.Error;
				}
				
				// メッセージを表示
				NbsMessageBox.Show( this, ex.Message, MessageBoxButtons.OK, icon );

				return false;
			}
		}

		void ValidateInput_Int( Control target, Label label, int min, int max )
		{
			int value;
			string msg;
			bool ok;

			ok = Int32.TryParse( target.Text, out value );
			if( !ok || value < min || max < value )
			{
				// 検証対象コントロールへフォーカスを移動
				target.Focus();
				if( target is TextBox )
				{
					((TextBox)target).SelectAll();
				}

				// 例外を投げる
				msg = String.Format(
						_Msg_MustBeInteger,
						Utl.RemoveMnemonic(label.Text),
						min, max
					);
				throw new ApplicationException( msg );
			}
		}
		#endregion

		#region Utilities
		class Utl
		{
			/// <summary>
			/// ラベル文字列からニーモニックを削除します。
			/// </summary>
			public static string RemoveMnemonic( string text )
			{
				string label, mnemonic;

				ParseMnemonic( text, out label, out mnemonic );

				return label;
			}

			/// <summary>
			/// ラベル文字列をラベルとニーモニックに分離
			/// </summary>
			public static void ParseMnemonic( string text, out string label, out string mnemonic )
			{
				Match match;

				// 末尾が (&?) ？
				match = Regex.Match( text, @"\(&(.)\)$" );
				if( match.Success )
				{
					label = text.Substring( 0, text.Length - 4 );
					mnemonic = match.Groups[1].ToString();

					return;
				}

				// 単独の & ？
				match = Regex.Match( text, "&([^&])" );
				if( match.Success )
				{
					int mnemonicPos = match.Groups[1].Index;
					label = text.Substring( 0, mnemonicPos )
						+ text.Substring( mnemonicPos + 1, text.Length - mnemonicPos - 1 );
					mnemonic = match.Groups[1].ToString();

					return;
				}

				label = text;
				mnemonic = null;
			}
		}
		#endregion
	}

}
