// 2010-01-31
using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using CultureInfo = System.Globalization.CultureInfo;
using FileTypeComboBoxItem = Sgry.AiBTools.Gui.MyComboBoxItem<string>;

namespace Sgry.AiBTools.AiBEdit
{
	using Sgry.AiBTools.AT;
	using Sgry.AiBTools.Gui;

	class ExportFileDialog : AiBFileDialog
	{
		string	_FilterString	= FilterString_en;
		const string	FilterString_en	= "NABCC format (*.brl)|*.brl|Braille Star 3 (*.bs)|*.bs|BASE format (*.bse)|*.bse|IBM-BES format (*.bes)|*.bes";
		const string	FilterString_ja	= "NABCC形式 (*.brl)|*.brl|ブレイルスターⅢ形式 (*.bs)|*.bs|ＢＡＳＥ形式 (*.bse)|*.bse|ＩＢＭ－ＢＥＳ形式 (*.bes)|*.bes";
		
		#region Property
		/// <summary>
		/// 点字ファイルフォーマット
		/// </summary>
		public FileFormat FileFormat
		{
			set
			{
				switch( value )
				{
					case FileFormat.BS3:	FilterIndex = 1;	return;
					case FileFormat.BASE:	FilterIndex = 2;	return;
					case FileFormat.BES:	FilterIndex = 3;	return;
					default:				FilterIndex = 0;	return;
				}
			}
			get
			{
				switch( FilterIndex )
				{
					case 1:		return FileFormat.BS3;
					case 2:		return FileFormat.BASE;
					case 3:		return FileFormat.BES;
					default:	return FileFormat.EX_NABCC;
				}
			}
		}

		/// <summary>
		/// ファイル変換オプション
		/// </summary>
		public BrlCvtParam ConversionParam
		{
			get{ return AppConfig.BrlCvtParams[FileFormat]; }
		}
		#endregion

		#region Dialog Customization
		/// <summary>
		/// ダイアログのコントロールを読み上げる内部スクリーンリーダのリソースを解放します。
		/// </summary>
		protected override void InstallBraillers()
		{
			base.InstallBraillers();

			_ControlBrailler.Add( _ConvOptionButton );
		}

		/// <summary>
		/// ダイアログの内容を確認して、その内容で決定して良いか検証。
		/// </summary>
		/// <param name="path">選択されたファイルのパス</param>
		/// <returns>その内容で決定して良い場合はtrue、良くない場合はfalse</returns>
		protected override bool ValidateFileName( string path )
		{
			DialogResult	result;
			string			format;

			// 同名ファイルが存在しなければ何もしない
			if( File.Exists(path) != true )
			{
				return true;
			}

			// 同名ファイルが存在するので、上書きするか確認する
			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				format = "{0} は既に存在します。上書きしますか？";
			}
			else
			{
				format = "{0} already exists. overwrite?";
			}
			result = NbsMessageBox.Show( this,
					String.Format(format, Path.GetFileName(path)),
					MessageBoxButtons.YesNo
				);
			if( result != DialogResult.Yes )
			{
				return false; // 上書きキャンセルされた
			}

			return true;
		}

		protected override void SetupCustomPanel()
		{
			_CustomPanel.Controls.AddRange( new Control[]{
				_ConvOptionButton
			} );
			// 
			// _ConvOptionButton
			// 
			_ConvOptionButton.Anchor = AnchorStyles.Top | AnchorStyles.Left;
			_ConvOptionButton.Location = new Point( 136, 0 );
			_ConvOptionButton.Name = "_ConvOptionButton";
			_ConvOptionButton.Size = new Size( 120, 24 );
			_ConvOptionButton.Text = "&Option...";
			_ConvOptionButton.Click += _ConvOptionButton_Click;

			_CustomPanel.Height = _ConvOptionButton.Bottom;
		}

		protected override void LocalizeComponent()
		{
			base.LocalizeComponent();

			Localizer.TryGetString( "ExportFileDialog._ConvOptionButton.Text", _ConvOptionButton );

			if( CultureInfo.CurrentUICulture.Name.StartsWith("ja") )
			{
				_OkButton.Text = "保存(&S)";
				_CancelButton.Text = "キャンセル";
				_FilterString = FilterString_ja;
				Text = "ファイルのエクスポート";
			}
			else
			{
				_OkButton.Text = "&Save";
				Text = "Export file as";
			}

			base.Filter = _FilterString;
		}
		#endregion

		#region UI Event Handlers
		void _ConvOptionButton_Click( object sender, EventArgs e )
		{
			ShowExportOptionDialog();
		}

		void ShowExportOptionDialog()
		{
			ExportFileOptionDialog dialog;
			DialogResult result;

			using( dialog = new ExportFileOptionDialog() )
			{
				// オプションダイアログを準備
				dialog.StartPosition = FormStartPosition.CenterParent;
				dialog.ConversionParam = AppConfig.BrlCvtParams[FileFormat];
				dialog.FileFormat = this.FileFormat;

				// 表示
				result = dialog.ShowDialog( this );
				if( result != DialogResult.OK )
				{
					return;
				}

				// 設定内容を反映
				AppConfig.BrlCvtParams[FileFormat] = dialog.ConversionParam;
			}
		}
		#endregion

		#region UI Components
		Button		_ConvOptionButton	= new Button();
		#endregion
	}
}
