// file: Document.cs
// brief: document class for AiB Edit
// update: 2010-04-11
//=========================================================
using System;
using System.IO;
using System.Collections.Generic;
using Encoding = System.Text.Encoding;
using IO_Path = System.IO.Path;
using Debug = System.Diagnostics.Debug;

namespace Sgry.AiBTools.AiBEdit
{
	using Gui;

	/// <summary>
	/// ҏWΏۃt@CɊւ鏈sNXB
	/// ʓIɃhLgNX UI ƓƗ̂
	/// RichEdit  UI ƃhLg؂藣ĎgȂB
	/// ̂ UI ̊TÕNXɂ͊܂܂ĂB
	/// </summary>
	class Document : IDisposable
	{
		#region Fields
		bool		_IsNewFile	= true;
		string		_Path		= null;
		string		_DisplayName= null;
		DateTime	_LastSaveTime;
		Encoding	_Encoding	= Encoding.Default;
		bool		_WithBom	= true;
		EolCode		_EolCode	= EolCode.CRLF;
		ITextEditor	_Editor;
		IFileType	_FileType;
		#endregion

		#region Init / Dispose
		/// <summary>
		/// \z
		/// </summary>
		public Document()
		{
			//DEBUG//Console.WriteLine( "Document.ctor ({0})", this.GetHashCode() );
			_Editor = TextEditorFactory.CreateEditor( AppConfig.EngineType );
			_Editor.IsDirty = false;
			_Editor.Tag = this;

			// ҏW̊ĎJn
			_Editor.ContentChanged += OnTextChanged;
		}

#		if DEBUG
		~Document()
		{
			// Dispose Rh~
			if( _Editor != null )
			{
				Debug.Fail( "Document ["+DisplayName+"] was destroyed but not disposed!" );
			}
		}
#		endif

		/// <summary>
		/// IuWFNgj邽߂̌㏈s
		/// </summary>
		public void Dispose()
		{
			Debug.Assert( _Editor != null );
			//DEBUG//Console.WriteLine( "Document.Dispose ({0})", this.GetHashCode() );

			_Editor.ContentChanged -= OnTextChanged;
			_Editor.Dispose();
			_Editor = null;
			_Path = null;

			// LvZXf[^炱̃hLg̍ڂ폜
			SharedData.RemoveProcessData( Path );
		}
		#endregion

		#region Properties
		/// <summary>
		/// ֘AtꂽGfB^i
		/// </summary>
		public ITextEditor Editor
		{
			get{ return _Editor; }
		}

		/// <summary>
		/// Ōɕۑꂽ擾
		/// </summary>
		public DateTime LastSaveTime
		{
			get{ return _LastSaveTime; }
		}
		
		/// <summary>
		/// t@C̓eύXꂽǂ\tO
		/// </summary>
		public bool IsModified
		{
			get
			{
				if( _Editor == null )
					throw new InvalidOperationException( "associated editor object was already disposed." );
				return _Editor.IsDirty;
			}
			set{ _Editor.IsDirty = value; }
		}

		/// <summary>
		/// ㏑[hǂ擾܂͐ݒ肵܂B
		/// </summary>
		public bool IsOverwriteMode
		{
			get{ return _Editor.IsOverwriteMode; }
			set{ _Editor.IsOverwriteMode = value; }
		}

		/// <summary>
		/// VK쐬ꂽhLgŁA
		/// t@CƊ֘AtĂȂǂ擾܂͐ݒ肵܂B
		/// </summary>
		public bool IsNewFile
		{
			get{ return _IsNewFile; }
			set{ _IsNewFile = value; }
		}

		/// <summary>
		/// ҏWΏۃt@C̃pX擾܂B
		/// L}̏̃t@CɊ֘AtĂȂꍇ null ɂȂ܂B
		/// Ȃ󕶎ɂȂ邱Ƃ͂܂B
		/// </summary>
		public string Path
		{
			get
			{
				Debug.Assert( _Path != String.Empty );
				return _Path;
			}
		}

		/// <summary>
		/// ނ̕\擾܂͐ݒ肵܂B
		/// \ݒ肳ĂȂꍇ
		/// t@CɂȂ܂B
		/// </summary>
		public string DisplayName
		{
			get
			{
				if( _DisplayName != null )
					return _DisplayName;
				else if( _Path != null )
					return FileName;
				else
					return base.ToString();
			}
			set{ _DisplayName = value; }
		}

		/// <summary>
		/// ҏWΏۃt@C̃t@C
		/// </summary>
		public string FileName
		{
			get{ return IO_Path.GetFileName(Path); } // PathȂ󕶎Ԃ
		}

		/// <summary>
		/// ҏWΏۃt@C̃t@C
		/// </summary>
		public IFileType FileType
		{
			get{ return _FileType; }
			set
			{
				_FileType = value;
				Editor.SetFileType( _FileType.Name, AppConfig.AutoIndentMode );
			}
		}

		/// <summary>
		/// ҏW̓e
		/// </summary>
		public string Text
		{
			get{ return _Editor.Text; }
		}

		/// <summary>
		/// ҏW̓e̒
		/// </summary>
		public int TextLength
		{
			get{ return _Editor.TextLength; }
		}

		/// <summary>
		/// t@C̃GR[fBO擾
		/// </summary>
		public Encoding Encoding
		{
			get{ return _Encoding; }
		}

		/// <summary>
		/// t@C Byte Order Mark tĂ邩ǂ
		/// </summary>
		public bool WithBom
		{
			get{ return _WithBom; }
		}

		/// <summary>
		/// t@C̉sɎgR[h
		/// </summary>
		public EolCode EolCode
		{
			get{ return _EolCode; }
		}

		/// <summary>
		/// LbgʒũCfbNX擾
		/// </summary>
		public int GetCaretIndex()
		{
			return _Editor.GetCaretIndex();
		}

		/// <summary>
		/// Lbg̈ʒu擾
		/// </summary>
		/// <param name="lineIndex">Lbg̍sCfbNX</param>
		/// <param name="columnIndex">LbǧCfbNX</param>
		public void GetCaretIndex( out int lineIndex, out int columnIndex )
		{
			_Editor.GetCaretIndex( out lineIndex, out columnIndex );
		}

		/// <summary>
		/// Lbgʒuݒ
		/// </summary>
		/// <param name="lineIndex">Lbgړ̍sCfbNX</param>
		/// <param name="columnIndex">Lbgړ̌CfbNX</param>
		public void SetCaretIndex( int lineIndex, int columnIndex )
		{
			_Editor.SetCaretIndex( lineIndex, columnIndex );
		}

		/// <summary>
		/// ނ̕\擾܂B
		/// ύXς݂̏ꍇ̓AX^XNɕt܂B
		/// </summary>
		public override string ToString()
		{
			string text = DisplayName;
			if( _Editor != null && IsModified )
			{
				text += "*";
			}

			return text;
		}
		#endregion

		#region Open
		/// <summary>
		/// t@CJēǂݍށB
		/// t@C݂ȂΕۑpX̐ݒ肾sB
		/// ܂AGR[fBOw肪 null Ȃ΃GR[fBO𐄒肷B
		/// </summary>
		/// <param name="filePath">Jt@C̃pX</param>
		/// <param name="encoding">t@C̕GR[fBO</param>
		/// <param name="withBom">t@C̐擪 Byte Order Mark tĂ邩ǂ</param>
		/// <exception cref="IOException">t@CJ I/O G[B</exception>
		public void Open( string filePath, Encoding encoding, bool withBom )
		{
			// w肪Ȃ΃GR[fBO𐄒
			if( encoding == null )
			{
				encoding = EncodingAnalyzer.Analyze( filePath, out withBom );
				if( encoding == null )
				{
					encoding = Encoding.Default; // łȂ̂ŃVXeW
				}
			}

			// ݂t@CJȂAeǂݏo
			if( File.Exists(filePath) )
			{
				OpenExistingFile( filePath, encoding, withBom );
			}
			// Vt@CȂAp[^ۑ̂
			else
			{
				_Path = filePath;
				_LastSaveTime = DateTime.Now;
				_Encoding = encoding;
				_WithBom = withBom;
			}

			// Lf[^ɃvZX
			SharedData.AddProcessData( filePath );
			
			IsModified = false;
		}

		/// <summary>
		/// łɑ݂t@CJ܂B
		/// </summary>
		/// <exception cref="IOException">t@CJ I/O G[B</exception>
		void OpenExistingFile( string filePath, Encoding encoding, bool withBom )
		{
			Debug.Assert( filePath != null && encoding != null && File.Exists(filePath) );
			int line, col;
			byte[] rawContent;
			string content;

			GetCaretIndex( out line, out col );

			// eׂēǂݏo
			rawContent = File.ReadAllBytes( filePath );
			content = encoding.GetString( rawContent );

			// eLXg{bNXɐݒ
			Editor.Text = content;
			Editor.SetCaretIndex( line, col ); // restore caret pos
			Editor.ClearUndo();

			// ǂݍ݂ɐep[^ۑ
			_IsNewFile = false;
			_Path = Utl.ReGetFilePath( filePath );
			_LastSaveTime = File.GetLastWriteTime( _Path );
			_Encoding = encoding;
			_WithBom = withBom;
			_EolCode = Utl.AnalyzeEolCode( content );
		}
		#endregion

		#region Reload
		/// <summary>
		/// t@Cǂݒ
		/// </summary>
		/// <param name="encoding">ǂݒۂɎgGR[fBO</param>
		/// <param name="withBom">t@C̐擪 Byte Order Mark tĂ邩ǂ</param>
		/// <exception cref="IOException">ǂݒɎs</exception>
		/// <exception cref="UnauthorizedAccessException">ANZXۂꂽ</exception>
		public void Reload( Encoding encoding, bool withBom )
		{
			int line, col;
			byte[] rawContent;
			string content;

			try
			{
				GetCaretIndex( out line, out col );

				// GR[fBOw肳ĂȂȂ琄
				if( encoding == null )
				{
					encoding = EncodingAnalyzer.Analyze( Path, out withBom );
					if( encoding == null )
					{
						encoding = Encoding.Default; // łȂ̂ŃVXeW
					}
				}

				// eׂēǂݏo
				rawContent = File.ReadAllBytes( Path );
				content = encoding.GetString( rawContent );

				// eLXg{bNXɐݒ
				Editor.Text = content;
				Editor.SetCaretIndex( line, col ); // restore caret pos
				Editor.ClearUndo();

				// ǂݍ݂ɐep[^ۑ
				_Path = Utl.ReGetFilePath( Path );
				_LastSaveTime = File.GetLastWriteTime( _Path );
				_Encoding = encoding;
				_WithBom = withBom;
				_EolCode = Utl.AnalyzeEolCode( content );

				// ύXς݃tO낷
				IsModified = false;
			}
			catch( NotSupportedException ex )
			{
				Debug.Fail( "Unexpected exception has been thrown.", ex.ToString() );
				throw;
			}
		}
		#endregion

		#region Save
		/// <summary>
		/// t@Cۑ
		/// </summary>
		/// <param name="filePath">ۑ̃pX</param>
		/// <param name="encoding">ۑt@C̃GR[fBO</param>
		/// <param name="eolCode">ۑɗpsR[h</param>
		/// <param name="withBom">Byte Order Marktăt@Cۑ邩ǂ</param>
		/// <exception cref="System.UnauthorizedAccessException">
		/// ANZXۂꂽit@CǂݎpȂǁj
		/// </exception>
		/// <exception cref="System.Security.SecurityException">
		/// t@CރANZX
		/// </exception>
		/// <exception cref="System.IO.IOException">
		/// t@C݂̏ɎsBvZXt@CgpȂǁB
		/// </exception>
		public void Save( string filePath, Encoding encoding, bool withBom, EolCode eolCode )
		{
			// p[^`FbN
			if( encoding == null )
			{
				encoding = Encoding.Default;
			}

			// t@Co
			Utl.WriteToFile( _Editor, filePath, encoding, eolCode, withBom );

			// ޏXV
			_IsNewFile = false;
			_LastSaveTime = DateTime.Now;
			_Encoding = encoding;
			_WithBom = withBom;
			_EolCode = eolCode;
			_DisplayName = null;
			IsModified = false;

			// Lf[^t@CXV
			SharedData.RemoveProcessData( filePath );
			_Path = Utl.ReGetFilePath( filePath ); // pXĎ擾ăfBNg̑啶ȂǂC
			SharedData.AddProcessData( filePath );
		}
		#endregion

		#region Event
		public event EventHandler	ContentModified;
		#endregion

		#region 
		/// <summary>
		/// ҏWΏۂҏW邽тɌĂ΂鏈B
		/// ҏWς݃tO𗧂ĂB
		/// </summary>
		void OnTextChanged( object sender, EventArgs e )
		{
			if( IsModified != true )
			{
				IsModified = true;
				OnContentModified( this, EventArgs.Empty );
			}
		}

		/// <summary>
		/// ContentModified Cxg𔭐܂B
		/// </summary>
		void OnContentModified( object sender, EventArgs e )
		{
			if( ContentModified != null )
			{
				ContentModified( this, EventArgs.Empty );
			}
		}

		static class Utl
		{
			/// <summary>
			/// t@CVXe琳mȃpXĎ擾B
			/// </summary>
			/// <param name="filePath">mȃpXĎ擾t@C̃pX</param>
			/// <returns>Ď擾pX</returns>
			/// <remarks>
			/// Windows ł͑啶ʂƕsȃpXłgĂ܂B
			/// 啶̕ʂmɎ擾邽߂ɌĂԁB
			/// </remarks>
			/// <exception cref="System.IO.IOException">I/OG[</exception>
			public static string ReGetFilePath( string filePath )
			{
				Debug.Assert( filePath != null, "argument null" );

				string dir, filter;

				// ݂Ȃꍇ͂̂܂
				if( File.Exists(filePath) == false )
				{
					return filePath;
				}

				dir = IO_Path.GetDirectoryName( filePath );
				dir = IO_Path.GetFullPath( dir );
				filter = IO_Path.GetFileName( filePath );
				return Directory.GetFiles( dir, filter )[0];
			}

			/// <summary>
			/// t@Cɓeo܂B
			/// </summary>
			/// <exception cref="IOException">oɎs܂B</exception>
			/// <exception cref="UnauthorizedAccessException">ANZXۂꂽ</exception>
			/// <exception cref="NotSupportedException">w肵pX̌`̓T|[gĂȂ</exception>
			public static void WriteToFile( ITextEditor editor, string filePath, Encoding encoding, EolCode eolCode, bool withBom )
			{
				Debug.Assert( editor != null );
				Debug.Assert( filePath != null );
				Debug.Assert( encoding != null );

				FileStream file;
				List<byte> decodedBytes = new List<byte>();
				string text;
				string dirPath;
				
				// eLXgoăfR[h
				text = editor.GetText( eolCode );
				if( withBom )
				{
					decodedBytes.AddRange( encoding.GetBytes("\xfeff") );
				}
				decodedBytes.AddRange( encoding.GetBytes(text) );

				// ۑ̃fBNg݂Ȃ΍Ă
				dirPath = IO_Path.GetDirectoryName( filePath );
				if( Directory.Exists(dirPath) == false )
				{
					Directory.CreateDirectory( dirPath );
				}

				// t@Cɏo
				using( file = File.Open(filePath, FileMode.Create, FileAccess.Write, FileShare.ReadWrite) )
				{
					file.Write( decodedBytes.ToArray(), 0, decodedBytes.Count );
					file.Close();
				}
			}

			public static string ToString( EolCode eolCode )
			{
				switch( eolCode )
				{
					case EolCode.CR:
						return "\r";

					case EolCode.LF:
						return "\n";

					default:
						return "\r\n";
				}
			}

			public static EolCode AnalyzeEolCode( String content )
			{
				for( int i=0; i<content.Length-1; i++ )
				{
					if( content[i] == '\r' )
					{
						if( content[i+1] == '\n' )
						{
							return EolCode.CRLF;
						}
						else
						{
							return EolCode.CR;
						}
					}
					else if( content[i] == '\n' )
					{
						return EolCode.LF;
					}
				}
				return EolCode.CRLF;
			}
		}
		#endregion //
	}
}
