//! @file		Sgry.h
//! @brief		SGRY's Perosnal Utilities for Debug
//   dependency	none
//   platform	Win32, C++/CLI
//! @date		2007-04-24 (SGRY)
//================================================
#ifndef __SGRY_H__
#define __SGRY_H__
#pragma once // (all library supported compiler can use "pragma once")

// prevent hundreds of warnings from Visual C++ 6.0
#ifdef _MSC_VER
#	pragma warning( disable : 4786 ) // too long identifier (over 255)
#	pragma warning( disable : 4514 ) // unreferenced inline function deleted
#	pragma warning( disable : 4512 ) // cannot to generate operator =
#	pragma warning( disable : 4996 ) // warning for old unsafe string.h functions
#endif

//
// Include Framework
//
#if defined( _WIN32 )		// Windows?
#	if defined( _MFC_VER )
#		include "StdAfx.h"			// MFC
#	else
#		include <windows.h>// Win32API
#	endif
#elif defined( __APPLE__ )	// MacOS X?
#	if defined( __CARBON__ )
#		include <Carbon/Carbon.h>	// Carbon
#	endif
#endif



//
// Include C/C++ standard libraries
//
#include <stdio.h>
#include <time.h>
#include <string>
#include <sstream>
#include <algorithm>




// include all into my original namespace
namespace Sgry {


//=============================================================================
//
//      Constants
//
//=============================================================================

//! Max length of path string
#ifdef MAX_PATH
	static const int kMaxPathLength = MAX_PATH; // Windows environment
#else
	static const int kMaxPathLength = 260; // other environment
#endif


//! Directory where the log file written
#ifdef _WIN32
	static const char* kLogPath = "C:\\log.txt";
#endif
#ifdef __CARBON__
	static const char* kLogPath = "~/Desktop/";
#endif




//=============================================================================
//
//      Types
//
//=============================================================================

//---- aliases for unsigned types ----//
typedef unsigned short		ushort;
typedef unsigned int		uint;
typedef unsigned long		ulong;
typedef unsigned char		uchar;


//---- fixed size integers ----//
// Microsoft C++?
#if defined( _MSC_VER )
	typedef unsigned __int8		uint8_t;
	typedef unsigned __int16	uint16_t;
	typedef unsigned __int32	uint32_t;
	typedef unsigned __int64	uint64_t;
	typedef __int8				int8_t;
	typedef __int16				int16_t;
	typedef __int32				int32_t;
	typedef __int64				int64_t;
#else
	typedef unsigned char		uint8_t;
	typedef unsigned short		uint16_t;
	typedef unsigned int		uint32_t;
	typedef unsigned hyper		uint64_t;
	typedef char				int8_t;
	typedef short				int16_t;
	typedef int					int32_t;
	typedef hyper				int64_t;
#endif

//
// size checking trick for fixed size integer.
// 
static union __size_checking_trick
{
	// if the size of int8_t is not 1 byte, sizeof(int8_t)==1 equals to 0(false).
	// then "int8_t_check" becomes array which size is zero.
	char    int8_t_check[sizeof(  int8_t) == 1];
	char   uint8_t_check[sizeof( uint8_t) == 1];
	char   int16_t_check[sizeof( int16_t) == 2];
	char  uint16_t_check[sizeof(uint16_t) == 2];
	char   int32_t_check[sizeof( int32_t) == 4];
	char  uint32_t_check[sizeof(uint32_t) == 4];
} __size_checking_trick;


//=============================================================================
//
//      Prototype Definitions
//
//=============================================================================
#ifndef __cplusplus_cli
inline static void LogFile( const char * inFormat, ... );
inline static void AlertWarning( const char * inFormat, ... );
#endif
#ifdef _WIN32
inline static short Reg_ReadStringA( HKEY keyRoot, const char * targetKeyName, const char * valueName, char * outStringValue, long bufferLength );
#endif


//=============================================================================
//      Path string manipulating functions
//=============================================================================

//-------------------------------------
//   function	Path_GetLocalAppDataPathA / Path_GetLocalAppDataPathW
//! @brief		Get user setting directory.
//! @param		outPath		... buffer to be written the path
//! @param		maxlength	... size of the buffer
//! @retval		 0	... success
//! @retval		-1	... failure
//-------------------------------------
#ifdef _WIN32
inline static short
Path_GetLocalAppDataPathA( long maxLength, char * outPath )
{
	return Reg_ReadStringA( HKEY_CURRENT_USER,
							"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",
							"Local AppData",
							outPath,
							maxLength );
}
#endif

//-------------------------------------
//   function	Path_GetModuleFilePath
//! @brief		Get full path of the file
//!				that contains a call of this function.
//! @returns	Path string (If failed, return default object of std::string)
//! @note		If called in standalone app., return executable file path.
//! 			If called in dynamically loaded external executable file,
//!				return path of the external executable file.
//-------------------------------------
#ifdef __AFXWIN_H__ // MFC
inline static std::string
Path_GetModuleFilePath()
{
	long rc; // return code
	char path[kMaxPathLength];

	rc = GetModuleFileName( AfxGetStaticModuleState()->m_hCurrentInstanceHandle, path, kMaxPathLength );
	if( rc == FALSE )
	{
		return std::string(); // failed
	}

	return std::string( path );
}
#endif

//-------------------------------------
//   function	Path_RemoveFileName
//! @brief		Remove file name from given path.
//! 			( example : "...xx\yy.txt"  will be "...xx\" )
//! @param		ioPath		... buffer to be written the path
//! @retval		 0	... success
//! @retval		-1	... failure
//! @note		If given path is only file name, function will fail.
//-------------------------------------
inline static short
Path_RemoveFileName( char * ioPath )
{
	char *	delimiterPos;

	// get position of the final path-delimiter char.
	delimiterPos = strrchr( ioPath, '\\' );
	if( delimiterPos == NULL )
	{
		delimiterPos = strrchr( ioPath, '/' );
		if( delimiterPos == NULL )
		{
			// no path-delimiter char found. (may be the path contains file name only)
			return -1;
		}
	}

	*(delimiterPos+1) = '\0';

	return 0;
}
inline static short
Path_RemoveFileName( std::string & ioPath )
{
	size_t	delimiterPos;

	// get position of the final path-delimiter char.
	delimiterPos = ioPath.find_last_of( '\\' );
	if( delimiterPos == std::string::npos )
	{
		delimiterPos = ioPath.find_last_of( '/' );
		if( delimiterPos == std::string::npos )
		{
			// no path-delimiter char found. (may be the path contains file name only)
			return -1;
		}
	}

	// shurink string
	ioPath.resize( delimiterPos+1 );

	return 0;
}

//-------------------------------------
//   function	Path_ReplaceFileName
//! @brief		Replace file name in given path string.
//! 			( example: "...xx\yy.txt" will be "...xx\yy.zip". )
//! @retval		 0 ... success
//! @retval		-1 ... failure
//-------------------------------------
inline static short
Path_ReplaceFileName( char * ioPath, size_t maxPathLength, const char * newName )
{
	char *	delimiterPos;

	// get position of the final path-delimiter char.
	delimiterPos = strrchr( ioPath, '\\' );
	if( delimiterPos == NULL )
	{
		delimiterPos = strrchr( ioPath, '/' );
	}

	// is the path string is only file name?
	if( delimiterPos == NULL )
	{
		// check if the operation causes buffer overflow
		if( maxPathLength < strlen(newName) )
		{
			return -1; // buffer size is insufficient
		}
		
		// replace
		strcpy( ioPath, newName );
	}
	// is not only file name?
	else
	{
		// check if the operation causes buffer overflow
		if( maxPathLength < strlen(ioPath) - strlen(delimiterPos+1) + strlen(newName) )
		{
			return -1; // buffer size is insufficient
		}

		// replace
		strcpy( delimiterPos+1, newName );
	}

	return 0;
}


//-------------------------------------
//   function	Path_RemoveFileExt
//! @brief		Remove extension of file name in given path string.
//!  			( example : "...xx\yy.txt" will be "...xx\yy". )
//! @retval		 0 ... success
//! @retval		-1 ... failure
//! @note		If path has no extension, function will fail.
//-------------------------------------
inline static short
Path_RemoveFileExt( char * ioPath )
{
	char *	dotPos;
	char *	delimiterPos;

	//---- check whether the file name has extension or not ----//
	// get position of the final dot in path.
	dotPos = strrchr( ioPath, '.' );
	if( dotPos == NULL )
	{
		return -1; // path has no extension
	}

	// get position of the final path-delimiter char.
	delimiterPos = strrchr( ioPath, '\\' );
	if( delimiterPos == NULL )
	{
		delimiterPos = strrchr( ioPath, '/' );
	}
	
	// check that '.' position is former than delimiter position
	if( dotPos < delimiterPos ) // in case of delimiter position doesn't exist, delimiter is 0(=NULL).
	{
		return -1; // no extension
	}

	//---- remove extension ----//
	*dotPos = '\0';

	return 0;
}

//-------------------------------------
//   function	Path_ReplaceFileExt
//! @brief		Replace extension of file name in the given path string.
//! 			( example : "...xx\yy.txt" will be "...xx\yy.zip". )
//! @retval		 0 ... success
//! @retval		-1 ... failure
//! @note		If path string has no extension, function will fail.
//-------------------------------------
inline static short
Path_ReplaceFileExt( char * ioPath, size_t maxPathLength, const char * newExt )
{
	char *	delimiterPos;
	char *	dotPos;
	size_t	oldExtLength;

	//---- check the buffer size sufficient ----//
	// get position of final dot
	dotPos = strrchr( ioPath, '.' );
	if( dotPos == NULL )
	{
		return -1; // no dot
	}

	// get position of final path-delimiter
	delimiterPos = strrchr( ioPath, '\\' );
	if( delimiterPos == NULL )
		delimiterPos = strrchr( ioPath, '/' );
	
	// check that '.' position is former than delimiter position
	if( dotPos < delimiterPos ) // in case of delimiter position doesn't exist, delimiter is 0(=NULL).
	{
		return -1; // no extension
	}

	// get length of current extension
	oldExtLength = strlen( dotPos );

	// is the buffer size sufficient?
	if( maxPathLength <= strlen(ioPath) - oldExtLength + strlen(newExt) )
	{
		return -1; // insufficient
	}
	
	//----- replace -----//
	strcpy( dotPos+1, newExt );

	return 0;
}
//! @overload
//! @brief (overload for std::string)
inline static short
Path_ReplaceFileExt( std::string & ioPath, const char * newExt )
{
	using std::string;
	size_t				delimiterPos;
	size_t				dotPos;
	size_t				oldExtLen;

	//---- check the buffer size sufficient ----//
	// get position of final dot
	dotPos = ioPath.find_last_of( '.' );
	if( dotPos == string::npos )
	{
		return -1; // no dot
	}

	// get position of final path-delimiter
	delimiterPos = ioPath.find_last_of( '\\' );
	if( delimiterPos == string::npos )
		delimiterPos = ioPath.find_last_of( '/' );

	// check that '.' position is former than delimiter position
	if( dotPos < delimiterPos ) // in case of delimiter position doesn't exist, delimiter is 0(=NULL).
	{
		return -1; // no extension
	}

	//----- replace -----//
	oldExtLen = ioPath.size() - dotPos;
	ioPath.resize( ioPath.size() - oldExtLen + 1 );
	ioPath.append( newExt );

	return 0;
}

//-------------------------------------
//   function	Path_CompareFileExt
//! @brief		Compare two path strings by extension of file name.
//! @retval		true	... same
//! @retval		false	... different
//! @note		If no extension in path string, function would fail.
//-------------------------------------
inline static bool
Path_CompareFileExt( const char * path1, const char * path2 )
{
	int		result;
	char *	_path1	= const_cast<char*>( path1 );
	char *	_path2	= const_cast<char*>( path2 );
	char *	dotPos1;
	char *	dotPos2;
	char *	delimiterPos1;
	char *	delimiterPos2;

	//---- get extension from path1 ----//
	// get position of the final dot
	dotPos1 = strrchr( _path1, '.' );
	if( dotPos1 == NULL )
	{
		return false; // no dot
	}

	// get position of the final path-delimiter char.
	delimiterPos1 = strrchr( _path1, '\\' );
	if( delimiterPos1 == NULL )
	{
		delimiterPos1 = strrchr( _path1, '/' );
	}
	
	// check that '.' position is former than delimiter position
	if( dotPos1 < delimiterPos1 ) // in case of delimiter position doesn't exist, delimiter is 0(=NULL).
	{
		return false; // no extension
	}
	
	//---- get extension from path2 ----//
	// get position of final dot
	dotPos2 = strrchr( _path2, '.' );
	if( dotPos2 == NULL )
	{
		return false; // no dot
	}

	// get position of final path-delimiter
	delimiterPos2 = strrchr( _path2, '\\' );
	if( delimiterPos2 == NULL )
		delimiterPos2 = strrchr( _path2, '/' );
	
	// check that '.' position is former than delimiter position
	if( dotPos2 < delimiterPos2 ) // in case of delimiter position doesn't exist, delimiter is 0(=NULL).
	{
		return false; // no extension
	}

	//----- compare extension -----//
	result = strcmp( dotPos1, dotPos2 );
	
	return (result == 0);
}
//! @overload
//! @brief (overload for std::string)
inline static bool
Path_CompareFileExt( std::string path1, std::string path2 )
{
	return Path_CompareFileExt( path1.c_str(), path2.c_str() );
}





#ifdef _WIN32
//=============================================================================
//      Windows registry
//=============================================================================

//--------------------------------------
//   function	Reg_ReadStringA / Reg_ReadStringW
//! @brief		Get string value of specified key.
//! @param		keyRoot		... predefined reserved handle (ex. HKEY_CLASSES_ROOT) which the key belongs to
//! @param		valueName	... if NULL, get default value of the key.
//! @retval		 0	... same
//! @retval		-1	... different
//--------------------------------------
inline static short
Reg_ReadStringA( HKEY keyRoot, const char * targetKeyName, const char * valueName, char * outStringValue, long bufferLength )
{
	int		rc = 0; // return code
	DWORD	valueType;

	HKEY	targetKey		= NULL;
	DWORD	maxLength		= bufferLength;

	
	// open key
	rc = RegOpenKeyA( keyRoot, targetKeyName, &targetKey );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return -1;
	}
	
	// get the value as a string
	valueType = REG_SZ;
	rc = RegQueryValueExA( targetKey, valueName, NULL, &valueType, (UCHAR*)outStringValue, &maxLength );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return -1;
	}

	// close key
	RegCloseKey( targetKey );
	
	return 0;
}
//--------------------------------------
//   function	Reg_ReadStringA / Reg_ReadStringW
//! @brief		Get string value of specified key.
//! @param		keyRoot		... predefined reserved handle (ex. HKEY_CLASSES_ROOT) which the key belongs to
//! @param		valueName	... if NULL, get default value of the key.
//! @return		string value (if failed, returns default object of std::string )
//--------------------------------------
inline static std::string
Reg_ReadStringA( HKEY keyRoot, const char * targetKeyName, const char * valueName )
{
	int		rc = 0; // return code
	DWORD	valueType = REG_SZ;

	HKEY	targetKey		= NULL;
	DWORD	valueLen;
	char *	value;
	std::string	retval;

	// open key
	rc = RegOpenKeyA( keyRoot, targetKeyName, &targetKey );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return std::string();
	}
	
	// get length of string value
	rc = RegQueryValueExA( targetKey, valueName, NULL, &valueType, NULL, &valueLen );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return std::string();
	}

	// get the value as a string
	value = new char[valueLen+1];
	rc = RegQueryValueExA( targetKey, valueName, NULL, &valueType, reinterpret_cast<BYTE*>(value), &valueLen );
	if( rc != ERROR_SUCCESS )
	{
		delete value;
		RegCloseKey( targetKey );
		return std::string();
	}
	retval = value;
	delete value;

	// close key and free buffer
	RegCloseKey( targetKey );
	
	return retval;
}
inline static std::wstring
Reg_ReadStringW( HKEY keyRoot, const char * targetKeyName, const wchar_t * valueName )
{
	int		rc = 0; // return code
	DWORD	valueType = REG_SZ;

	HKEY	targetKey		= NULL;
	DWORD	valueLen;
	wchar_t *	value;
	std::wstring	retval;

	// open key
	rc = RegOpenKeyA( keyRoot, targetKeyName, &targetKey );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return std::wstring();
	}
	
	// get length of string value
	rc = RegQueryValueExW( targetKey, valueName, NULL, &valueType, NULL, &valueLen );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( targetKey );
		return std::wstring();
	}

	// get the value as a string
	value = new wchar_t[valueLen+1];
	rc = RegQueryValueExW( targetKey, valueName, NULL, &valueType, reinterpret_cast<BYTE*>(value), &valueLen );
	if( rc != ERROR_SUCCESS )
	{
		delete value;
		RegCloseKey( targetKey );
		return std::wstring();
	}
	retval = value;
	delete value;

	// close key and free buffer
	RegCloseKey( targetKey );
	
	return retval;
}



//--------------------------------------
//   function	Reg_WriteString
//! @brief		Write string value to the specified key.
//! @param		keyRoot		... predefined reserved handle (ex. HKEY_CLASSES_ROOT) which the key belongs to
//! @param		valueName	... value name to write (to write a default value of the key, set NULL).
//! @retval		 0	... same
//! @retval		-1	... different
//--------------------------------------
inline static short
Reg_WriteStringA( HKEY keyRoot, const char * targetKey, const char * valueName, const char * value, long valueLength )
{
	long	rc; // return code
	HKEY	hKey;
	
	// open key
	rc = RegCreateKeyExA( keyRoot, targetKey, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, NULL );
	if( rc != ERROR_SUCCESS )
	{
		return -1;
	}
	
	// write value
	RegSetValueExA( hKey, valueName, 0, REG_SZ, (UCHAR*)&value, valueLength );
	
	// close key
	RegCloseKey( hKey );
	
	return 0;
}



//--------------------------------------
//   function	Reg_WriteDword
//! @brief		Write DWORD value to specified key.
//! @param		keyRoot		... predefined reserved handle (ex. HKEY_CLASSES_ROOT) which the key belongs to
//! @param		valueName	... value name to write (to write a default value of the key, set NULL).
//! @retval		 0	... same
//! @retval		-1	... different
//--------------------------------------
inline static short
Reg_WriteDword( HKEY keyRoot, const char * targetKey, const char * valueName, DWORD value )
{
	long	rc; // return code
	HKEY	hKey;
	
	// open key
	rc = RegCreateKeyExA( keyRoot, targetKey, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, NULL );
	if( rc != ERROR_SUCCESS )
	{
		return -1;
	}
	
	// write value
	RegSetValueExA( hKey, valueName, 0, REG_DWORD_LITTLE_ENDIAN, (UCHAR*)&value, sizeof(DWORD) );
	
	// close key
	RegCloseKey( hKey );
	
	return 0;
}


//--------------------------------------
//   function	Reg_DeleteKeyTree
//! @brief		Delete key with its subkeys.
//! @retval		 0	... same
//! @retval		-1	... different
//--------------------------------------
inline static short
Reg_DeleteKeyTree( HKEY keyRoot, const char * keyName )
{
	DWORD	rc = 0; // return code

	const int	bufLen	= 255;
	HKEY	currentKey;
	char	subKeyName[bufLen];
	DWORD	subKeyNameLength	= bufLen;
	

	// open key
	rc = RegOpenKeyExA( keyRoot, keyName, 0, KEY_ALL_ACCESS, &currentKey );
	if( rc != ERROR_SUCCESS )
	{
		RegCloseKey( currentKey );
		return -1;
	}
	
	// delete subkeys
	while( rc == 0 )
	{
		// get first subkey
		rc = RegEnumKeyExA( currentKey, 0, subKeyName, &subKeyNameLength, NULL, NULL, NULL, NULL );
		if(  (rc == ERROR_NO_MORE_ITEMS)  ||  (rc == ERROR_BADKEY)  )
		{
			// no more subkeys. delete the key
			RegDeleteKeyA( keyRoot, keyName );
			RegCloseKey( currentKey );
			return -1;
		}

		// delete the subkey
		rc = static_cast<DWORD>( Reg_DeleteKeyTree(currentKey, subKeyName) );
	}

	// close key
	RegCloseKey( currentKey );
	
	return 0;
}

#endif // _WIN32


//=============================================================================
//      alert message boxes
//=============================================================================
#ifdef _WIN32
#ifndef __cplusplus_cli
//-------------------------------------
// function : AlertInfo
// brief    : Display information for users in message box
// note     : can use "printf()" format string
//-------------------------------------
inline static void
AlertInfo( const char * inFormat, ... )
{
	va_list	argumentVector;
	char	logString[1024];
	char	title[32];
	
	strcpy( title, "Information" );

	// Display message
	va_start( argumentVector, inFormat );
	vsprintf( logString, inFormat, argumentVector );
	MessageBoxA( NULL, logString, title, MB_OK | MB_ICONINFORMATION );
	va_end( argumentVector );
}
inline static void
AlertInfo( const wchar_t * inFormat, ... )
{
	va_list	argumentVector;
	wchar_t	logString[1024];
	wchar_t	title[32];
	
	mbstowcs( title, "Information", 32 );

	// Display message
	va_start( argumentVector, inFormat );
	vswprintf( logString, inFormat, argumentVector );
	MessageBoxW( NULL, logString, title, MB_OK | MB_ICONINFORMATION );
	va_end( argumentVector );
}

//-------------------------------------
// function : AlertWarning
// brief    : Display warning for users in message box
// note     : can use "printf()" format string
//-------------------------------------
inline static void
AlertWarning( const char * inFormat, ... )
{
	va_list	argumentVector;
	char	logString[1024];
	char	title[32];
	
	strcpy( title, "Information" );

	// Display message
	va_start( argumentVector, inFormat );
	vsprintf( logString, inFormat, argumentVector );
	MessageBoxA( NULL, logString, title, MB_OK | MB_ICONWARNING );
	va_end( argumentVector );
}
inline static void
AlertWarning( const wchar_t * inFormat, ... )
{
	va_list	argumentVector;
	wchar_t	logString[1024];
	wchar_t	title[32];
	
	mbstowcs( title, "Information", 32 );

	// Display message
	va_start( argumentVector, inFormat );
	vswprintf( logString, inFormat, argumentVector );
	MessageBoxW( NULL, logString, title, MB_OK | MB_ICONWARNING );
	va_end( argumentVector );
}

//-------------------------------------
// function : AlertError
// brief    : Display fatal error in message box
// note     : can use "printf()" format string
//-------------------------------------
inline static void
AlertError( const char * inFormat, ... )
{
	va_list	argumentVector;
	char	logString[1024];
	char	title[32];
	
	strcpy( title, "Information" );

	// Display message
	va_start( argumentVector, inFormat );
	vsprintf( logString, inFormat, argumentVector );
	MessageBoxA( NULL, logString, title, MB_OK | MB_ICONWARNING );
	va_end( argumentVector );
}
inline static void
AlertError( const wchar_t * inFormat, ... )
{
	va_list	argumentVector;
	wchar_t	logString[1024];
	wchar_t	title[32];
	
	mbstowcs( title, "Information", 32 );

	// Display message
	va_start( argumentVector, inFormat );
	vswprintf( logString, inFormat, argumentVector );
	MessageBoxW( NULL, logString, title, MB_OK | MB_ICONERROR );
	va_end( argumentVector );
}
#endif // __cplusplus_cli
#endif // _WIN32





//=============================================================================
//
//      utilities for Win32API
//
//=============================================================================
#ifdef _WIN32
namespace Win32
{
	//-------------------------------------
	//   function	AlertLastError
	//! @brief		Display the Windows' "LastError" in message box.
	//-------------------------------------
	inline static void
	AlertLastError()
	{
		char	message[1024];

		// Get "LastError"
		FormatMessageA( FORMAT_MESSAGE_FROM_SYSTEM,
			NULL,
			GetLastError(),
			MAKELANGID( LANG_NEUTRAL,SUBLANG_DEFAULT ),
			message,
			256,
			NULL );

		// show it in message box
		MessageBoxA( NULL, message, "Last Error", MB_OK );

		LocalFree( message );
	}


	//-------------------------------------
	//   function	GetLastErrorMessage
	//! @brief		Get the message of Windows' "LastError"
	//-------------------------------------
	inline static std::string
	GetLastErrorMessage()
	{
		char	message[1024];

		// Get "LastError"
		FormatMessageA( FORMAT_MESSAGE_FROM_SYSTEM,
			NULL,
			GetLastError(),
			MAKELANGID( LANG_NEUTRAL,SUBLANG_DEFAULT ),
			message,
			256,
			NULL );

		// Write it to log file
		return message;
	}


	//--------------------------------------
	//   function	GetVirtualKeyName
	//! @brief		Get virtual key name by key-code.
	//--------------------------------------
	inline static std::string
	GetVirtualKeyName( UINT keyCode )
	{
		std::stringstream ss;

		if( ('a' <= keyCode && keyCode <= 'z') ||
			('A' <= keyCode && keyCode <= 'Z') ||
			('0' <= keyCode && keyCode <= '9')     )
		{
			ss << "VK_" << static_cast<char>( keyCode );
			return ss.str();
		}
		
		switch( keyCode )
		{
			// mouse
			case VK_LBUTTON:	return "VK_LBUTTON";
			case VK_RBUTTON:	return "VK_RBUTTON";
			case VK_CANCEL:		return "VK_CANCEL";
			case VK_MBUTTON:	return "VK_MBUTTON";
			
			case VK_BACK:		return "VK_BACK";
			case VK_TAB:		return "VK_TAB";
			case VK_CLEAR:		return "VK_CLEAR";
			case VK_RETURN:		return "VK_RETURN";
			
			case VK_SHIFT:		return "VK_SHIFT";
			case VK_CONTROL:	return "VK_CONTROL";
			case VK_MENU:		return "VK_MENU";
			case VK_PAUSE:		return "VK_PAUSE";
			case VK_CAPITAL:	return "VK_CAPITAL";
			
			//case VK_KANA:		return "VK_KANA"; // DMC cannot handle this
			
			case VK_ESCAPE:		return "VK_ESCAPE";
			
			case VK_SPACE:		return "VK_SPACE";
			case VK_PRIOR:		return "VK_PRIOR";
			case VK_NEXT:		return "VK_NEXT";
			case VK_END:		return "VK_END";
			case VK_HOME:		return "VK_HOME";
			case VK_LEFT:		return "VK_LEFT";
			case VK_UP:			return "VK_UP";
			case VK_RIGHT:		return "VK_RIGHT";
			case VK_DOWN:		return "VK_DOWN";
			case VK_SELECT:		return "VK_SELECT";
			case VK_PRINT:		return "VK_PRINT";
			case VK_EXECUTE:	return "VK_EXECUTE";
			case VK_SNAPSHOT:	return "VK_SNAPSHOT";
			case VK_INSERT:		return "VK_INSERT";
			case VK_DELETE:		return "VK_DELETE";
			case VK_HELP:		return "VK_HELP";
			case VK_LWIN:		return "VK_LWIN";
			case VK_RWIN:		return "VK_RWIN";
			case VK_APPS:		return "VK_APPS";
			
			case VK_NUMPAD0:	return "VK_NUMPAD0";
			case VK_NUMPAD1:	return "VK_NUMPAD1";
			case VK_NUMPAD2:	return "VK_NUMPAD2";
			case VK_NUMPAD3:	return "VK_NUMPAD3";
			case VK_NUMPAD4:	return "VK_NUMPAD4";
			case VK_NUMPAD5:	return "VK_NUMPAD5";
			case VK_NUMPAD6:	return "VK_NUMPAD6";
			case VK_NUMPAD7:	return "VK_NUMPAD7";
			case VK_NUMPAD8:	return "VK_NUMPAD8";
			case VK_NUMPAD9:	return "VK_NUMPAD9";
			case VK_MULTIPLY:	return "VK_MULTIPLY";
			case VK_ADD:		return "VK_ADD";
			case VK_SEPARATOR:	return "VK_SEPARATOR";
			case VK_SUBTRACT:	return "VK_SUBTRACT";
			case VK_DECIMAL:	return "VK_DECIMAL";
			case VK_DIVIDE:		return "VK_DIVIDE";
			case VK_F1:			return "VK_F1";
			case VK_F2:			return "VK_F2";
			case VK_F3:			return "VK_F3";
			case VK_F4:			return "VK_F4";
			case VK_F5:			return "VK_F5";
			case VK_F6:			return "VK_F6";
			case VK_F7:			return "VK_F7";
			case VK_F8:			return "VK_F8";
			case VK_F9:			return "VK_F9";
			case VK_F10:		return "VK_F10";
			case VK_F11:		return "VK_F11";
			case VK_F12:		return "VK_F12";
			case VK_F13:		return "VK_F13";
			case VK_F14:		return "VK_F14";
			case VK_F15:		return "VK_F15";
			case VK_F16:		return "VK_F16";
			case VK_F17:		return "VK_F17";
			case VK_F18:		return "VK_F18";
			case VK_F19:		return "VK_F19";
			case VK_F20:		return "VK_F20";
			case VK_F21:		return "VK_F21";
			case VK_F22:		return "VK_F22";
			case VK_F23:		return "VK_F23";
			case VK_F24:		return "VK_F24";
			
			case VK_NUMLOCK:	return "VK_NUMLOCK";
			case VK_SCROLL:		return "VK_SCROLL";
			
			default:			return "(unknown key)";
		}
	}
} // namespace Win32
#endif // _WIN32



//=============================================================================
//
//      Component Object Model (COM)
//
//=============================================================================
#ifdef GUID_DEFINED
namespace Com
{
	//-------------------------------------
	// function : IsEqual
	// brief    : compare two GUIDs
	//-------------------------------------
	inline static bool
	IsEqual( const GUID & x, const GUID & y )
	{
		return ( ::memcmp(&x, &y, sizeof(GUID)) == 0 );
	}
#	ifndef __cplusplus_cli
	//-------------------------------------
	// function : LogFileGuid
	// brief    : Write GUID into log file
	//-------------------------------------
	inline static void
	LogFileGuid( const GUID & guid )
	{
		LogFile( "{%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X}",
			guid.Data1,
			guid.Data2,
			guid.Data3,
			guid.Data4[0], guid.Data4[1],
			guid.Data4[2], guid.Data4[3], guid.Data4[4], guid.Data4[5], guid.Data4[6], guid.Data4[7] );
	}
#	endif


	//-------------------------------------
	//   function	SetGuid
	//! @brief		Set GUID value.
	//-------------------------------------
	inline static void
	SetGuid( GUID * outGuid, uint32_t data1, uint16_t data2, uint16_t data3,
				uint8_t data4_0, uint8_t data4_1,
				uint8_t data4_2, uint8_t data4_3, uint8_t data4_4, uint8_t data4_5, uint8_t data4_6, uint8_t data4_7 )
	{
		outGuid->Data1 = data1;
		outGuid->Data2 = data2;
		outGuid->Data3 = data3;
		outGuid->Data4[0] = data4_0;
		outGuid->Data4[1] = data4_1;
		outGuid->Data4[2] = data4_2;
		outGuid->Data4[3] = data4_3;
		outGuid->Data4[4] = data4_4;
		outGuid->Data4[5] = data4_5;
		outGuid->Data4[6] = data4_6;
		outGuid->Data4[7] = data4_7;
	}
	//-------------------------------------
	//   function	SetGuid
	//! @brief		Set GUID value by GUID-formatted string.
	//-------------------------------------
	inline static void
	SetGuid( GUID * outGuid, const char * stringGuid )
	{
		uint32_t	data1;
		uint16_t	data2, data3;
		uint8_t		data4[8];
		char		buf[63];
		if( outGuid==NULL || stringGuid==NULL || strlen(stringGuid) < 38 ) // 37(GUID+{}) + 1(\0)
			return;
		
		// parse
		//(cut each part)						(add term.)		(convert to integer)
		strncpy( buf, &stringGuid[1], 8 );		buf[8]='\0';	data1 = (uint32_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[10], 4 );		buf[4]='\0';	data2 = (uint16_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[15], 4 );		buf[4]='\0';	data3 = (uint16_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[20], 2 );		buf[2]='\0';	data4[0] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[22], 2 );		buf[2]='\0';	data4[1] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[25], 2 );		buf[2]='\0';	data4[2] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[27], 2 );		buf[2]='\0';	data4[3] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[29], 2 );		buf[2]='\0';	data4[4] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[31], 2 );		buf[2]='\0';	data4[5] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[33], 2 );		buf[2]='\0';	data4[6] = (uint8_t)strtoul( buf, NULL, 16 );
		strncpy( buf, &stringGuid[35], 2 );		buf[2]='\0';	data4[7] = (uint8_t)strtoul( buf, NULL, 16 );
		
		// set
		SetGuid( outGuid,
			data1,
			data2, data3,
			data4[0], data4[1],
			data4[2], data4[3], data4[4], data4[5], data4[6], data4[7] );
	}

	//-------------------------------------
	//   function	GuidToString
	//! @brief		Convert GUID struct to string.
	//! @retval		 0	... success
	//! @retval		-1	... failure
	//-------------------------------------
	inline static short
	GuidToString( const GUID & guid, long maxStringLength, char * outString )
	{
		// Is the buffer size enough?
		if( maxStringLength < 38 ) // 37(GUID) + 1(\0)
		{
			return -1; // insufficient buffer size
		}
		
		sprintf( outString, "{%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X}",
			guid.Data1,
			guid.Data2,
			guid.Data3,
			guid.Data4[0], guid.Data4[1],
			guid.Data4[2], guid.Data4[3], guid.Data4[4], guid.Data4[5], guid.Data4[6], guid.Data4[7] );

		return 0;
	}

	//-------------------------------------
	//   function	GuidToString
	//! @brief		Convert GUID struct to string.
	//-------------------------------------
	inline static std::string
	GuidToString( const GUID & guid )
	{
		char str[64];
		
		sprintf( str, "{%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X}",
			guid.Data1,
			guid.Data2,
			guid.Data3,
			guid.Data4[0], guid.Data4[1],
			guid.Data4[2], guid.Data4[3], guid.Data4[4], guid.Data4[5], guid.Data4[6], guid.Data4[7] );
		
		return std::string( str );
	}


	//-------------------------------------
	//   function	GetInterfaceName
	//! @brief		Get the COM-Interface name which the given GUID means.
	//! @note		Interface names are HARDCODED in this function.
	//! 			Use this function for only debug.
	//-------------------------------------
	inline static std::string
	GetInterfaceName( GUID interfaceId )
	{
		GUID	guid;
		
		// interface IDs of Microsoft standard COM interfaces
		SetGuid( &guid, "{00000000-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IUnknown";
		SetGuid( &guid, "{00000001-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IClassFactory";
		SetGuid( &guid, "{00000002-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IMalloc";
		SetGuid( &guid, "{00000003-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IMarshal";
		SetGuid( &guid, "{00000018-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IStdMarshalInfo";
		SetGuid( &guid, "{00000019-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IExternalConnection";
		SetGuid( &guid, "{0000001d-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IMallocSpy";
		SetGuid( &guid, "{00000109-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IPersistStream";
		SetGuid( &guid, "{00000114-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IOleWindow";
		SetGuid( &guid, "{000001cf-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IMarshal2";
		SetGuid( &guid, "{00020400-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IDispatch";
		SetGuid( &guid, "{00020404-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "IEnumVARIANT";
		SetGuid( &guid, "{1c733a30-2a1c-11ce-ade5-00aa0044773d}" );
		if( IsEqual(guid, interfaceId) )	return "ICallFactory";
		SetGuid( &guid, "{b196b284-bab4-101a-b69c-00aa00341d07}" );
		if( IsEqual(guid, interfaceId) )	return "IConnectionPointContainer";
		SetGuid( &guid, "{b196b28b-bab4-101a-b69c-00aa00341d07}" );
		if( IsEqual(guid, interfaceId) )	return "ISpecifyPropertyPages";
		SetGuid( &guid, "{b196b28d-bab4-101a-b69c-00aa00341d07}" );
		if( IsEqual(guid, interfaceId) )	return "IPropertyPage";
		SetGuid( &guid, "{0000001B-0000-0000-C000-000000000046}" );
		if( IsEqual(guid, interfaceId) )	return "(?)IdentifyUnmarshal";
		SetGuid( &guid, "{4C1E39E1-E3E3-4296-AA86-EC938D896E92}" );
		if( IsEqual(guid, interfaceId) )	return "(?)IMarshalingRelatedSomething";
		SetGuid( &guid, "{6d5140c1-7436-11ce-8034-00aa006009fa}" );
		if( IsEqual(guid, interfaceId) )	return "IServiceProvider";

		// interface IDs for MSAA
		SetGuid( &guid, "{618736e0-3c3d-11cf-810c-00aa00389b71}" );
		if( IsEqual(guid, interfaceId) )	return "IAccessible";
		SetGuid( &guid, "{03022430-ABC4-11d0-BDE2-00AA001A1953}" );
		if( IsEqual(guid, interfaceId) )	return "IAccessibleHandler";
		SetGuid( &guid, "{7852b78d-1cfd-41c1-a615-9c0c85960b5f}" );
		if( IsEqual(guid, interfaceId) )	return "IAccIdentity";
		SetGuid( &guid, "{76c0dbbb-15e0-4e7b-b61b-20eeea2001e0}" );
		if( IsEqual(guid, interfaceId) )	return "IAccPropServer";
		SetGuid( &guid, "{6e26e776-04f0-495d-80e4-3330352e3169}" );
		if( IsEqual(guid, interfaceId) )	return "IAccPropServices";
		
		// interface IDs of OPT
		SetGuid( &guid, 0x0958ECA7,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPAsyncOutput";
		SetGuid( &guid, 0x0958ECA7,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPAsyncOutput";
		SetGuid( &guid, 0x0958ECA1,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPInitialise";
		SetGuid( &guid, 0x0958ECA2,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPTimeInfo";
		SetGuid( &guid, 0x0958ECA5,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPEventQueue";
		SetGuid( &guid, 0x0958ECA3,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPEventFilter";
		SetGuid( &guid, 0x0958ECA6,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPTimeSink";
		SetGuid( &guid, 0x0958ECA8,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPEventList";
		SetGuid( &guid, 0x0958ECA9,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPClientServices";
		SetGuid( &guid, 0x0958ECAA,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPClientInfo";
		SetGuid( &guid, 0x0958ECAB,0x6709,0x4267,0x96,0x96,0x33,0xD5,0x6F,0xEC,0x4F,0x6A );
		if( IsEqual(guid, interfaceId) )	return "IMPNotificationSink";

		// not found
		return GuidToString( interfaceId );
	}

	//-------------------------------------
	//   function	GetErrorMessage
	//! @brief		Get message describes given COM's error code.
	//-------------------------------------
	inline static std::string
	GetErrorMessage( HRESULT errorCode )
	{
		switch( errorCode )
		{
			case 0:				return "S_OK; Succeeded.";
			case 1:				return "S_FALSE; Successfully false(?).";
			case 0x80004005:	return "E_FAIL; Unspecified error.";
			case 0x80004001:	return "E_NOTIMPL; Not implemented.";
			case 0x80004002:	return "E_NOINTERFACE; No such interface supported.";
			case 0x80004003:	return "E_POINTER; Invalid pointer.";
			case 0x80070006:	return "E_HANDLE; Invalid handle.";
			case 0x8000FFFF:	return "E_UNEXPECTED; Catastrophic failure.";
			case 0x8007000E:	return "E_OUTOFMEMORY; Run out of memory.";
			case 0x80070057:	return "E_INVALIDARG; One or more arguments are invalid.";
			case 0x80040154:	return "REGDB_E_CLASSNOTREG; Class not registered.";
			case 0x80040110:	return "CLASS_E_NOAGGREGATION; Class does not support aggregation (or class object is remote).";
		}

		return std::string();
	}
} // namespace Com
#endif // GUID_DEFINED







//=============================================================================
//
//      YAMAHA OPT SDK
//
//=============================================================================
#ifdef SZ_MIDI_PLUGIN_REGKEY
namespace Opt
{
	//-------------------------------------
	//   function	GetErrorName
	//! @brief		Get OPT SDK error name.
	//-------------------------------------
	inline static std::string
	GetErrorName( HRESULT optError )
	{
		std::string	name;

		switch( optError )
		{
			case S_OK:				name = "S_OK";				break;
			case S_MPPARAMS:		name = "S_MPPARAMS";		break;
			case S_MPNOCOMPOSITE:	name = "S_MPNOCOMPOSITE";	break;
			case S_MPNOMACROUNDO:	name = "S_MPNOMACROUNDO";	break;
			case E_MPBADUNIQUEID:	name = "E_MPBADUNIQUEID";	break;
			case E_MPBADCHANNEL:	name = "E_MPBADCHANNEL";	break;
			case E_MPBADTYPE:		name = "E_MPBADTYPE";		break;
			case E_MPBADPORT:		name = "E_MPBADPORT";		break;
			case E_FAIL:			name = "E_FAIL";			break;
			case E_NOTIMPL:			name = "E_NOTIMPL";			break;
			case E_OUTOFMEMORY:		name = "E_OUTOFMEMORY";		break;
			case E_POINTER:			name = "E_POINTER";			break;
			case E_NOINTERFACE:		name = "E_NOINTERFACE";		break;
			default:				name = "(unknown error)";	break;
		};
		
		return name;
	}

	//-------------------------------------
	//   function	GetInterfaceName
	//! @brief		Get name of specified COM-Interface.
	//-------------------------------------
	inline static std::string
	GetInterfaceName( GUID interfaceId )
	{
		return COM_GetInterfaceName( interfaceId );
	}

	//-------------------------------------
	//   function	GetClassName
	//! @brief		Get name of the class that has specified GUID.
	//-------------------------------------
	inline static std::string
	GetClassName( GUID classId )
	{
		using std::string;
		GUID guid;

		SetGuid( &guid, "{4FB99E76-8CC5-4E26-B42F-CDE5E49C5539}" );
		if( guid == classId )	return string( "CRandomizerCore" );
		SetGuid( &guid, "{FB332438-AE07-4410-AB67-3AFCCBBED27E}" );
		if( guid == classId )	return string( "CRandomizerPropertyPage" );
		SetGuid( &guid, "{79E7C183-FBD3-4d4b-8B79-88A0AD1CA5BC}" );
		if( guid == classId )	return string( "CSGOptListCore" );
		SetGuid( &guid, "{B6BD3426-6804-4d58-9672-66D64EFAD2C3}" );
		if( guid == classId )	return string( "CSGOptListPropertyPage" );

		return string( "(unknown class)" );
	}

	//-------------------------------------
	//   function	GetBlockTypeName
	//! @brief		Get block type name.
	//-------------------------------------
	inline static std::string
	GetBlockTypeName( MPBLOCKINFO blockInfo )
	{
		switch( blockInfo.eStatus )
		{
			case MPBLOCK_TYPE_NULL:		return "MPBLOCK_TYPE_NULL";
			case MPBLOCK_TYPE_REAL:		return "MPBLOCK_TYPE_REAL";
			case MPBLOCK_TYPE_GHOST:	return "MPBLOCK_TYPE_GHOST";
			case MPBLOCK_TYPE_SEQ:		return "MPBLOCK_TYPE_SEQ";
			case MPBLOCK_TYPE_EMPTY:	return "MPBLOCK_TYPE_EMPTY";
			default:					return "(unknown block type)";
		};
	}

	//-------------------------------------
	//   function	GetEventTypeName
	//! @brief		Get event type name.
	//! @return		temporary pointer to the name
	//-------------------------------------
	inline static char *
	GetEventTypeName( MPEVENT event )
	{
		switch( event.eStatus )
		{
			case MPNULL:			return "MPNULL";
			case MPNRPN:			return "MPNRPN";
			case MPRPN:				return "MPRPN";
			case MPPATCH:			return "MPPATCH";
			case MPSEQSHORT:		return "MPSEQSHORT";
			case MPNOTE_OFF:		return "MPNOTE_OFF";
			case MPNOTE_ON:			return "MPNOTE_ON";
			case MPPOLY_AFTER:		return "MPPOLY_AFTER";
			case MPCNTRL_CHNG:		return "MPCNTRL_CHNG";
			case MPPROG_CHNG:		return "MPPROG_CHNG";
			case MPAFTER_TOUCH:		return "MPAFTER_TOUCH";
			case MPPITCH_BEND:		return "MPPITCH_BEND";
			default:				return "(unknown event type)";
		};
	}

	//-------------------------------------
	//   function	GetQueueFlagName
	//! @brief		Get name of MPQUEUE_FLAGS enum. member
	//-------------------------------------
	inline static std::string
	GetQueueFlagName( _MPQUEUE_FLAGS flag )
	{
		std::string	name;
		if( flag & MPQPLAYPR )	name = "MPQPLAYPR";
		if( flag & MPQPLAYJIT )	name = "MPQPLAYJIT";
		if( flag & MPQPLAYAT )	name = "MPQPLAYAT";
		if( flag & MPQTHRU )	name = "MPQTHRU";
		if( flag & MPQINPUT )	name = "MPQINPUT";
		if( flag & MPQINRAW )	name = "MPQINRAW";
		if( flag & MPQOFFLINE )	name = "MPQOFFLINE";
		if( flag & MPQSTART )	name = "MPQSTART";
		if( flag & MPQSTOP )	name = "MPQSTOP";
		if( flag & MPQLOOP )	name = "MPQLOOP";
		if( flag & MPQASYNC )	name = "MPQASYNC";
		if( flag & MPQBREAK )	name = "MPQBREAK";
		
		return name;
	}

	//-------------------------------------
	//   function	GetNotificationName
	//! @brief		Get name of MPMSG
	//-------------------------------------
	inline static std::string
	GetNotificationName( MPMSG msg )
	{
		GUID	guid;
		
		SetGuid( &guid, 0x9dd48de0, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_DATACHANGE";
		SetGuid( &guid, 0x9dd48de1, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_ACTIVATE";
		SetGuid( &guid, 0x9dd48de2, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_DEVICECHANGE";
		SetGuid( &guid, 0x9dd48de3, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_KEYSIGCHANGE";
		SetGuid( &guid, 0x9dd48de4, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_METERCHANGE";
		SetGuid( &guid, 0x9dd48de5, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_TEMPOCHANGE";
		SetGuid( &guid, 0x9dd48de6, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_SYSCHANGE";
		SetGuid( &guid, 0x9dd48de7, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_PPQCHANGE";
		SetGuid( &guid, 0x9dd48de8, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_SAMPLECHANGE";
		SetGuid( &guid, 0x9dd48de9, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_STATUS";
		SetGuid( &guid, 0x9dd48dea, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_LOOPCHG";
		SetGuid( &guid, 0x9dd48deb, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_SONGNAME";
		SetGuid( &guid, 0x9dd48dec, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_DISPREQ";
		SetGuid( &guid, 0x9dd48dee, 0x9f3f, 0x4e80, 0xbc, 0xb5, 0x44, 0x82, 0xa5, 0xb5, 0x24, 0xa );
		if( guid == msg.guidType )	return "MPNOTIFY_SELECTION";
	}
} // namespace Opt
#endif // SZ_MIDI_PLUGIN_REGKEY






//=============================================================================
//      Others
//=============================================================================

#ifdef __cplusplus_cli
//-------------------------------------
//   function	LogFile
//! @brief		Write message into log file.
//! @note		for C++/CLI only
//-------------------------------------
inline static void
LogFile( const char * message )
{
	FILE *	fp;
	time_t	timeInt;
	tm		timeStruct;
	char	logString[1024];

	// open the log file
	fp = fopen( kLogPath, "a" );
	if ( fp == NULL )
	{
		return; // failed
	}

	// generate local time
	time( &timeInt );
	timeStruct = *localtime( &timeInt );

	// make the time into calender value
	mktime( &timeStruct );

	// make the calender value into string
	sprintf( logString, "%04d%02d%02d %02d:%02d:%02d %s\n",
		timeStruct.tm_year+1900, timeStruct.tm_mon, timeStruct.tm_mday,
		timeStruct.tm_hour, timeStruct.tm_min, timeStruct.tm_sec,
		message );

	// write message
	fwrite( logString, 1/*sizeof(char)*/, strlen(logString), fp );

	fclose( fp );
}
#endif

#ifndef __cplusplus_cli
//-------------------------------------
//   function	LogOut
//! @brief		Write message into stdout.
//! @note		Can use "printf" format for message.
//-------------------------------------
inline static void
LogOut( const char * format, ... )
{
	va_list	argumentVector;
	time_t	timeInt;
	tm		timeStruct;
	char	logString[1024];

	// generate local time
	time( &timeInt );
	timeStruct = *localtime( &timeInt );

	// make the time into calender value
	mktime( &timeStruct );

	// make the calender value into string
	sprintf( logString, "%02d:%02d.%02d ",
		timeStruct.tm_hour, timeStruct.tm_min, timeStruct.tm_sec );

	fwrite( logString, 1/*sizeof(char)*/,  strlen(logString), stdout );

	// write message
	va_start( argumentVector, format );
	vsprintf( logString, format, argumentVector );
	strcat( logString, "\n" );
	fwrite( logString, 1/*sizeof(char)*/, strlen(logString), stdout );
	va_end( argumentVector );
}
#endif
#ifndef __cplusplus_cli
//-------------------------------------
//   function	LogFile
//! @brief		Write message into log file.
//! @note		Can use "printf" format for message.
//-------------------------------------
inline static void
LogFile( const char * format, ... )
{
	FILE *	fp;
	va_list	argumentVector;
	time_t	timeInt;
	tm		timeStruct;
	char	logString[1024];

	// open the log file
	fp = fopen( kLogPath, "a" );
	if ( fp == NULL )
	{
		return; // failed
	}

	// generate local time
	time( &timeInt );
	timeStruct = *localtime( &timeInt );

	// make the time into calender value
	mktime( &timeStruct );

	// make the calender value into string
	sprintf( logString, "%04d%02d%02d %02d:%02d:%02d ",
		timeStruct.tm_year+1900, timeStruct.tm_mon, timeStruct.tm_mday, timeStruct.tm_hour, timeStruct.tm_min, timeStruct.tm_sec );

	fwrite( logString, 1/*sizeof(char)*/,  strlen(logString), fp );

	// write message
	va_start( argumentVector, format );
	vsprintf( logString, format, argumentVector );
	strcat( logString, "\n" );
	fwrite( logString, 1/*sizeof(char)*/, strlen(logString), fp );
	va_end( argumentVector );

	fclose( fp );
}
#endif
#ifndef __cplusplus_cli
//-------------------------------------
//   function	LogFile
//! @brief		Write message into log file.
//! @note		Can use "printf" format for message.
//-------------------------------------
inline static void
LogFile( const wchar_t * inFormat, ... )
{
	FILE *	fp;
	va_list	argumentVector;
	time_t	timeInt;
	tm		timeStruct;
	wchar_t	logString[1024];

	// open the log file
	fp = fopen( kLogPath, "ab" );
	if ( fp == NULL )
	{
		return; // failed
	}

	// generate local time
	time( &timeInt );
	timeStruct = *localtime( &timeInt );

	// make the time into calender value
	mktime( &timeStruct );

	// make the calender value into string
	swprintf( logString, L"%04d%02d%02d %02d:%02d:%02d ",
		timeStruct.tm_year+1900, timeStruct.tm_mon, timeStruct.tm_mday, timeStruct.tm_hour, timeStruct.tm_min, timeStruct.tm_sec );

	fwprintf( fp, logString );

	// write message
	va_start( argumentVector, inFormat );
	vswprintf( logString, inFormat, argumentVector );
	wcscat( logString, L"\r\n" );
	fwprintf( fp, logString );
	va_end( argumentVector );

	fclose( fp );
}
#endif

#ifdef _WIN32
static CRITICAL_SECTION __CritialSectionForInterlockedLogFile__;
inline static void InterlockedLogFile_Init()
{
	InitializeCriticalSection( &__CritialSectionForInterlockedLogFile__ );
}
inline static void InterlockedLogFile_Dispose()
{
	DeleteCriticalSection( &__CritialSectionForInterlockedLogFile__ );
}
//-------------------------------------
//   function	InterlockedLogFile
//! @brief		Write message into log file in critical section
//! @note		do not forget that this function takes small time.
//-------------------------------------
inline static void
InterlockedLogFile( const char * message )
{
	FILE *	fp;
	
	EnterCriticalSection( &__CritialSectionForInterlockedLogFile__ );
	{
		// open the log file
		fp = fopen( kLogPath, "ab" );
		if( fp == NULL )
		{
			return; // failed
		}
		
		// write message
		fputs( message, fp );
		
		fclose( fp );
	}
	LeaveCriticalSection( &__CritialSectionForInterlockedLogFile__ );
}
#endif


//=============================================================================
//
//    String operations
//
//=============================================================================
namespace String
{
	//! Flags for specifying code page
	enum Encoding
	{
		Default	= -1,	//!< System default code page
		Unicode,		//!< Unicode (UTF16) code page
		UTF8			//!< UTF-8 code page
	};

#	ifdef _JAVASOFT_JNI_H_
	//-------------------------------------
	//   function	ToStdString
	//! @brief		Convert JNI string into std::wstring
	//! @retval		(success) ... converted wstring object
	//! @retval		(failure) ... empty wstring
	//-------------------------------------
	inline static std::wstring
	ToStdString( JNIEnv * env, jstring jstr )
	{
		jsize			length;
		const jchar *	strContent;
		jboolean		isCopy;
		std::wstring	wstr;
		
		// prepare buffer
		length = env->GetStringLength( jstr );
		if( length <= 0 )
		{
			return std::wstring();
		}
		
		// get charactors from JNI string
		strContent = env->GetStringChars( jstr, &isCopy );
		if( strContent == 0 )
		{
			return std::wstring();
		}
	
		// make std::wstring object
		wstr = reinterpret_cast<const wchar_t*>( strContent );
		wstr.resize( length );
		
		// release temporary copy of the JNI string
		if( isCopy == JNI_TRUE )
		{
			env->ReleaseStringChars( jstr, strContent );
		}
		
		return wstr;
	}
#	endif // _JAVASOFT_JNI_H_


#	ifndef __cplusplus_cli
	//-------------------------------------
	//   function	Format
	//! @brief		create string with printf style format
	//! @retval		(success) ... formatted string
	//! @retval		(failure) ... empty string
	//-------------------------------------
	inline static std::string
	Format( const char * format, ... )
	{
		int		rc; // result code
		va_list	argumentVector;
		char	str[1024];
		
		// format string
		va_start( argumentVector, format );
		rc = vsprintf( str, format, argumentVector );
		if( rc < 0 )
		{
			str[0] = 0; // make the string empty
		}
		va_end( argumentVector );

		// return formatted string
		return std::string( str );
	}
#	endif

#	ifndef __cplusplus_cli
	//-------------------------------------
	//   function	Format
	//! @brief		create string with printf style format
	//! @retval		(success) ... formatted string
	//! @retval		(failure) ... empty string
	//-------------------------------------
	inline static std::wstring
	Format( const wchar_t * format, ... )
	{
		int		rc; // result code
		va_list	argumentVector;
		wchar_t	str[1024];
		
		// format string
		va_start( argumentVector, format );
		rc = vswprintf( str, format, argumentVector );
		if( rc < 0 )
		{
			str[0] = 0; // make the string empty
		}
		va_end( argumentVector );

		// return formatted string
		return std::wstring( str );
	}
#	endif

	//-------------------------------------
	//   function	Narrow
	//! @brief		Convert Unicode string to platform default encoding string
	//! @retval		(success) ... converted string
	//! @retval		(failure) ... empty string
	//-------------------------------------
	inline static std::string
	Narrow( const std::wstring & str )
	{
#		ifdef _WIN32
		int			rc; // result code
		char *		charBuf;
		long		neededBufSize;
		std::string	narrowStr;
		const UINT	codePage = CP_ACP;

		// get buffer size needed
		neededBufSize = WideCharToMultiByte( codePage, 0, (const WCHAR*)str.c_str(), (int)str.length(),
												NULL, 0, NULL, NULL );
		if( neededBufSize == 0 )
		{
			return std::string(); // invalid string for conversation?
		}

		// allocate buffer
		charBuf = new char[neededBufSize+1];

		// convert encoding
		rc = WideCharToMultiByte( codePage, 0, (const WCHAR*)str.c_str(), (int)str.length(),
									charBuf, neededBufSize, NULL, NULL );
		if( rc == FALSE )
		{
			return std::string(); // failed to convert
		}
		charBuf[neededBufSize] = 0; // null terminate

		// make string
		narrowStr = charBuf;

		// delete buffer and return value
		delete charBuf;
		return narrowStr;
#		endif
	}

	//-------------------------------------
	//   function	Widen
	//! @brief		Convert string to Unicode string
	//! @param		encoding ... encoding of the string to be converted
	//! @returns	(success) ... converted std::wstring object
	//              (failure) ... empty string
	//-------------------------------------
	inline static std::wstring
	Widen( const std::string & narrowStr, Sgry::String::Encoding encoding = Default )
	{
#		ifdef _WIN32
		int				rc; // result code
		WCHAR *			wcharBuf;
		long			neededBufSize;
		std::wstring	widenStr;
		UINT			codePage;

		// convert encoding ID to Win32API's one
		switch( encoding )
		{
			case( Unicode ):	return std::wstring(); // cannot narrow...
			case Default:		codePage = CP_ACP;	break;
			default:			codePage = CP_UTF8;	break;
		}

		// get buffer size needed
		neededBufSize = MultiByteToWideChar( codePage, WC_COMPOSITECHECK, narrowStr.c_str(), (int)narrowStr.length(),
												NULL, 0 );
		if( neededBufSize == 0 )
		{
			return std::wstring(); // invalid string for conversation?
		}

		// allocate buffer
		wcharBuf = new WCHAR[neededBufSize+1];

		// convert encoding
		rc = MultiByteToWideChar( codePage, WC_COMPOSITECHECK, narrowStr.c_str(), (int)narrowStr.length(),
									(WCHAR*)wcharBuf, neededBufSize );
		if( rc == FALSE )
		{
			return std::wstring(); // failed to convert
		}
		wcharBuf[neededBufSize] = 0; // null terminate

		// make wstring
		widenStr = std::wstring( wcharBuf );

		// delete buffer and return value
		delete wcharBuf;
		return widenStr;
#		endif
	}
} // namespace String


#ifdef _WIN32
//-------------------------------------
//   function	ShellExecuteSync
//! @brief		Execute external program with shell, and wait until the process ends.
//! @param		showWindow	... if false, the program's window will be hid
//! @retval		 0	... success
//! @retval		-1	... failure
//-------------------------------------
inline static short
ShellExecuteSync( const char * commandLine, bool showWindow = true )
{
	unsigned long		result;
	STARTUPINFOA		sinfo	= {0};
	PROCESS_INFORMATION pinfo	= {0};

	// set the process to create
	sinfo.cb			= sizeof( STARTUPINFO );
	sinfo.dwFlags		= STARTF_USESHOWWINDOW;
	sinfo.wShowWindow	= (unsigned short)( showWindow?SW_SHOW:SW_HIDE );

	// create process
	CreateProcessA( NULL, const_cast<char*>(commandLine), NULL, NULL, false, 0, NULL, NULL, &sinfo, &pinfo );
	CloseHandle( pinfo.hThread );

	// wait until the process ends
	result = WaitForSingleObject( pinfo.hProcess, INFINITE );
	if ( result != WAIT_OBJECT_0 )
	{
		return -1;
	}
	
	CloseHandle( pinfo.hProcess );
	
	return 0;
}
#endif // _WIN32


} // namespace SGLib


#endif // __SGLIB_H__
