// file: SgRichEdit.cpp
// brief: a customed RichEdit for source code editing.
// author: YAMAMOTO Suguru
// update: 2010-09-05
// license: zlib License (see END of this file)
//=========================================================
#include "SgRichEdit.h"
#include "WordLogic.h"
#include "AccObj.h"
#include <string>
#include <map>
//#		include "Sgry.h"

using namespace std;

//=========================================================
//
//  types
//
//=========================================================

// data structure for additional feature of SgRichEdit
class CustomData
{
	public:
		CustomData()
		{
			indentMode = SRE_AIM_None;
			accName = L"";
			accDesc = L"";
			useMsaa = false;
			useSmartHome = false;
			hookProc = NULL;
			limitLen = 65536;
		}

		AutoIndentMode	indentMode;
		wstring			accName;
		wstring			accDesc;
		bool			useMsaa;
		bool			useSmartHome;
		SRE_WMHOOKPROC	hookProc;
		int				limitLen;
};

typedef struct _MyThreadData
{
	HWND	parent;
	DWORD	windowStyle;
	HWND	richEdit;
} MyThreadData;

//=========================================================
//
//  global data
//
//=========================================================
static map<HWND, CustomData> gCustomData;
static WNDPROC gRichEditWndProc = NULL;

#define RICHEDIT_DIRECT_FUNCTION	gRichEditWndProc
#include "RichEditGlues.h"


//=========================================================
//
//  inner function prototypes
//
//=========================================================
static LRESULT CALLBACK CustomWindowProc( HWND window, UINT message, WPARAM wParam, LPARAM lParam );
static void DoSmartHomeAction( HWND window, bool withShift );
static void DoShiftEndAction( HWND window );
static LRESULT OnCloseCurlyBracketInCIndentMode( HWND window, bool & outProcessed );
static LRESULT DoAutoIndent( HWND window, bool & outProcessed );
static bool IsShiftKeyDown();
static bool IsControlKeyDown();

#if defined( _WIN32 )
#	define SetWindowLongPtrW_(w, i, l)  SetWindowLongPtrW( (w), (i), (LONG)(l) )
#elif defined( _WIN64 )
#	define SetWindowLongPtrW_(w, i, l)  SetWindowLongPtrW( (w), (i), (l) )
#endif

//=========================================================
//
//  functions
//
//=========================================================

//-------------------------------------
// function : CreateSgRichEdit
// brief : create a SgRichEdit window
//-------------------------------------
HWND WINAPI
CreateSgRichEdit( HWND parent, DWORD style )
{
	BOOL ok;
	LONG_PTR rc;
	HWND window;
	WNDCLASSW wndClass;

	if( parent != NULL )
	{
		style |= WS_CHILD;
	}

	// retrieve original window procedure of RichEdit
	if( gRichEditWndProc == NULL )
	{
		// get info about the window class
		ok = GetClassInfoW( NULL, L"RichEdit20W", &wndClass );
		if( ok == FALSE )
		{
			return NULL;
		}
		
		// retrieve window procedure of RichEdit
		gRichEditWndProc = wndClass.lpfnWndProc;
		if( gRichEditWndProc == NULL )
		{
			return NULL;
		}
	}

	// create window
	window = CreateWindowW( L"RichEdit20W", L"",
			style,
			0, 0,
			300, 300,
			parent, NULL,
			GetModuleHandle(NULL), NULL );
	if( window == NULL )
	{
		// failed to create window.
		return NULL;
	}

	// prepare custom data for this window
	gCustomData[window] = CustomData();

	// set custom word break logic
	RichEdit_Exec( window, EM_SETWORDBREAKPROC, 0, (LPARAM)&SgRichEdit_WordBreak );

	// set custom word break logic
	RichEdit_Exec( window, EM_LIMITTEXT, (WPARAM)gCustomData[window].limitLen, 0 );

	// rewrite window procedure
	rc = SetWindowLongPtrW_( window, GWLP_WNDPROC, (LONG_PTR)&CustomWindowProc );
	if( rc == 0 )
	{
		DestroyWindow( window );
		return NULL;
	}

	return window;
}

//-------------------------------------
// function : CustomWindowProc
// brief : customized window procedure
//-------------------------------------
static LRESULT CALLBACK
CustomWindowProc( HWND window, UINT message, WPARAM wParam, LPARAM lParam )
{
	// if preview proc exists, call it
	if( gCustomData[window].hookProc != NULL )
	{
		LRESULT rc;
		BOOL executed = gCustomData[window].hookProc( window, message, wParam, lParam, &rc );
		if( executed )
		{
			return rc;
		}
	}
	
	// override key handling
	if( WM_KEYFIRST <= message && message <= WM_KEYLAST )
	{
		int key = (int)wParam;
//Sgry::LogFile("key event(%d)", message);

		// ignore Tab with Ctrl key
		if( key == VK_TAB && IsControlKeyDown() )
		{
			return 1;
		}
		
		// do smart "Home" action if the option was set so
		if( message == WM_KEYDOWN
			&& key == VK_HOME
			&& IsControlKeyDown() == false
			&& gCustomData[window].useSmartHome )
		{
			DoSmartHomeAction( window, IsShiftKeyDown() );
			return 1;
		}

		// do customed Shift+End action
		if( message == WM_KEYDOWN
			&& key == VK_END
			&& IsControlKeyDown() == false
			&& IsShiftKeyDown() )
		{
			DoShiftEndAction( window );
			return 1;
		}

		// if any selection exists and Ctrl + Delete/Back was pressed, delete selection.
		// (by default, pressing Ctrl + Delete/Back with selection causes
		// same action as pressing that with no selection.)
		if( message == WM_KEYDOWN
			&& IsControlKeyDown()
			&& (key == VK_DELETE || key == VK_BACK)
			&& RichEdit_IsReadOnly(window) == false )
		{
			int selStart, selEnd;

			RichEdit_GetSelection( window, selStart, selEnd );
			if( 0 < selEnd-selStart )
			{
				RichEdit_Delete( window );
				return 1;
			}
		}

		// if it's in C indent mode, and user input closes curly bracket, unindent
		if( message == WM_CHAR
			&& (int)key == L'}'
			&& RichEdit_IsReadOnly(window) == false
			&& gCustomData[window].indentMode == SRE_AIM_C )
		{
			bool processed;
			LRESULT rc = OnCloseCurlyBracketInCIndentMode( window, processed );
			if( processed )
				return rc;
		}

		// do auto indent
		if( message == WM_KEYDOWN
			&& key == VK_RETURN
			&& RichEdit_IsReadOnly(window) == false )
		{
			bool processed;
			LRESULT rc = DoAutoIndent( window, processed );
			if( processed )
				return rc;
		}

		// on entering character, expand internal limit if needed
		if( RichEdit_IsReadOnly(window) == false )
		{
			if( message == WM_CHAR
				|| message == WM_UNICHAR
				//|| (message == WM_KEYDOWN && IsControlKeyDown() && key == L'V' ) // I do not know why but RichEdit never receives this so preprocessing seems to be inpossible...
				)
			{
				int textLen = RichEdit_Exec( window, WM_GETTEXTLENGTH, 0, 0 );
				int limitLen = gCustomData[window].limitLen;
				if( limitLen < textLen+32 )
				{
					limitLen = textLen + (textLen % 4096);
					RichEdit_Exec( window, EM_LIMITTEXT, (WPARAM)limitLen, 0 );
					gCustomData[window].limitLen = limitLen;
				}
			}
		}
	}

	// handle additional messages
	switch( message )
	{
		case SRE_SETSMARTHOME:
			gCustomData[window].useSmartHome = (wParam != 0 ? 1 : 0);
			return 0;
			break;

		case SRE_GETSMARTHOME:
			return (gCustomData[window].useSmartHome ? 1 : 0);
			break;

		case SRE_SETAUTOINDENTMODE:
			gCustomData[window].indentMode = (AutoIndentMode)wParam;
			return 0;
			break;

		case SRE_GETAUTOINDENTMODE:
			return (AutoIndentMode)gCustomData[window].indentMode;
			break;

		case SRE_SETACCNAME:
		{
			gCustomData[window].accName = (wchar_t*)wParam;
			return 0;
		}
		break;

		case SRE_GETACCNAME:
		{
			wstring & accName = gCustomData[window].accName;
			if( lParam < (LPARAM)accName.length() )
				return 1;

			wcscpy( (wchar_t*)wParam, accName.c_str() );
			return 0;
		}
		break;

		case SRE_SETACCDESCRIPTION:
		{
			gCustomData[window].accDesc = (wchar_t*)wParam;
			return 0;
		}
		break;

		case SRE_GETACCDESCRIPTION:
		{
			wstring & accDesc = gCustomData[window].accDesc;
			if( lParam < (LPARAM)accDesc.length() )
				return 1;

			wcscpy( (wchar_t*)wParam, accDesc.c_str() );
			return 0;
		}
		break;

		case SRE_SETENABLEMSAA:
		{
			bool enable = (wParam != 0);
			gCustomData[window].useMsaa = enable;
			return 0;
		}
		break;

		case SRE_SETWMHOOKPROC:
		{
			gCustomData[window].hookProc = (SRE_WMHOOKPROC)wParam;
			return 0;
		}
		break;

		case WM_GETOBJECT:
		{
			if( gCustomData[window].useMsaa == false ) // (this always false now)
				break;

			if( lParam == OBJID_CLIENT
				|| lParam == OBJID_WINDOW )
			{
				SgRichEdit::AccObj * pObj;
				LRESULT lObj;

				pObj = new SgRichEdit::AccObj( window );
				lObj = LresultFromObject(
						IID_IAccessible,
						wParam,
						dynamic_cast<IAccessible*>( pObj )
					);
				pObj->Release(); // decrement reference count back

				return lObj;
			}
		}
		break;

		// remove custom data when the window was destroyed
		case WM_NCDESTROY:
		{
			gCustomData.erase( window );
			//DO_NOT//return 0;
		}
		break;
	}

	return gRichEditWndProc( window, message, wParam, lParam );
}

//-------------------------------------
// function : DoSmartHomeAction
// brief : move caret to the first non-whitespace char in the line
//-------------------------------------
static void
DoSmartHomeAction( HWND window, bool withShift )
{
	int i;
	int caretIndex = RichEdit_GetCaretIndex( window );
	int caretLineIndex, caretLineHeadIndex;
	wstring line;
	int newCaretIndex = -1;

	// get content of this line
	caretLineIndex = RichEdit_GetLineIndexFromCharIndex( window, caretIndex );
	caretLineHeadIndex = RichEdit_GetLineHeadIndex( window, caretLineIndex );
	line = RichEdit_GetLine( window, caretLineIndex );

	// find first non-white char in this line
	for( i=0; i<(int)line.length(); i++ )
	{
		wchar_t ch = line[i];
		if( ch != L'\t' && ch != L' ' && ch != L'\x3000' )
		{
			// found first non-whitespace char
			//NO_NEED//newCaretIndex = caretLineHeadIndex + i;
			break;
		}
	}
	newCaretIndex = caretLineHeadIndex + i;

	// if the caret is already at the first non-whitespace char, move caret to the line head
	if( newCaretIndex == caretIndex && 0 < newCaretIndex )
	{
		newCaretIndex = caretLineHeadIndex;
	}

	// apply new selection
	if( withShift )
	{
		RichEdit_SetSelection( window, RichEdit_GetSelectionAnchor(window), newCaretIndex );
	}
	else
	{
		RichEdit_SetSelection( window, newCaretIndex, newCaretIndex );
	}
}

//-------------------------------------
// function : DoShiftEndAction
// brief : select between current caret and line end
//-------------------------------------
static void
DoShiftEndAction( HWND window )
{
	int caretIndex = RichEdit_GetCaretIndex( window );
	int selAnchor = RichEdit_GetSelectionAnchor( window );
	int caretLineIndex, caretLineHeadIndex, caretLineLength, caretLineEndIndex;

	// get indexes around the caret line
	caretLineIndex = RichEdit_GetLineIndexFromCharIndex( window, caretIndex );
	caretLineHeadIndex = RichEdit_GetLineHeadIndex( window, caretLineIndex );
	caretLineLength = RichEdit_GetLineLengthFromCharIndex( window, caretLineHeadIndex );
	caretLineEndIndex = caretLineHeadIndex + caretLineLength;

	// if the action results nothing, exit
	if( caretIndex == caretLineEndIndex )
	{
		return;
	}

	// select between selection-anchor and new-caret-index
	RichEdit_SetSelection( window, selAnchor, caretLineEndIndex );
}

//-------------------------------------
// function : OnCloseCurlyBracketInCIndentMode
// brief : unindent
//-------------------------------------
static LRESULT
OnCloseCurlyBracketInCIndentMode( HWND window, bool & outProcessed )
{
	wstring line, indent = L"";
	int caretIndex, caretLineIndex;
	int selStart, selEnd;
	size_t i;

	// get indent chars
	caretIndex = RichEdit_GetCaretIndex( window );
	caretLineIndex = RichEdit_GetLineIndexFromCharIndex( window, caretIndex );
	line = RichEdit_GetLine( window, caretLineIndex );
	for( i=0; i<line.length(); i++ )
	{
		if( line[i] == L' ' || line[i] == L'\t' )
		{
			indent += line[i];
		}
		else
		{
			outProcessed = false;
			return 0; // not empty line
		}
	}

	// expand selection to 1 before char
	RichEdit_GetSelection( window, selStart, selEnd );
	if( RichEdit_GetLineHeadIndex(window, caretLineIndex) < selStart )
	{
		RichEdit_SetSelection( window, selStart - 1, selEnd );
	}
	
	// then, replace them into close curly bracket
	RichEdit_Replace( window, L"}" );

	outProcessed = true;
	return 0;
}


//-------------------------------------
// function : DoAutoIndent
// brief : execute auto indent
//-------------------------------------
static LRESULT
DoAutoIndent( HWND window, bool & outProcessed )
{
	size_t i;
	wstring line, indent;
	int caretIndex, caretLineIndex;

	switch( gCustomData[window].indentMode )
	{
		case SRE_AIM_General:
		{
			// get indent chars
			caretIndex = RichEdit_GetCaretIndex( window );
			caretLineIndex = RichEdit_GetLineIndexFromCharIndex( window, caretIndex );
			line = RichEdit_GetLine( window, caretLineIndex );
			indent = L"\r";
			for( i=0; i<line.length(); i++ )
			{
				if( line[i] == L' ' || line[i] == L'\t' )
					indent += line[i];
				else
					break;
			}

			// end line and do auto indent
			RichEdit_Insert( window, indent.c_str() );

			outProcessed = true;
			return 0;
		}
		break;

		case SRE_AIM_C:
		{
			int caretColumnIndex;

			// get indent chars
			RichEdit_GetCaretIndex( window, caretLineIndex, caretColumnIndex );
			line = RichEdit_GetLine( window, caretLineIndex );
			indent = L"\r";
			for( i=0; i<line.length(); i++ )
			{
				if( line[i] == L' ' || line[i] == L'\t' )
					indent += line[i];
				else
					break;
			}

			// if there is an '{' before caret
			// and is no '}' after caret, add indentation
			if( caretColumnIndex < (int)line.length()
				&& line.find_last_of(L'{', caretColumnIndex) != wstring::npos
				&& line.find_first_of(L'}', caretColumnIndex) == wstring::npos )
			{
				indent += L'\t';
			}
			else if( caretColumnIndex == (int)line.length() )
			{
				size_t lineLength = line.length();
				if( 0 < lineLength
					&& line.at(lineLength-1) == L'{' )
				{
					indent += L'\t';
				}
			}

			// end line and do auto indent
			RichEdit_Insert( window, indent.c_str() );

			outProcessed = true;
			return 0;
		}
		break;

		default:
		{
			outProcessed = false;
			return 0;
		}
		break;
	}
}

static bool
IsShiftKeyDown()
{
	short state = GetKeyState( VK_SHIFT );
	return ( state < 0 );
}

static bool
IsControlKeyDown()
{
	short state = GetKeyState( VK_CONTROL );
	return ( state < 0 );
}


/**********************************************************
Copyright (C) 2006-2010 YAMAMOTO Suguru

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
**********************************************************/
