// file: RichEditGlues.h
// brief: routines for RichEdit
// author: SGRY (YAMAMOTO Suguru)
// update: 2007-12-08
// license: zlib License (see END of this file)
//=========================================================
#include <richedit.h>
#include <string>

// if user defines RICHEDIT_DIRECT_FUNCTION as the wndproc of RichEdit,
// these routines prefer to call WndProc than to call SendMessage.
#ifdef RICHEDIT_DIRECT_FUNCTION
#define RichEdit_Exec( window, message, wParam, lParam ) \
		RICHEDIT_DIRECT_FUNCTION( window, message, wParam, lParam )
#else
#define RichEdit_Exec( window, message, wParam, lParam ) \
		SendMessageW( window, message, wParam, lParam )
#endif


#ifndef EM_SETTEXTEX

	#define EM_SETTEXTEX	(WM_USER + 97)
	#define ST_DEFAULT		0
	#define ST_KEEPUNDO		1
	#define ST_SELECTION	2
	#define ST_NEWCHARS 	4

	typedef struct _settextex {
	    DWORD flags;
	    UINT codepage;
	} SETTEXTEX;

#endif

static inline bool
RichEdit_IsReadOnly( HWND window )
{
	int style = GetWindowLong( window, GWL_STYLE );
	return (style & ES_READONLY) != 0;
}

static inline void
RichEdit_Delete( HWND window )
{
	RichEdit_Exec( window, WM_CLEAR, 0, 0 );
}

static inline void
RichEdit_Replace( HWND window, const wchar_t* text, bool addUndoBuffer = true )
{
	RichEdit_Exec( window, EM_REPLACESEL, (WPARAM)addUndoBuffer, (LPARAM)text );
}

static inline void
RichEdit_Insert( HWND window, const wchar_t* text )
{
	SETTEXTEX st;
	st.codepage = 1200; // Unicode
	st.flags = ST_KEEPUNDO | ST_SELECTION;
	
	RichEdit_Exec( window, EM_SETTEXTEX, (WPARAM)&st, (LPARAM)text );
}

static inline void
RichEdit_GetSelection( HWND window, int & selectionStart, int & selectionEnd )
{
	RichEdit_Exec( window, EM_GETSEL, (WPARAM)&selectionStart, (LPARAM)&selectionEnd );
}

static inline void
RichEdit_SetSelection( HWND window, int selectionAnchor, int selectionEnd )
{
	RichEdit_Exec( window, EM_SETSEL, (WPARAM)selectionAnchor, (LPARAM)selectionEnd );
}

static inline void
RichEdit_GetLine( HWND window, int lineIndex, wchar_t* outText, int maxLength )
{
	outText[0] = (wchar_t)maxLength;
	RichEdit_Exec( window, EM_GETLINE, (WPARAM)lineIndex, (LPARAM)outText );
}

static inline int
RichEdit_GetLineLengthFromCharIndex( HWND window, int charIndex )
{
	return (int)RichEdit_Exec( window, EM_LINELENGTH, (WPARAM)charIndex, (LPARAM)0 );
}

static inline int
RichEdit_GetLineIndexFromCharIndex( HWND window, int charIndex )
{
	return (int)RichEdit_Exec( window, EM_LINEFROMCHAR, (WPARAM)charIndex, (LPARAM)0 );
}

static inline int
RichEdit_GetCaretLineIndex( HWND window )
{
	return (int)RichEdit_Exec( window, EM_LINEFROMCHAR, (WPARAM)-1, (LPARAM)0 );
}

static inline int
RichEdit_GetLineHeadIndex( HWND window, int lineIndex )
{
	return (int)RichEdit_Exec( window, EM_LINEINDEX, (WPARAM)lineIndex, (LPARAM)0 );
}

static inline void
RichEdit_GetLineColumnPosFromCharIndex( HWND window, int charIndex, int & outLineIndex, int & outColumnIndex )
{
	int lineHeadIndex;

	// sCfbNX擾
	outLineIndex = RichEdit_GetLineIndexFromCharIndex( window, charIndex );
	
	// ̍s̐擪̃CfbNX擾
	lineHeadIndex = RichEdit_GetLineHeadIndex( window, outLineIndex );

	// CfbNX擾
	outColumnIndex = charIndex - lineHeadIndex;
}





static inline void
RichEdit_GetScreenPosFromCharIndex( HWND window, int charIndex, int & x, int & y )
{
	POINTL	pos;
	RichEdit_Exec( window, EM_POSFROMCHAR, (WPARAM)&pos, (LPARAM)charIndex );
	x = pos.x;
	y = pos.y;
}

static inline int
RichEdit_GetCharIndexFromScreenPos( HWND window, int x, int y )
{
	POINTL	pos;
	pos.x = x;
	pos.y = y;

	return (int)RichEdit_Exec( window, EM_CHARFROMPOS, (WPARAM)0, (LPARAM)&pos );
}







static inline int
RichEdit_GetCaretIndex( HWND window )
{
	int selStart, selEnd;
	POINT caretScreenPos;
	int caretIndex;
	
	// I͈͂擾
	RichEdit_GetSelection( window, selStart, selEnd );

	// IԂȂIJnʒuIIʒuLbgʒu
	if( selStart == selEnd )
	{
		return selStart;
	}

	// IȂALbgIJnʒuɂ邩XN[WŔf
	GetCaretPos( &caretScreenPos );
	caretIndex = RichEdit_GetCharIndexFromScreenPos( window, caretScreenPos.x, caretScreenPos.y );
	if( selStart == caretIndex )
	{
		return selStart;
	}
	else
	{
		return selEnd;
	}
}

static inline void
RichEdit_GetCaretIndex( HWND window, int & outLine, int & outColumn )
{
	RichEdit_GetLineColumnPosFromCharIndex( window, RichEdit_GetCaretIndex(window), outLine, outColumn );
}

/*
static inline void
RichEdit_SetCaretIndex( HWND window, int index )
{
	int x, y;

	RichEdit_GetScreenPosFromCharIndex( window, index, x, y );
	SetCaretPos( x, y );
}
*/

static inline int
RichEdit_GetSelectionAnchor( HWND window )
{
	int selStart, selEnd, selAnchor;
	RichEdit_GetSelection( window, selStart, selEnd );

	// IȂA֌đI𒆂Ƃ
	if( selStart == selEnd )
	{
		selAnchor = selStart;
		return selAnchor;
	}

	// I͈͂̐擪ɃLbg΁A擪ɌđILĂŒ
	if( selStart == RichEdit_GetCaretIndex(window) )
	{
		selAnchor = selEnd;
	}
	else
	{
		selAnchor = selStart;
	}

	return selAnchor;
}

static inline std::wstring
RichEdit_GetLine( HWND window, int lineIndex )
{
	std::wstring line;
	int lineLength, lineHeadIndex;
	
	// get informations needed to get line content
	lineHeadIndex = RichEdit_GetLineHeadIndex( window, lineIndex );
	lineLength = RichEdit_GetLineLengthFromCharIndex( window, lineHeadIndex );
	
	// then, get line content
	wchar_t * buf = new wchar_t[ lineLength+1 ];
	{
		buf[0] = (TCHAR)lineLength;
		RichEdit_Exec( window, EM_GETLINE, (WPARAM)lineIndex, (LPARAM)buf );
		buf[ lineLength ] = 0;
		line = buf;
	}
	delete buf;

	return line;
}

/**********************************************************
Copyright (C) 2006-2009 YAMAMOTO Suguru

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
**********************************************************/
