// file: AccObj.cpp
// brief: accessible object for SgRichEdit component
// author: SGRY (YAMAMOTO Suguru)
// update: 2007-04-30
// license: zlib License (see END of this file)
//=========================================================
#include "AccObj.h"
#include "OleWindowAdapter.h"
#include <assert.h>

namespace SgRichEdit {

//=========================================================
//  Init / Dispose
//=========================================================

//-------------------------------------
// function : AccObj::ctor
//-------------------------------------
AccObj::AccObj( HWND sgRichEdit )
{
	MY_LOG("AccObj::ctor");
	HRESULT rc;

	_window = sgRichEdit;
	_refCount = 1;
	_oleWindowItf = new OleWindowAdapter( this );

	rc = CreateStdAccessibleProxyW( sgRichEdit, L"RichEdit20W", OBJID_CLIENT, IID_IAccessible, (void**)&_stdRichAccObj );
	if( rc != S_OK )
	{
		MY_LOG("    failed to create standard acc-proxy");
		assert( _stdRichAccObj != NULL );
		_stdRichAccObj = NULL;
	}

	_stdRichAccObj->AddRef();
}

//-------------------------------------
// function : AccObj::dtor
//-------------------------------------
AccObj::~AccObj()
{
	MY_LOG("AccObj::dtor");
	_stdRichAccObj->Release();
	delete _oleWindowItf;
}


//=========================================================
//  Others
//=========================================================

//-------------------------------------
// function : AccObj::GetWindow
//-------------------------------------
HWND
AccObj::GetWindow() const
{
	MY_LOG("AccObj::GetWindow");
	return _window;
}


//=========================================================
//  IUnknown
//=========================================================

//-------------------------------------
// function : AccObj::AddRef
//-------------------------------------
ULONG STDMETHODCALLTYPE
AccObj::AddRef()
{
	_refCount++;
	return _refCount;
}

//-------------------------------------
// function : AccObj::Release
//-------------------------------------
ULONG STDMETHODCALLTYPE
AccObj::Release()
{
	_refCount--;
	if( _refCount == 0 )
	{
		MY_LOG("AccObj::Release (deleting self)");
		delete this;
	}

	return _refCount;
}

//-------------------------------------
// function : AccObj::QueryInterface
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::QueryInterface( const IID & iid, void ** out_objects )
{
	//MY_LOG("AccObj::QueryInterface (%s)", Sgry::Com::GetInterfaceName(iid).c_str());
	if( iid == IID_IUnknown )
	{
		*out_objects = dynamic_cast<IUnknown*>( this );
		return S_OK;
	}
	else if( iid == IID_IDispatch )
	{
		*out_objects = dynamic_cast<IDispatch*>( this );
		return S_OK;
	}
	else if( iid == IID_IAccessible )
	{
		*out_objects = dynamic_cast<IAccessible*>( this );
		return S_OK;
	}
	else if( iid == IID_IOleWindow )
	{
		*out_objects = _oleWindowItf;
		return S_OK;
	}

	*out_objects = NULL;
	return E_NOINTERFACE;
}


//=========================================================
//  IDispatch
//=========================================================

//-------------------------------------
// function : AccObj::GetTypeInfoCount
// brief : get the number of type information that this COM-object provides
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::GetTypeInfoCount( UINT * out_typeInfoCount )
{
	MY_LOG("AccObj::GetTypeInfoCount");
	*out_typeInfoCount = 0;
	return E_NOTIMPL;
}

//-------------------------------------
// function : AccObj::GetTypeInfo
// brief : get type information of this COM-object
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::GetTypeInfo( UINT /*typeInfoIndex*/, LCID /*lcid*/, ITypeInfo ** out_typeInfo )
{
	MY_LOG("AccObj::GetTypeInfo");
	*out_typeInfo = 0;
	return E_NOTIMPL;
}

//-------------------------------------
// function : AccObj::GetIDsOfNames
// brief : get the COM-object's member IDs by member names
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::GetIDsOfNames( const IID & /*iid*/, OLECHAR ** /*names*/, UINT /*nameCount*/, LCID /*lcid*/, DISPID * out_dispId )
{
	MY_LOG("AccObj::GetIDsOfNames");
	*out_dispId = 0;
	return E_NOTIMPL;
}

//-------------------------------------
// function : AccObj::Invoke
// brief : invoke the method specified with ID
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::Invoke( DISPID, const IID&, LCID, WORD, DISPPARAMS*, VARIANT*, EXCEPINFO*, UINT* )
{
	MY_LOG("AccObj::Invoke");
	return E_NOTIMPL;
}


//=========================================================
//  IAccessible
//=========================================================

//-------------------------------------
// function : AccObj::get_accParent
// brief : get IDispatch interface of the parent UI element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accParent( IDispatch ** out_parentDispatchItf )
{
	MY_LOG("AccObj::get_accParent");
	// get parent
	HWND parent = GetParent( _window );
	if( parent == NULL )
	{
		*out_parentDispatchItf = NULL;
		return S_FALSE;
	}

	// get parent's accessible object
	HRESULT rc = AccessibleObjectFromWindow( parent, OBJID_WINDOW, IID_IDispatch, (void**)out_parentDispatchItf );
	if( rc != S_OK )
	{
		return S_FALSE;
	}

	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accChildCount
// brief : count child elements contained in this UI Element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accChildCount( long * out_childrenCount )
{
	MY_LOG("AccObj::get_accChildCount");
	*out_childrenCount = 0;
	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accChild
// brief : get IDispatch interface of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accChild( VARIANT childId, IDispatch ** out_childDispItf )
{
	MY_LOG("AccObj::get_accChild");
	return _stdRichAccObj->get_accChild( childId, out_childDispItf );
}

//-------------------------------------
// function : AccObj::get_accName
// brief : get the name of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accName( VARIANT /*child*/, BSTR * out_name )
{
	MY_LOG("AccObj::get_accName");
	wchar_t accName[1024];
	LRESULT rc = SendMessage( _window, SRE_GETACCNAME, (WPARAM)accName, 1023 );
	if( rc == 0 )
	{
		SysReAllocString( out_name, accName );
	}

	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accValue
// brief : get the value of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accValue( VARIANT child, BSTR * out_value )
{
	MY_LOG("AccObj::get_accValue");
	return _stdRichAccObj->get_accValue( child, out_value );
}

//-------------------------------------
// function : AccObj::get_accDescription
// brief : get description of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accDescription( VARIANT /*child*/, BSTR * out_description )
{
	MY_LOG("AccObj::get_accDescription");
	wchar_t accDescription[1024];
	LRESULT rc = SendMessage( _window, SRE_GETACCDESCRIPTION, (WPARAM)accDescription, 1023 );
	if( rc == 0 )
	{
		SysReAllocString( out_description, accDescription );
	}

	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accRole
// brief : get role information of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accRole( VARIANT /*child*/, VARIANT * out_role )
{
	MY_LOG("AccObj::get_accRole");
	out_role->vt = VT_I4;
	out_role->lVal = ROLE_SYSTEM_TEXT;
	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accState
// brief : get current state of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accState( VARIANT /*child*/, VARIANT * out_state )
{
	MY_LOG("AccObj::get_accState");
	out_state->vt = VT_I4;
	out_state->lVal = STATE_SYSTEM_DEFAULT;
	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accHelp
// brief : get help property string for the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accHelp( VARIANT /*child*/, BSTR * out_help )
{
	MY_LOG("AccObj::get_accHelp");
	*out_help = 0;
	return S_FALSE;
}

//-------------------------------------
// function : AccObj::get_accHelpTopic
// brief : get help file path and its topic for the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accHelpTopic( BSTR * out_helpFile, VARIANT child, long * out_idTopic )
{
	MY_LOG("AccObj::get_accHelpTopic");
	*out_helpFile = 0;
	child.vt = VT_EMPTY;
	*out_idTopic = 0;

	return S_FALSE;
}

//-------------------------------------
// function : AccObj::get_accKeyboardShortcut
// brief : get keyboard shortcut (mnemonic) for the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accKeyboardShortcut( VARIANT /*child*/, BSTR * out_keyboardShortcut )
{
	MY_LOG("AccObj::get_accKeyboardShortcut");
	*out_keyboardShortcut = 0;
	return S_FALSE;
}

//-------------------------------------
// function : AccObj::get_accFocus
// brief : get ID of the focused child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::get_accFocus( VARIANT * out_id )
{
	MY_LOG("AccObj::get_accFocus");
	out_id->vt = VT_I4;
	out_id->lVal = CHILDID_SELF;

	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accSelection
// brief : get ID of the selected child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accSelection( VARIANT * out_children )
{
	MY_LOG("AccObj::get_accSelection");
	out_children->vt = VT_I4;
	out_children->lVal = CHILDID_SELF;
	return S_OK;
}

//-------------------------------------
// function : AccObj::get_accDefaultAction
// brief : do default action for this UI element
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::get_accDefaultAction( VARIANT /*child*/, BSTR * out_defaultAction )
{
	MY_LOG("AccObj::get_accDefaultAction");
	SysReAllocString( out_defaultAction, L"" );
	return S_OK;
}

//-------------------------------------
// function : AccObj::accSelect
// brief : select specified child
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::accSelect( long flagsSelect, VARIANT /*child*/ )
{
	MY_LOG("AccObj::accSelect");
	switch( flagsSelect )
	{
		case SELFLAG_TAKEFOCUS:
		case SELFLAG_TAKESELECTION:
			::SetFocus( _window );
			return S_OK;

		default:
			return S_FALSE;
	}
}

//-------------------------------------
// function : AccObj::accLocation
// brief : get location of the specified child element
//-------------------------------------
HRESULT STDMETHODCALLTYPE
AccObj::accLocation( long * out_x, long * out_y,
		long * out_width, long * out_height,
		VARIANT /*child*/ )
{
	MY_LOG("AccObj::accLocation");
	RECT rect;
	BOOL okay;

	okay = GetWindowRect( _window, &rect );
	if( okay )
	{
		*out_x = rect.left;
		*out_y = rect.top;
		*out_width = rect.right - rect.left;
		*out_height = rect.bottom - rect.top;
		
		return S_OK;
	}

	return S_FALSE;
}

//-------------------------------------
// function : AccObj::accNavigate
// brief : navigate through child elements (if this UI element is a container)
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::accNavigate( long /*navDir*/, VARIANT start, VARIANT * out_end )
{
	MY_LOG("AccObj::accNavigate");
	if( start.vt != VT_I4
		|| start.lVal != CHILDID_SELF )
	{
		out_end->vt = VT_EMPTY;
		return S_FALSE;
	}

	out_end->vt = VT_EMPTY;
	return S_FALSE;
}

//-------------------------------------
// function : AccObj::accHitTest
// brief : retrieve the element at given coordinate
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::accHitTest( long /*x*/, long /*y*/, VARIANT * out_child )
{
	MY_LOG("AccObj::accHitTest");
	out_child->vt = VT_I4;
	out_child->lVal = CHILDID_SELF;
	return S_OK;
}

//-------------------------------------
// function : AccObj::accDoDefaultAction
// notes : deprecated API
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::accDoDefaultAction( VARIANT /*child*/ )
{
	MY_LOG("AccObj::accDoDefaultAction");
	// do nothing
	return S_OK;
}

//-------------------------------------
// function : AccObj::put_accName
// notes : deprecated API
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::put_accName( VARIANT /*child*/, BSTR /*name*/ )
{
	MY_LOG("AccObj::put_accName");
	return E_NOTIMPL;
}

//-------------------------------------
// function : AccObj::put_accValue
// notes : deprecated API
//-------------------------------------
HRESULT STDMETHODCALLTYPE 
AccObj::put_accValue( VARIANT /*child*/, BSTR /*value*/ )
{
	MY_LOG("AccObj::put_accValue");
	return E_NOTIMPL;
}

} // namespace SgRichEdit

/**********************************************************
Copyright (C) 2006-2009 YAMAMOTO Suguru

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.

2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
**********************************************************/
