// file    : RegisterTo95Reader.c
// brief   : register AiB Tools as 95Reader's MSAA reading target
// author  : SGRY (YAMAMOTO Suguru)
// update  : 2007-02-24 (SGRY)
// version : 1.0.0
// license : MIT License (see END of this file)
//=========================================================
#include <windows.h>
#include <stdio.h>

#if _MSC_VER > 0x500
#	pragma warning( disable:4996 ) // ignore "unsafe strxxx was used" warning by VC8
#endif


//=========================================================
//
//  constants / types
//
//=========================================================
#define null	0

// VC6 has not PSAPI. So load and use some API dynamically from the DLL.
typedef BOOL (WINAPI * EnumProcessModulesProc)( HANDLE process, HMODULE* outModuleAry, DWORD arySize, DWORD* outNeededBufSize );
EnumProcessModulesProc EnumProcessModules_ = null;

typedef DWORD (WINAPI * GetModuleFileNameExAProc)( HANDLE process, HMODULE module, char* outFilePath, DWORD maxPathLen );
GetModuleFileNameExAProc GetModuleFileNameExA_ = null;


//=========================================================
//
//  inner function prototypes
//
//=========================================================
static int Get95ReaderAppPath( char* outPath, int maxLength );
static BOOL FileExists( const char* path );
static int RegisterApp( const char* iniPath, const char* appPath, BOOL doRegister );


//=========================================================
//
//  functions
//
//=========================================================

//-------------------------------------
// function : main
// brief    : application entry point
//-------------------------------------
int
main( int argc, char* argv[] )
{
	int		i;
	int		rc; // result code
	char	iniPath[ MAX_PATH ];
	char	aibPath[ MAX_PATH ];
	char	path[ MAX_PATH ];
	char *	bsPos;
	BOOL	doRegister = TRUE;

	// parse command line option
	for( i=1; i<argc; i++ )
	{
		// if one of the options is "-u", unregistration
		if( strcmp(argv[i], "-u") == 0 )
		{
			doRegister = FALSE;
			break;
		}
	}

	// get 95Reader's application directory
	rc = Get95ReaderAppPath( iniPath, MAX_PATH );
	if( rc != 0 )
	{
		return 1;
	}

	// make 95Reader's MSAA setting ini path
	bsPos = strrchr( iniPath, '\\' );
	if( bsPos == null )
	{
		return 1;
	}
	*bsPos = 0;
	strcat( iniPath, "\\APPDATA\\APPDATA.INI" );

	// get current directory path
	rc = GetModuleFileName( null, aibPath, MAX_PATH );
	if( rc == 0 )
	{
		return 1;
	}
	bsPos = strrchr( aibPath, '\\' );
	if( bsPos == null )
	{
		return 1;
	}
	*bsPos = 0;

	// register AiB Edit
	sprintf( path, "%s\\%s", aibPath, "AiBEditWin.exe" );
	RegisterApp( iniPath, path, doRegister );

	// register AiB Terminal
	sprintf( path, "%s\\%s", aibPath, "AiBTerminal.exe" );
	RegisterApp( iniPath, path, doRegister );

	// register _javac
	sprintf( path, "%s\\%s", aibPath, "_javac.exe" );
	RegisterApp( iniPath, path, doRegister );

	return 0;
}


//=========================================================
//
//  inner functions
//
//=========================================================

//-------------------------------------
// function : Get95ReaderAppPath
//-------------------------------------
static int
Get95ReaderAppPath( char* outPath, int maxLength )
{
	DWORD	pid = null;
	HWND	xprWnd;
	HANDLE	xprApp;
	HMODULE	xprModule;
	DWORD	actualLen;
	BOOL	ok;

	// get some PSAPI from psapi.dll
	{
		HMODULE	psApiLib;
		psApiLib = LoadLibrary( "psapi.dll" );
		if( psApiLib == null )
		{
			return -1; // Win9x?
		}
		EnumProcessModules_ = (EnumProcessModulesProc)GetProcAddress( psApiLib, "EnumProcessModules" );
		if( EnumProcessModules_ == null )
		{
			return -1;
		}
		GetModuleFileNameExA_ = (GetModuleFileNameExAProc)GetProcAddress( psApiLib, "GetModuleFileNameExA" );
		if( GetModuleFileNameExA_ == null )
		{
			return -1;
		}
	}

	// find application (not controller) window of 95Reader
	xprWnd = FindWindowA( "95Reader", null );
	if( xprWnd == null )
	{
		return -1;
	}
	
	// get application handle of the running 95Reader
	GetWindowThreadProcessId( xprWnd, &pid );
	xprApp = OpenProcess( PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, TRUE, pid );
	if( xprApp == null )
	{
		return -1;
	}
	
	// retrieve first module used in the process
	ok = EnumProcessModules_( xprApp, &xprModule, sizeof(HMODULE) * 1, &actualLen );
	if( !ok )
	{
		return -1;
	}

	// get path of the module
	actualLen = GetModuleFileNameExA_( xprApp, xprModule, outPath, maxLength );
	if( actualLen == 0 )
	{
		return -1;
	}

	return 0;
}

//-------------------------------------
// function : FileExists
// brief    : check whether specified file exists or not
//-------------------------------------
static BOOL
FileExists( const char* path )
{
	FILE * fp;

	fp = fopen( path, "rb" );
	if( fp == null )
	{
		return FALSE;
	}
	fclose( fp );

	return TRUE;
}

//-------------------------------------
// function : RegisterApp
// brief    : register/unregister specified application to MSAA target of 95Reader
//-------------------------------------
static int
RegisterApp( const char* iniPath, const char* appPath, BOOL doRegister )
{
	BOOL ok;

	// register?
	if( doRegister && FileExists(appPath) ) // do nothing if target does not exist
	{
		ok = WritePrivateProfileString( "MSAA", appPath, "1", iniPath );
		if( !ok )
		{
			return -1;
		}
	}
	// unregister?
	else
	{
		WritePrivateProfileString( "MSAA", appPath, null, iniPath );
	}

	return 0;
}


/**********************************************************
Copyright (c) 2007 YAMAMOTO Suguru

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
copies of the Software, and to permit persons to whom the Software is 
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
THE SOFTWARE.
**********************************************************/
