// file    : RedirectedProcess.cpp
// brief   : library for process execution with pipe
// author  : SGRY (YAMAMOTO Suguru)
// update  : 2007-04-30 (SGRY)
// version : 1.0.1
// license : MIT License (see END of this file)
//=========================================================
#define UNICODE
#define WIN32_LEAN_AND_MEAN
#if _MSC_VER > 0x500
#	pragma warning( disable:4996 )
#endif
#include <windows.h>
#include <string>
#include <iostream>

using namespace std;

//-----------------------------------------------------------------------------
//
//  Types
//
//-----------------------------------------------------------------------------
typedef struct _RedirectedProcess
{
	HANDLE	processHdl;
	DWORD	processId;
	HANDLE	outputHdl;
	HANDLE	inputHdl;
} RedirectedProcess;


//-----------------------------------------------------------------------------
//
//  Prototypes
//
//-----------------------------------------------------------------------------
extern "C" RedirectedProcess * RedirectedProcess_Create( const wchar_t * commandLine );
extern "C" int RedirectedProcess_Dispose( RedirectedProcess * rp );
extern "C" int RedirectedProcess_Read( RedirectedProcess * rp );
extern "C" int RedirectedProcess_ReadByte( RedirectedProcess * rp );
extern "C" int RedirectedProcess_PeekByte( RedirectedProcess * rp );
extern "C" int RedirectedProcess_WriteByte( RedirectedProcess * rp, int ch );
extern "C" int RedirectedProcess_Write( RedirectedProcess * rp, const wchar_t * str, int length );
extern "C" int RedirectedProcess_WaitForExit( RedirectedProcess * rp );
extern "C" int RedirectedProcess_Wait( RedirectedProcess * rp, int milliseconds );
extern "C" int RedirectedProcess_IsAlive( RedirectedProcess * rp );
extern "C" int RedirectedProcess_GetExitCode( RedirectedProcess * rp );
extern "C" int RedirectedProcess_SendCtrlBreak( RedirectedProcess * rp );
extern "C" int RedirectedProcess_SendCtrlC( RedirectedProcess * rp );

static int MyCreatePipes( HANDLE & stdOutWriteEnd, HANDLE & stdOutReadEnd, HANDLE & stdInWriteEnd, HANDLE & stdInReadEnd );
static wstring ToUnicode( const char * narrowStr, int strLength );
static string ToSJIS( const wchar_t * str, int length );
static std::string GetLastErrorMessage();
inline static void MyCloseHandle( HANDLE & handle );
inline static bool IsSjisFirstByte( char ch );

inline static void DebugPrint( const char * inFormat, ... );
inline static void DebugPrint( const wchar_t * inFormat, ... );


//-----------------------------------------------------------------------------
//
//  main
//
//-----------------------------------------------------------------------------
int
wmain( int argc, wchar_t* argv[] )
{
	int		ch;
	wchar_t	commandLine[1024];
	RedirectedProcess *	rp;
	
	// make command-line string to execute
	swprintf( commandLine, L"%s", argv[1] );
	for( int i=2; i<argc; i++ )
	{
		wcscat( commandLine, L" " );
		wcscat( commandLine, argv[i] );
	}
	
	// launch process
	rp = RedirectedProcess_Create( commandLine );
	if( rp == NULL )
	{
		DebugPrint( "failed to call RedirectedProcess_Create." );
		return -1;
	}
	
	// continue to get output
	ch = RedirectedProcess_ReadByte( rp );
	while( ch != -1 )
	{
		std::cout << (char)ch;
		
		// read next char
		ch = RedirectedProcess_ReadByte( rp );
	}
	
	// dispose resources
	RedirectedProcess_Dispose( rp );

	return 0;
}


//-----------------------------------------------------------------------------
//
//   exported functions
//
//-----------------------------------------------------------------------------

//-------------------------------------
// function : RedirectedProcess_Create
// brief    : Create process which stdin/stdout/stderr was redirected with pipes
// returns  : (success) ... RedirectedProcess object
//            (failure) ... NULL
// note     : use DisposeRedirectedProcess to release resources before exit.
//-------------------------------------
extern "C" RedirectedProcess *
RedirectedProcess_Create( const wchar_t * commandLine )
{
	const bool			inheritHandles		= true;
	const wchar_t *		currentDirectory	= NULL;
	const LONG			creationFlag		= CREATE_NEW_PROCESS_GROUP;
	
	BOOL				rc; // result code
	RedirectedProcess * obj;
	wchar_t				commandLine_[1024];
	STARTUPINFO			startupInfo		= {0};
	PROCESS_INFORMATION	processInfo		= {0};
	HANDLE				stdOutWriteEnd = 0, stdInReadEnd = 0;

	// create object
	obj = new RedirectedProcess;
	if( obj == NULL )
	{
		DebugPrint( "failed to create object; no enough memory" );
		goto error;
	}

	// create pipes with handle inheritance settings and more
	rc = MyCreatePipes( stdOutWriteEnd, obj->outputHdl, obj->inputHdl, stdInReadEnd );
	if( rc == -1 )
	{
		DebugPrint( "failed to create pipes" );
		goto error;
	}
	
	// prepare to launch process
	startupInfo.cb = sizeof( STARTUPINFO );
	startupInfo.dwFlags = STARTF_USESTDHANDLES;
	startupInfo.hStdInput = stdInReadEnd;
	startupInfo.hStdOutput = stdOutWriteEnd;
	startupInfo.hStdError = stdOutWriteEnd;

	// launch process
	wcscpy( commandLine_, commandLine );
	rc = CreateProcessW( NULL, commandLine_, NULL, NULL, inheritHandles, creationFlag, NULL, currentDirectory, &startupInfo, &processInfo );
	if( rc == FALSE )
	{
		DebugPrint( "failed to create process / %s", GetLastErrorMessage().c_str() );
		goto error;
	}

	// dispose unnecessary pipes
	MyCloseHandle( stdInReadEnd );
	MyCloseHandle( stdOutWriteEnd );
	
	// set needed info to the object's fields
	obj->processHdl = processInfo.hProcess;
	obj->processId = processInfo.dwProcessId;

	return obj;
error:
	CloseHandle( stdInReadEnd );
	CloseHandle( stdOutWriteEnd );
	if( obj != NULL )
	{
		delete obj;
	}
	return NULL;
}

//-------------------------------------
// function : RedirectedProcess_Dispose
// brief    : dispose RedirectedProcess object
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_Dispose( RedirectedProcess * rp )
{
	BOOL rc; // result code
	
	// close handle of the pipe connected to the stdin
	rc = CloseHandle( rp->inputHdl );
	if( rc == FALSE )
	{
		DebugPrint( "failed to close input handle of the process / %s", GetLastErrorMessage().c_str() );
		return -1;
	}
	rp->inputHdl = 0;

	// close handle of the pipe connected to the stdout & stderr
	rc = CloseHandle( rp->outputHdl );
	if( rc == FALSE )
	{
		DebugPrint( "failed to close output handle of the process / %s", GetLastErrorMessage().c_str() );
		return -1;
	}
	rp->outputHdl = 0;

	// try to terminate process
	GenerateConsoleCtrlEvent( CTRL_BREAK_EVENT, rp->processId );
	
	// close process
	rc = CloseHandle( rp->processHdl );
	if( rc == FALSE )
	{
		DebugPrint( "failed to close process handle / %s", GetLastErrorMessage().c_str() );
		return -1;
	}
	rp->processHdl = 0;
	rp->processId = 0;
	
	return 0;
}

//-------------------------------------
// function : RedirectedProcess_Read
// brief    : read 1 char from given process's output
// returns  : (success) ... read char
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_Read( RedirectedProcess * rp )
{
	DWORD	rc; // result code
	DWORD	readCount;
	char	str[3] = {0};
	wstring wstr;
	
	// try to read 1 byte from stream
	rc = ReadFile( rp->outputHdl, str, 1, &readCount, NULL );
	if( rc == FALSE || readCount == 0 )
	{
		return -1;
	}

	// if it's a first byte of double byte char in SJIS, read one more
	if( IsSjisFirstByte(*str) )
	{
		rc = ReadFile( rp->outputHdl, (str+1), 1, &readCount, NULL );
		if( rc == FALSE || readCount == 0 )
		{
			return -1;
		}
	}

	// try to convert unicode
	wstr = ToUnicode( str, (int)strlen(str) );
	if( wstr.empty() )
	{
		return -1;
	}

	return wstr[0];
}

//-------------------------------------
// function : RedirectedProcess_ReadByte
// brief    : read 1 byte from given process's output
// returns  : (success) ... read byte
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_ReadByte( RedirectedProcess * rp )
{
	DWORD	rc; // result code
	DWORD	readCount;
	int		ch = 0;
	
	// try to read 1 byte from stream
	rc = ReadFile( rp->outputHdl, &ch, 1, &readCount, NULL );
	if( rc == FALSE || readCount == 0 )
	{
		return -1;
	}

	return ch;
}

//-------------------------------------
// function : RedirectedProcess_PeekByte
// brief    : read 1 byte from given process's output without moving the file pointer
// returns  : (success) ... read byte
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_PeekByte( RedirectedProcess * rp )
{
	DWORD	rc; // result code
	DWORD	readCount;
	int		ch = 0;
	
	// try to read 1 byte from stream
	rc = PeekNamedPipe( rp->outputHdl, &ch, 1, &readCount, NULL, NULL );
	if( rc == FALSE || readCount == 0 )
	{
		return -1;
	}

	return ch;
}

//-------------------------------------
// function : RedirectedProcess_WriteByte
// brief    : write one byte into process's stdin.
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_WriteByte( RedirectedProcess * rp, int by )
{
	DWORD	rc; // result code
	DWORD	wroteLength;

	// write
	rc = WriteFile( rp->inputHdl, &by, 1, &wroteLength, NULL );
	if( rc == FALSE || wroteLength != 1 )
	{
		DebugPrint( "failed to write into stdin" );
		return -1;
	}

	return 0;
}

//-------------------------------------
// function : RedirectedProcess_Write
// brief    : write string into process's stdin.
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_Write( RedirectedProcess * rp, const wchar_t * str, int length )
{
	DWORD	rc; // result code
	DWORD	wroteLength;
	string	nstr;

	// convert to default encoding
	nstr = ToSJIS( str, length );
	if( nstr.empty() )
	{
		DebugPrint( "failed to convert Unicode to SJIS" );
		return -1;
	}

	// write
	rc = WriteFile( rp->inputHdl, nstr.c_str(), (DWORD)nstr.length(), &wroteLength, NULL );
	if( rc == FALSE || wroteLength != (DWORD)nstr.length() )
	{
		DebugPrint( "failed to write into stdin" );
		return -1;
	}

	return 0;
}

//-------------------------------------
// function : RedirectedProcess_WaitForExit
// brief    : wait till the given process has exited
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_WaitForExit( RedirectedProcess * rp )
{
	DWORD retReason;

	retReason = WaitForSingleObject( rp->processHdl, INFINITE );
	if( retReason != WAIT_OBJECT_0 )
	{
		return -1;
	}

	return 0;
}

//-------------------------------------
// function : RedirectedProcess_Wait
// brief    : wait for given time
// returns  : 0 ... process exited
//            -1 ... time-out
//            -2 ... unknown error
//-------------------------------------
extern "C" int
RedirectedProcess_Wait( RedirectedProcess * rp, int milliseconds )
{
	DWORD retReason;

	retReason = WaitForSingleObject( rp->processHdl, milliseconds );
	if( retReason == WAIT_OBJECT_0 )
	{
		return 0;
	}
	else if( retReason == WAIT_TIMEOUT )
	{
		return -1;
	}

	return -2;
}



//-------------------------------------
// function : RedirectedProcess_IsAlive
// brief    : check the process still alive
// returns  : 1 ... alive
//            0 ... not alive
//-------------------------------------
extern "C" int
RedirectedProcess_IsAlive( RedirectedProcess * rp )
{
	BOOL	rc; // result code
	DWORD	exitCode;

	rc = GetExitCodeProcess( rp->processHdl, &exitCode );
	if( rc == FALSE )
	{
		DebugPrint( "failed to get exit code / %s", GetLastErrorMessage().c_str() );
		return 0; // unknown error
	}
	else if( exitCode == STILL_ACTIVE )
	{
		return 1; // still alive
	}

	// not alive
	return 0;
}

//-------------------------------------
// function : RedirectedProcess_GetExitCode
// brief    : retrieve the exit code of the exited process
// returns  : exit code (if failed to get code, returns -1.)
// note     : should call RedirectedProcess_IsAlive before calling this,
//            because 
//-------------------------------------
extern "C" int
RedirectedProcess_GetExitCode( RedirectedProcess * rp )
{
	DWORD	rc; // result code
	DWORD	exitCode;

	// get
	rc = GetExitCodeProcess( rp->processHdl, &exitCode );
	if( rc == FALSE && exitCode == STILL_ACTIVE)
	{
		return -1; // still alive
	}

	// not alive
	return static_cast<int>( exitCode );
}

//-------------------------------------
// function : RedirectedProcess_SendCtrlBreak
// brief    : generate Ctrl+Break event (signal) in given process
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_SendCtrlBreak( RedirectedProcess * rp )
{
	BOOL	rc; // result code
	
	rc = GenerateConsoleCtrlEvent( CTRL_BREAK_EVENT, rp->processId );
	if( rc == FALSE )
	{
		return -1;
	}
	
	return 0;
}

//-------------------------------------
// function : RedirectedProcess_SendCtrlC
// brief    : generate Ctrl+C event (signal) in given process
// returns  : (success) ... 0
//            (failure) ... -1
//-------------------------------------
extern "C" int
RedirectedProcess_SendCtrlC( RedirectedProcess * rp )
{
	BOOL	rc; // result code
	
	rc = GenerateConsoleCtrlEvent( CTRL_C_EVENT, rp->processId );
	if( rc == FALSE )
	{
		return -1;
	}
	
	return 0;
}


//-------------------------------------
// function : DllMain
//-------------------------------------
BOOL APIENTRY
DllMain( HMODULE /*moduleHdl*/, ULONG /*callReason*/, void* /*reserved*/ )
{
	return TRUE;
}



//---------------------------------------------------------------------------//



static int
MyCreatePipes( HANDLE & stdOutWriteEnd, HANDLE & stdOutReadEnd, HANDLE & stdInWriteEnd, HANDLE & stdInReadEnd )
{
	BOOL				rc; // result code
	SECURITY_ATTRIBUTES	security		= {0};
	HANDLE				currentProcess;
	HANDLE				tempHdl;
	
	// prepare to create pipes
	security.nLength = sizeof( SECURITY_ATTRIBUTES );
	security.bInheritHandle = TRUE;
	currentProcess = GetCurrentProcess();
	
	// create pipes to be connected to stderr of child process
	rc = CreatePipe( &tempHdl, &stdOutWriteEnd, &security, 0 );
	if( rc == FALSE )
	{
		goto error;
	}
	
	// make only the write-end of the pipe can be inherited
	rc = DuplicateHandle( currentProcess, tempHdl,
		currentProcess, &stdOutReadEnd,
		0, FALSE, DUPLICATE_SAME_ACCESS );
	if( rc == FALSE )
	{
		goto error;
	}
	MyCloseHandle( tempHdl );
	
	// create pipes to be connected to stdin of child process
	rc = CreatePipe( &stdInReadEnd, &tempHdl, &security, 0 );
	if( rc == FALSE )
	{
		goto error;
	}
	
	// make only the read-end of the pipe can be inherited
	rc = DuplicateHandle( currentProcess, tempHdl,
		currentProcess, &stdInWriteEnd,
		0, FALSE, DUPLICATE_SAME_ACCESS );
	if( rc == FALSE )
	{
		goto error;
	}
	MyCloseHandle( tempHdl );

	return 0;
error:
	MyCloseHandle( stdInWriteEnd );
	MyCloseHandle( stdInReadEnd );
	MyCloseHandle( stdOutReadEnd );
	MyCloseHandle( stdOutWriteEnd );
	return -1;
}

//-------------------------------------
//   function	ToSJIS
//! @brief		Convert Unicode string to string in specified codepage.
//! @retval		(success) ... converted std::wstring object
//! @retval		(failure) ... empty string
//-------------------------------------
static string
ToSJIS( const wchar_t * str, int length )
{
#	ifdef _WIN32
	const DWORD	convertFlag = 0;//MB_ERR_INVALID_CHARS;
	int			rc; // result code
	char *		charBuf;
	long		neededBufSize;
	string		narrowStr;

	// get buffer size needed
	neededBufSize = WideCharToMultiByte( CP_ACP, convertFlag, str, length,
											NULL, 0, NULL, NULL );
	if( neededBufSize == 0 )
	{
		return string(); // invalid string for conversation?
	}

	// allocate buffer
	charBuf = new char[neededBufSize+1];

	// convert code page
	rc = WideCharToMultiByte( CP_ACP, convertFlag, str, length,
								charBuf, neededBufSize, NULL, NULL );
	if( rc == FALSE )
	{
		return string(); // failed to convert
	}
	charBuf[neededBufSize] = 0; // null terminate

	// make string
	narrowStr = charBuf;

	// delete buffer and return value
	delete charBuf;
	return narrowStr;
#	endif
}


//-------------------------------------
//   function	ToUnicode
//! @brief		Convert string into Unicode string
//! @returns	(success) ... converted std::wstring object
//              (failure) ... empty string
//-------------------------------------
static wstring
ToUnicode( const char * narrowStr, int strLength )
{
#	ifdef _WIN32
	const DWORD	convertFlag = MB_ERR_INVALID_CHARS;
	int			rc; // result code
	wchar_t *	wcharBuf;
	long		neededBufSize;
	wstring		widenStr;

	// get buffer size needed
	neededBufSize = MultiByteToWideChar( CP_ACP, convertFlag, narrowStr, strLength,
											NULL, 0 );
	if( neededBufSize == 0 )
	{
		return wstring(); // invalid string for conversation?
	}

	// allocate buffer
	wcharBuf = new wchar_t[neededBufSize+1];

	// convert code page
	rc = MultiByteToWideChar( CP_ACP, convertFlag, narrowStr, strLength,
								(wchar_t*)wcharBuf, neededBufSize );
	if( rc == FALSE )
	{
		return wstring(); // failed to convert
	}
	wcharBuf[neededBufSize] = 0; // null terminate

	// make wstring
	widenStr = wstring( wcharBuf );

	// delete buffer and return value
	delete wcharBuf;
	return widenStr;
#	endif // _WIN32
}


//-------------------------------------
//   function	GetLastErrorMessage
//! @brief		Get the message of Windows' "LastError"
//-------------------------------------
static std::string
GetLastErrorMessage()
{
	char	message[1024];

	// Get "LastError"
	FormatMessageA( FORMAT_MESSAGE_FROM_SYSTEM,
		NULL,
		GetLastError(),
		MAKELANGID( LANG_NEUTRAL,SUBLANG_DEFAULT ),
		message,
		256,
		NULL );

	// Write it to log file
	return message;
}

inline static void
MyCloseHandle( HANDLE & handle )
{
	CloseHandle( handle );
	handle = NULL;
}

inline static bool
IsSjisFirstByte( char ch )
{
	int c;
	
	// ignore upper 3 words of the int
	// (upper 3 words of casted 'ch' is unknown)
	c = 0xFF & ch;
	
	return ( (0x81 <= c && c <= 0x9F) || (0xE0 <= c && c <= 0xFC) );
}


//---------------------------------------------------------------------------//


inline static void
DebugPrint( const char * inFormat, ... )
{
#	ifndef NDEBUG
	va_list	argumentVector;
	char	message[1024];
	
	// Display message
	va_start( argumentVector, inFormat );
	vsprintf( message, inFormat, argumentVector );
	std::cerr << "[RedirectedProcess.dll]" << message << std::endl;
	va_end( argumentVector );
#	endif
}
inline static void
DebugPrint( const wchar_t * inFormat, ... )
{
#	ifndef NDEBUG
	va_list	argumentVector;
	wchar_t	message[1024];
	
	// Display message
	va_start( argumentVector, inFormat );
	vswprintf( message, inFormat, argumentVector );
	std::cerr << L"[RedirectedProcess.dll]" << message << std::endl;
	va_end( argumentVector );
#	endif
}

/**********************************************************
Copyright (c) 2006 YAMAMOTO Suguru

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
copies of the Software, and to permit persons to whom the Software is 
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
THE SOFTWARE.
**********************************************************/
